
#ifdef WIN32
#pragma comment(lib, "ws2_32.lib")
#else
#define inaddrr(x) (*(struct in_addr *) &ifr->x[sizeof sa.sin_port])
#define IFRSIZE   ((int)(size * sizeof (struct ifreq))) 
#endif

#ifndef WIN32
#include <unistd.h>
#include <netinet/in.h>   
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#ifdef HAVE_NET_IF_ARP_H
#include <net/if_arp.h>
#endif

#include <net/if.h>
#else
#include <winsock.h>
#include <windows.h>
#endif

#include <stdarg.h>
#include <getopt.h>

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>    
 
#include <fcntl.h>
#include <string.h>

#include "upnpscan.h"
#include "ipparser/ipparser.h"

#ifndef WIN32
#include "config.h"
#else
#define VERSION "0.4"
#endif

#define UPN_PORT 1900
#define MCAST_GROUP_ADDR "239.255.255.250"

#define AUTHOR "patrik@cqure.net"

static int bVerbose = 0;
static int nTimeout = 100000;
static int bIsMcast = 0;
static FILE *pLogger = NULL;

#ifdef WIN32
#define snprintf _snprintf
#define vsnprintf _vsnprintf
#endif

int logprintf( const char *format, ... ) {
	
	va_list args;
	int i;
	
	va_start( args, format );
	
	if ( pLogger ) {
		i = vfprintf( pLogger, format, args );
		vprintf( format, args );
	}
	else {
		i = vprintf( format, args );
	}
	
	va_end( args );
	
	return i;
}

#ifndef WIN32
int sendDiscoveryPacket( struct in_addr *pSrc, char *pDest ) {
#else
SOCKET sendDiscoveryPacket( struct in_addr *pSrc, char *pDest ) {
#endif

  struct sockaddr_in addr_dest, addr_src;
  struct in_addr src;

  const unsigned char TTL = 1;
  const int BCAST = 1;
  int l;
  
#ifdef WIN32
  SOCKET nSocket;
#else
  int nSocket;
#endif

  if ( NULL != pSrc )
	  src = *pSrc;
  
  nSocket = socket( AF_INET, SOCK_DGRAM, 0 );

  addr_src.sin_family = AF_INET;
  addr_src.sin_addr = src;
  addr_src.sin_port = 0;

  addr_dest.sin_family = AF_INET;
  addr_dest.sin_addr.s_addr = inet_addr( pDest );
  addr_dest.sin_port = htons( UPN_PORT );

  if ( NULL != pSrc ) {
	  
  	if ( 0 != bind( nSocket, (const struct sockaddr *)&addr_src, sizeof( addr_src ) ) ) {
		perror( "bind" );
		return -1;
  	}
	
  }

  if ( 0 > setsockopt( nSocket, IPPROTO_IP, IP_MULTICAST_TTL, (char *)&TTL, 
		       sizeof( TTL ) ) ) {
    perror( "setsockopt multicast" );
    return -1;
  }

  if ( 0 >  setsockopt( nSocket, SOL_SOCKET, SO_BROADCAST, (char *)&BCAST, 
			sizeof(BCAST) ) ) {
    perror( "setsockopt broadcast" );
    return -1;
  }
 
  l = sizeof( addr_dest );

  if ( -1 == sendto( nSocket, UPNP_SEARCH, (int)strlen( UPNP_SEARCH ), 0,
		     (struct sockaddr *)&addr_dest, l ) ) {
    perror( "sendto" );
    return -1;
  }


  return nSocket;

}

#ifndef WIN32
void recvResponsePacket( int nSocket, int bIsMulticast ) {
#else
void recvResponsePacket( SOCKET nSocket, int bIsMulticast ) {
#endif

  struct sockaddr_in addr_remote;
  struct timeval tv;
  fd_set rfds;

  char buf[1024];
  int nBytes, ret;
  unsigned int l;

  while ( 1 ) {

    FD_ZERO(&rfds);
    FD_SET(nSocket, &rfds);

    tv.tv_sec = 0;
    tv.tv_usec = nTimeout;

    ret = select((int)nSocket + 1, &rfds, NULL, NULL, &tv);

    if ( -1 == ret || !FD_ISSET( nSocket, &rfds ) )
      break;

    l = sizeof( addr_remote );
    memset( buf, 0, sizeof( buf ) );
    nBytes = recvfrom( nSocket, buf, sizeof( buf ), 0, 
						(struct sockaddr*)&addr_remote, &l );

	if ( 0 < nBytes ) {
		logprintf( "\n" );
	    logprintf( "[%s]\n", inet_ntoa( addr_remote.sin_addr ) );
		logprintf( "\n" );
		logprintf( "%s\n", buf );
		logprintf( "---\n" );
	}

    if ( !bIsMulticast )
      break;

  }

#ifdef WIN32
  closesocket( nSocket );
#else
  close( nSocket );
#endif

}

/*
 * Prints and underlines a row ( max 80 chars )
 *
 */
int printf_underline( const char *format, ... ) {
	
	va_list args;
	int i, j;
	char row[80];

	memset( row, 0, sizeof( row ) );

	va_start( args, format );
	
	i = vsnprintf( row, sizeof( row ) - 1, format, args );
	logprintf( "%s", row );

	for( j = 0; j<(int)strlen( row ) -1; j++ )
	  logprintf("-");

	logprintf( "\n" );

	va_end( args );
	
	return i;
}


void banner() {
  printf_underline("UPnP Discovery Tool v%s by %s\n", VERSION, AUTHOR );
}

void usage(char *pProg) {

  printf("\n");
  printf("Usage: %s -t <target> | -m [options]\n", pProg);
  printf("\n");
  printf("Options\n");
  printf("  -D\t\t - Dump interface list\n");
  printf("  -i <iface>\t - Src. interface\n" );
  printf("  -t <target>\t - Scan targets\n" );
  printf("  -m\t\t - Use UPnP multicast address\n");
  printf("  -o <file>\t - Report to file\n");
  printf("  -v\t\t - Be verbose\n");
  printf("\n" );

}

void print_interfaces( const struct in_addr *pAddr, int nCount ) {

	int i = 0;

	for ( i=0; i<nCount; i++ ) {
		printf("(%d) %s\n", i, inet_ntoa( pAddr[i] ) );
	}

	printf("(A) All interfaces\n");
	
}

#ifdef WIN32
/* Tubby solution ... Needs to be rewritten at some point */ 
int enum_interfaces( struct in_addr *pAddr, int nSize ) {
	
	char name[255];
	int i;

	struct hostent *pHEnt;

	memset( name, 0, sizeof( name ) );
	
	if ( 0 != gethostname( name, sizeof( name ) ) )
		return -1;

	pHEnt = gethostbyname( name );
	
	for ( i=0; 0 != pHEnt->h_addr_list[i]; i++  ) {
		pAddr[i] = *(struct in_addr *)pHEnt->h_addr_list[i];
	}

	return (i);

}
#else
int enum_interfaces( struct in_addr *pAddr, int nSize ) {

	int	sockfd, size  = 1, nCount = 0;
  	struct ifreq       *ifr;
  	struct ifconf      ifc;
  	struct sockaddr_in sa;

  	if (0 > (sockfd = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP))) {
    	fprintf(stderr, "Cannot open socket.\n");
    	return (-1);
  	}

  	ifc.ifc_len = IFRSIZE;
  	ifc.ifc_req = NULL;

  	do {
    	++size;
    	
		/* realloc buffer size until no overflow occurs  */
    	if (NULL == (ifc.ifc_req = realloc(ifc.ifc_req, IFRSIZE))) {
      		fprintf(stderr, "Out of memory.\n");
      		return (-1);
    	}
    
		ifc.ifc_len = IFRSIZE;
    
		if (ioctl(sockfd, SIOCGIFCONF, &ifc)) {
      		perror("ioctl SIOCFIFCONF");
      		return (-1);
    	}
  
	} while  (IFRSIZE <= ifc.ifc_len);

  	ifr = ifc.ifc_req;
  	
	for (;(char *) ifr < (char *) ifc.ifc_req + ifc.ifc_len; ++ifr) {

		if (ifr->ifr_addr.sa_data == (ifr+1)->ifr_addr.sa_data) {
		  continue;  /* duplicate, skip it */
		}

		if (ioctl(sockfd, SIOCGIFFLAGS, ifr)) {
		  continue;  /* failed to get flags, skip it */
		}

		if (! ifr->ifr_flags & IFF_UP) {
		  continue;  /* interface not up, skip it */
		}

		/* Do not add loopback interface */
		/* WORKAROUND cygwin: Do not add interfaces having IP 0.0.0.0 */
		if (( htonl(INADDR_LOOPBACK) != ((struct in_addr )(inaddrr( ifr_addr.sa_data ))).s_addr ) &&
			( 0 != ((struct in_addr )(inaddrr( ifr_addr.sa_data ))).s_addr ) ) {
			pAddr[ nCount ] = inaddrr(ifr_addr.sa_data);
			nCount ++;
		}
		
	}

  	close(sockfd);
  
	return nCount; 

}
#endif

int main( int argc, char **argv ) {

  int c;

#ifndef WIN32
  int nSocket;
#else
  SOCKET nSocket;
  WSADATA wsa;
#endif

  char *pIP;
  char filter[255];

  struct in_addr iflist[255];
  int nIfCount = 0, nIfSel = -1, nIfAll = 0, i = 0;

#ifdef WIN32
  WSAStartup( MAKEWORD( 2, 2 ), &wsa );
#endif

  memset( filter, 0, sizeof( filter ) );
  memset( iflist, 0, sizeof( iflist ) );

  nIfCount = enum_interfaces( iflist, sizeof( iflist ) );
  banner();

  while ((c = getopt(argc, argv, "vt:T:hmo:Di:")) != -1) {
    	
    switch(c) {
    
		case 'D':
			printf("\nInterfaces;\n");
			print_interfaces( iflist, nIfCount );
			goto cleanup;

		case 'i':
			if ( nIfCount <= atoi( optarg ) ) {
				printf("Unknown interface: Use -D to check for valid interfaces\n");
				goto cleanup;
			}
			
			if ( 'A' == optarg[0] )
				nIfAll = 1;
			
			nIfSel = atoi( optarg );
			break;

        case 'v':
	  bVerbose = 1;
	  break;
   
        case 't':
	  strncpy( filter, optarg, sizeof( filter ) );
	  break;

        case 'T':
	  nTimeout = atol( optarg ) * 1000;
	  break;

        case 'h':
	  usage( argv[0] );
	  exit( 1 );

        case 'm':
	  strcpy( filter, MCAST_GROUP_ADDR );
	  bIsMcast = 1;
	  break;

        case 'o':
	  pLogger = fopen( optarg, "w" );
	  break;

        default:
			usage( argv[0] );
			goto cleanup;

    }

  }

  if ( 0 == strlen( filter ) ) {
    usage( argv[0] );
    exit ( 1 );
  }

  set_addr_filter( filter );

  if ( !nIfAll && bVerbose )
	  printf("[i] Source interface: %s\n", inet_ntoa( iflist[ nIfSel ] ) );

  while ( NULL != ( pIP = get_next_addr() ) ) {

    if ( bVerbose )
      printf( "[i] Scanning %s\n", pIP );

	if ( nIfAll ) {
		/* Send discovery to all interfaces */
		for ( i = 0; i<nIfCount; i++ ) {

			if ( bVerbose )
				printf("[i] Source interface: %s\n", inet_ntoa( iflist[ i ] ) );

			nSocket = sendDiscoveryPacket( &iflist[ i ], pIP );
			
			if ( -1 != nSocket )
      			recvResponsePacket( nSocket, bIsMcast );
		}
	}
	else if ( -1 != nIfSel ) {
		nSocket = sendDiscoveryPacket( &iflist[ nIfSel ], pIP );

    	if ( -1 != nSocket )
      		recvResponsePacket( nSocket, bIsMcast );	
	}
	else {
		nSocket = sendDiscoveryPacket( NULL, pIP );
	
    	if ( -1 != nSocket )
      		recvResponsePacket( nSocket, bIsMcast );
	}
  
  }

cleanup:
#ifdef WIN32
  WSACleanup();
#endif

  return (0);

}
