import yoda
import hepdata_lib as hdlib

def _isZero(val, tolerance=1e-8):
    return abs(val) < tolerance

def _fuzzyEquals(a,b, tolerance=1e-5):
    absavg = 0.5*(abs(a) + abs(b))
    absdiff = abs(a - b)
    return (_isZero(a) and _isZero(b)) or absdiff < tolerance*absavg


def _Est2Tab(AO, includeOverflows, includeMaskedBins):
    table = hdlib.Table(AO.name())
    table.description = AO.title()

    for ia, axisT in enumerate(AO.axisConfig()):
        indep = hdlib.Variable(f"axis {ia}", is_independent=True, is_binned=bool(axisT == 'd'))
        # Set edges for this axis
        edges = None
        if ia == 0:
            edges = [ (b.xMin(), b.xMax()) if axisT == 'd' else \
                      b.xEdge() for b in AO.bins(includeOverflows) ]
        elif ia == 1:
            edges = [ (b.yMin(), b.yMax()) if axisT == 'd' else \
                      b.yEdge() for b in AO.bins(includeOverflows) ]
        elif ia == 2:
            edges = [ (b.zMin(), b.zMax()) if axisT == 'd' else \
                      b.zEdge() for b in AO.bins(includeOverflows) ]
        else:
            raise Exception(f"Binning dimension {ia+1} not supported by Python extension!")
        indep.values = edges
        # Add independent variable to table
        table.add_variable(indep)

    dep =  hdlib.Variable(AO.name(), is_independent=False, is_binned=False, zero_uncertainties_warning=False)
    dep.values = AO.vals(includeOverflows, includeMaskedBins) if AO.dim() > 1 else [ AO.val() ]
    errs = dict([ (src, hdlib.Uncertainty(src, is_symmetric = bool('stat' in src))) for src in AO.sources() ])
    for b in ([AO] if 'Estimate0D' == AO.type() else AO.bins(includeOverflows, includeMaskedBins)):
        for src, err in errs.items():
            if b.hasSource(src):
                if 'stat' in src:
                    err.values.append( abs(b.errUp(src)) )
                elif _fuzzyEquals(*b.err(src)):
                    err.values.append( (-b.errDown(src), b.errUp(src)) )
                else:
                    err.values.append( (b.errDown(src), b.errUp(src)) )
            else:
                err.values.append(None)
    for src, err in errs.items():
        dep.add_uncertainty(err)

    # Set qualifiers
    dep.add_qualifier("Custom Rivet identifier", AO.name())
    for key, val in AO.annotationsDict().items():
        if any(x in key for x in ["Path", "Type", "Title", "IsRef"]):  continue
        dep.add_qualifier(key, val)
    # Add dependent variable to table
    table.add_variable(dep)

    return table


def _Scat2Tab(AO, includeOverflows, includeMaskedBins):
    table = hdlib.Table(AO.name())
    table.description = AO.title()

    for ia in range(AO.dim()):
        isDep = bool(ia+1 == AO.dim())
        hd_var = hdlib.Variable(AO.name() if isDep else f"axis {ia}",
                                is_independent=not isDep, is_binned=False,
                                zero_uncertainties_warning=False)
        # Set central values for this dimension
        hd_var.values = AO.vals(ia)
        # Set errors for this dimension
        errs = hdlib.Uncertainty("", is_symmetric=False)
        errs.values = [ list(-1.0*neg,pos) if _fuzzyEquals(neg,pos) else list(neg,pos) for neg,pos in AO.errs(ia) ]
        hd_var.add_uncertainty(err)
        # Set qualifiers
        if isDep:
            hd_var.add_qualifier("Custom Rivet identifier", AO.name())
            for key, val in AO.annotationsDict().items():
                if any(x in key for x in ["Path", "Type", "Title", "IsRef"]):  continue
                hd_var.add_qualifier(key, val)
        # Add variable to table
        table.add_variable(hd_var)

    return table


def to_table(yoda_obj, includeOverflows=False, includeMaskedBins=False):

    AO = yoda_obj.mkInert(yoda_obj.path())
    if "Estimate" in AO.type():
        return _Est2Tab(AO, includeOverflows, includeMaskedBins)
    if "Scatter" in AO.type():
        return _Scat2Tab(AO, includeOverflows, includeMaskedBins)
    else:
        raise Exception(f"Unknown/unsupported type for conversion: {yoda_obj.type()}")
