// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/ChargedFinalState.hh"
#include "Rivet/Projections/IdentifiedFinalState.hh"

namespace Rivet {


  /// @brief gamma gamma -> mu+ mu-
  class ATLAS_2021_I1832628 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ATLAS_2021_I1832628);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projections
      ChargedFinalState cfs(Cuts::abseta < 2.4 and Cuts::pT>100*MeV);
      declare(cfs,"CFS");
      // Get muons which pass the initial kinematic cuts
      IdentifiedFinalState muon_fs(Cuts::abseta < 2.4 && Cuts::pT > 4*GeV);
      muon_fs.acceptIdPair(PID::MUON);
      declare(muon_fs, "MUON_FS");
      // histograms
      vector<double> mbins = {10,20,40,80}, ybins = {0.,0.8,1.6,2.4};
      book(_d_y_m,mbins);
      for(auto & b : _d_y_m->bins()) book(b,b.index(),1,1);
      book(_d_m_y,ybins);
      for(auto & b : _d_m_y->bins()) book(b,b.index()+6,1,1);
      book(_d_cTheta_dm[0],mbins);
      for(auto & b : _d_cTheta_dm[0]->bins()) book(b,b.index()+12,1,1);
      book(_d_cTheta_dm[1],mbins);
      for(auto & b : _d_cTheta_dm[1]->bins()) book(b,b.index()+18,1,1);
      book(_h_aco,29,1,1);
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const ChargedFinalState& cfs = apply<ChargedFinalState>(event, "CFS");
      if (cfs.size() != 2) vetoEvent; // no other charged particles in 2.5
      // extract the muons
      const Particles& muonFS = apply<IdentifiedFinalState>(event, "MUON_FS").particles();
      // have 2 muons with opposite charge
      if (muonFS.size() != 2) vetoEvent;
      if (muonFS[0].pid() != -muonFS[1].pid()) vetoEvent;
      // mu+mu- pair invariant mass
      FourMomentum pmumu = muonFS[0].momentum()+muonFS[1].momentum();
      const double mmumu = pmumu.mass();
      if (mmumu<10.*GeV) vetoEvent;
      // mu+mu- pT
      const double pTmumu=pmumu.perp();
      if(pTmumu>2.) vetoEvent;
      double ymumu = abs(pmumu.rapidity());
      _d_y_m->fill(mmumu,ymumu);
      _d_m_y->fill(ymumu,mmumu);
      double yp = muonFS[0].rapidity();
      double ym = muonFS[1].rapidity();
      double cTheta = abs(tanh(0.5*(yp-ym)));
      _d_cTheta_dm[0]->fill(mmumu,cTheta);
      if(ymumu<0.8) _d_cTheta_dm[1]->fill(mmumu,cTheta);
      double aco = 1. - deltaPhi(muonFS[0],muonFS[1])/M_PI;
      _h_aco->fill(aco);
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_d_y_m      , crossSection()/microbarn/sumOfWeights());
      scale(_d_m_y      , crossSection()/microbarn/sumOfWeights());
      scale(_d_cTheta_dm, crossSection()/microbarn/sumOfWeights());
      scale(_h_aco      , crossSection()/microbarn/sumOfWeights());
    }

    /// @}


    /// @name Histograms
    /// @{
    Histo1DGroupPtr _d_y_m,_d_m_y,_d_cTheta_dm[2];
    Histo1DPtr _h_aco;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(ATLAS_2021_I1832628);

}
