// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Cross section for $e^+e^-\to J/\psi\pi^+\pi^-$ at energies between 3.77 and 4.6 GeV
  class BESIII_2016_I1495838 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2016_I1495838);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      for (size_t ix=0; ix<2; ++ix) {
        book(_sigmaPsi[ix], 1+ix, 1, 1);
        for (const string& en : _sigmaPsi[ix].binning().edges<0>()) {
          const double eval = stod(en);
          if (isCompatibleWithSqrtS(eval)) {
            _sqs[ix] = en; break;
          }
        }
      }
      raiseBeamErrorIf(_sqs[0].empty() && _sqs[1].empty());
    }

    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle &child : p.children()) {
        if (child.children().empty()) {
          --nRes[child.pid()];
          --ncount;
        }
        else {
          findChildren(child,nRes,ncount);
        }
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");
      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p: fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      const FinalState& ufs = apply<FinalState>(event, "UFS");
      // find the psis
      for (const Particle& p : ufs.particles(Cuts::pid==443)) {
        if (p.children().empty()) continue;
        map<long,int> nRes = nCount;
        int ncount = ntotal;
        findChildren(p,nRes,ncount);
        // psi pi+pi-
        if (ncount!=2) continue;
        bool matched = true;
        for (const auto& val : nRes) {
          if (abs(val.first)==211) {
            if (val.second !=1) {
              matched = false;
              break;
            }
          }
          else if (val.second!=0) {
            matched = false;
            break;
          }
        }
        if (matched) {
          _sigmaPsi[0]->fill(_sqs[0]);
          _sigmaPsi[1]->fill(_sqs[1]);
          break;
        }
      }
    }

    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_sigmaPsi, crossSection()/ sumOfWeights() /picobarn);
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _sigmaPsi[2];
    string _sqs[2];
    /// @}


  };



  RIVET_DECLARE_PLUGIN(BESIII_2016_I1495838);

}
