// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/FastJets.hh"
#include "Rivet/Projections/DISFinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Dijet angular distributions in photoproduction of charm
  class ZEUS_2003_I613625 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ZEUS_2003_I613625);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Projections
      const DISFinalState dfs(DISFrame::LAB);
      declare(dfs, "dfs");

      declare(FastJets(dfs, JetAlg::KT, 1.0), "Jets");

      declare(UnstableParticles(Cuts::abspid == 413), "DStars");

      // Table 1 T1, F2
      book(_h["X_Obs_Gamma"], 1, 1, 1);

      // Table 2 T2, F3
      book(_h["X_Obs_Proton"], 2, 1, 1);

      // Table 3 T3, F4
      book(_h["abs_cos_theta_res"], 3, 1, 1);
      book(_h["abs_cos_theta_dir"], 3, 1, 2);

      // Table 4
      book(_h["cos_theta_res"], 4, 1, 1);
      book(_h["cos_theta_dir"], 4, 1, 2);

    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {

       // DIS Kinematics & Warnings
      const DISKinematics& kin = apply<DISFinalState>(event, "dfs").kinematics();
      if ( kin.failed() )  vetoEvent;

      const double Q2 = kin.Q2();
      const double W  = sqrt(kin.W2());
      const double y = kin.y();

      // Kinematic cuts on virtuality and inelasticity.
      if ( Q2 >= 1*GeV2 )  vetoEvent;
      if ( !inRange(W, 130*GeV, 280*GeV) )  vetoEvent;
      if ( !inRange(y, 0.19, 0.87) ) vetoEvent; // from the HZTOOL

      const int orientation = kin.orientation();

      // Jets
      const Jets jets = apply<FastJets>(event, "Jets").jetsByPt(Cuts::abseta < 2.4 && Cuts::pT > 5*GeV);
      if (jets.size() < 2)  vetoEvent;

      // Vector of DStars
      const Particles dstars = apply<ParticleFinder>(event, "DStars").particles(Cuts::pT > 3*GeV && Cuts::abseta < 1.5);

      // Veto if no DStars
      if (dstars.empty())  vetoEvent;

      // Define leading jets
      const double dr1 = deltaR(jets[0], dstars[0]);
      const double dr2 = deltaR(jets[1], dstars[0]);
      if (dr1 >= 1.0 && dr2 >= 1.0)  vetoEvent;
      const Jet& j1 = jets[dr1<dr2? 0 : 1];
      const Jet& j2 = jets[dr1<dr2? 1 : 0];

      // Define jet eta (with orientations)
      const double eta1 = orientation*j1.eta(), eta2 = orientation*j2.eta();
      const double phi1 = orientation*j1.phi(), phi2 = orientation*j2.phi();

      // Same definitions issue as the 2000 ZEUS paper, where the definition
      // is actually a sum over all jet energies and pseudorapities, rather than 2
      const double x_gamma = (j1.Et() * exp(-eta1) + j2.Et() * exp(-eta2)) / (2*kin.y()*kin.beamLepton().E());

      const double x_proton = (j1.Et() * exp(eta1) + j2.Et() * exp(eta2)) / (2*kin.beamHadron().E());

      // Define cos_theta_star
      const double cos_theta = tanh(0.5*(eta1 - eta2));

      // Define Mjj (using Et mean here as I am not certain what Et jet is supposed to be) check!!
      //const double mjj = 2.0 * j1.Et() * j2.Et() / sqrt( 1.0 - sqr(cos_theta) );
      const double mjj = sqrt(2.0 * j1.Et() * j2.Et() * (cosh(eta1-eta2) - cos(phi1-phi2)) );

      //cut with Mjj
      if (mjj < 18*GeV) vetoEvent;

      // Define eta_bar
      const double eta_bar = 0.5*(eta1 + eta2);

      // eta_bar cut
      if (abs(eta_bar) > 0.7) vetoEvent;

      _h["X_Obs_Gamma"]->fill(x_gamma);
      _h["X_Obs_Proton"]->fill(x_proton);
      if (x_gamma < 0.75) {
        _h["abs_cos_theta_res"]->fill(abs(cos_theta));
        _h["cos_theta_res"]->fill(cos_theta);
      }
      else {
        _h["abs_cos_theta_dir"]->fill(abs(cos_theta));
        _h["cos_theta_dir"]->fill(cos_theta);
      }
    }



    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_h, crossSection()/nanobarn/sumW());
    }

    /// @}


    /// @name Histograms
    map<string, Histo1DPtr> _h;
    /// @{

  };


  RIVET_DECLARE_PLUGIN(ZEUS_2003_I613625);

}
