// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief Add a short analysis description here
  class CELLO_1982_I12010 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(CELLO_1982_I12010);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");

      // Book histograms
      size_t ih = 1;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en+"EEC"],   1, 1, ih);
        book(_h[en+"AEEC"],  3, 1, ih);
        book(_c[en], "TMP/weightSum"+en);
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {

      if (edges[0].empty())  edges[0] = _h[_sqs+"EEC"]->xEdges();
      if (edges[1].empty())  edges[1] = _h[_sqs+"AEEC"]->xEdges();

      // First, veto on leptonic events by requiring at least 4 charged FS particles
      const FinalState& fs = apply<FinalState>(event, "FS");
      // Even if we only generate hadronic events, we still need a cut on numCharged >= 2.
      if (fs.particles().size() < 2) vetoEvent;
      _c[_sqs]->fill();

      const double Evis2 = sqr(sum(fs.particles(), Kin::E, 0.0));
      // (A)EEC
      // Need iterators since second loop starts at current outer loop iterator, i.e. no "foreach" here!
      for (Particles::const_iterator p_i = fs.particles().begin(); p_i != fs.particles().end(); ++p_i) {
        for (Particles::const_iterator p_j = p_i; p_j != fs.particles().end(); ++p_j) {
          const Vector3 mom3_i = p_i->p3();
          const Vector3 mom3_j = p_j->p3();
          const double energy_i = p_i->E();
          const double energy_j = p_j->E();
          const double thetaij = mom3_i.unit().angle(mom3_j.unit());
          double eec = (energy_i*energy_j) / Evis2;
          if (p_i != p_j)  eec *= 2.;
          _h[_sqs+"EEC"]->fill(map2string(thetaij, 0), eec);
          if (thetaij < 0.5*M_PI) {
            _h[_sqs+"AEEC"]->fill(map2string(thetaij, 1), -eec);
          }
          else {
            _h[_sqs+"AEEC"]->fill(map2string(M_PI-thetaij, 1), eec);
          }
        }
      }
    }

    string map2string(const double val, const size_t axis) const {
      const size_t idx = axes[axis].index(val);
      if (idx && idx <= edges[axis].size())  return edges[axis][idx-1];
      return "OTHER";
    }

    /// Normalise histograms etc., after the run
    void finalize() {
      for (const auto& item : _c) {
        size_t ih = 0;
        for (const string& obs : vector<string>{"EEC"s, "AEEC"s}) {
          scale(_h[item.first+obs], 1.0/item.second->sumW());
          for (auto& b: _h[item.first+obs]->bins()) {
            b.scaleW(1.0/axes[ih].width(b.index()));
          }
          ++ih;
        }
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,BinnedHistoPtr<string>> _h;
    map<string,CounterPtr> _c;
    vector<string> edges[2];
    string _sqs = "";
    YODA::Axis<double> axes[2] = { YODA::Axis<double>(50, 0.0, M_PI),
                                   YODA::Axis<double>(24, 0.0628, 0.5*M_PI) };
    /// @}


  };

  RIVET_DECLARE_PLUGIN(CELLO_1982_I12010);
}
