
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "gamepick.h"
#include "read_line.h"
#include "add_games.h"
#include "click_game.h"
#include "save_lists.h"
#include "edits.h"
#include "icons.h"
#include "auto_games.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


gboolean known_which (char *cmd_buffer, char *results, const gchar *cmd)
{
  FILE *file;
  gint length;
  
  snprintf (cmd_buffer, buffer_size, "which %s 2>/dev/null", cmd);
  results [0] = '\0';
  
  if (!(file = popen (cmd_buffer, "r")))
    return FALSE;
  
  fgets (results, buffer_size, file);
  pclose (file);
  if (results [0] == '\0')
    return FALSE;
  length = strlen (results);
  if ((length > 0) && (results [length -1] == '\n'))
    results [length -1] = '\0';
  return TRUE;
}


void calculate_icon_path (char *cmd_buffer, char *results, const gchar *icon)
{
  gint length;
    
  length = readlink (results, cmd_buffer, buffer_size - 1);
  if (length < 0)
  {
    results [0] = '\0';
    return;
  }
  cmd_buffer [length] = '\0';
  snprintf (results, buffer_size, "%s/%s", find_parent_dir (cmd_buffer), icon);
  if (!file_readable (results))
    results [0] = '\0';
}


void empty_auto_games (gamepick_info *info)
{
  gint total, index;
  
  if (!info->auto_games_list)
    return;
  
  total = g_list_length (info->auto_games_list);

  for (index = 0; index < total; index++)
    g_free (g_list_nth (info->auto_games_list, index)->data);

  g_list_free (info->auto_games_list);
  info->auto_games_list = NULL;
}


void auto_add_tab (gamepick_info *info, const gchar *title)
{
  GtkWidget *label, *table;
  
  label = gtk_label_new (title);
  gtk_widget_show (label);
  
  table = gtk_table_new (auto_rows * 4, 3, FALSE);
  gtk_widget_show (table);
  gtk_container_set_border_width (GTK_CONTAINER (table), info->icon_spacing);
  gtk_table_set_row_spacings (GTK_TABLE (table), info->icon_spacing);
  gtk_table_set_col_spacings (GTK_TABLE (table), info->icon_spacing);
  gtk_notebook_append_page (GTK_NOTEBOOK (info->auto_tabs), table, label);
  
  info->auto_table = table;
  info->auto_games_count = 0;
}


void auto_add_game (gamepick_info *info, const game_struct *game)
{
  GtkWidget *label, *action, *icon;
  GString *tab_title;
  game_struct *target;
  tab_struct *tab;
  gint x, y, total, index;

  if (info->auto_games_count >= (auto_cols * auto_rows))
  {
    tab_title = g_string_new ("");
    g_string_sprintf (tab_title, "Page %d", ++info->auto_games_page);
    auto_add_tab (info, tab_title->str);
  }

  x = info->auto_games_count % auto_cols;
  y = info->auto_games_count / auto_cols;
  info->auto_games_count++;

  x = x * 2;
  y = y * 4;
  
  icon = scaled_icon (game->icon->str, 48, 48);
  if (icon)
  {
    gtk_table_attach (GTK_TABLE (info->auto_table), icon,
      x, x+1, y, y+1, GTK_SHRINK, GTK_SHRINK, 0, 0);
    gtk_widget_show (icon);
  }

  label = gtk_label_new (fragment (info, game->name->str));
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_CENTER);
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (info->auto_table), label,
    x, x+1, y+1, y+2, GTK_SHRINK, GTK_SHRINK, 0, 0);

  action = gtk_combo_box_new_text();
  gtk_widget_show (action);
  gtk_table_attach (GTK_TABLE (info->auto_table), action,
    x, x+1, y+2, y+3, GTK_SHRINK, GTK_SHRINK, 0, 0);
  append (action, "");
  total = g_list_length (info->tab_list);
  for (index = 0; index < total; index++)
  {
    tab = (tab_struct*) g_list_nth (info->tab_list, index)->data;
    append (action, tab->tab_name->str);
  }
  gtk_combo_box_set_active (GTK_COMBO_BOX (action), 0);

  target = g_malloc0 (sizeof (game_struct));
  target->info = info;
  target->name = g_string_new (game->name->str);
  target->start_dir = g_string_new (game->start_dir->str);
  target->cmd = g_string_new (game->cmd->str);
  target->icon = g_string_new (game->icon->str);
  target->fsaa = 0;
  target->aitf = 0;
  target->button = action;
  info->auto_games_list = g_list_append (info->auto_games_list, target);
}


void maybe_known_entry (gamepick_info *info, game_struct *game)
{
  if (text_here (game->name) && game->cmd_found)
    auto_add_game (info, game);
  zero_game_entry (game);
}


gboolean knowns_keyword_valid (const gchar *text)
{
  static const char* keyword_array[] = 
  {
    "name", "cmd", "icon dir", "icon", "icon sym", "icon file", ""
  };
  int count = 0;
  
  while (TRUE)
  {
    if (strcmp (keyword_array [count], "") == 0) return FALSE;
    if (strcmp (text, keyword_array [count]) == 0) return TRUE;
    count++;
  }
}


gboolean knowns_keyword_test (const gchar *left, const gchar* keyword)
{
  if (!knowns_keyword_valid (keyword))
  {
    g_print ("internal error: unknown keyword %s\n", keyword);
    return FALSE;
  }
  if (strcmp (left, keyword) == 0)
    return TRUE;
  else
    return FALSE;
}


void load_knowns (gamepick_info *info, const gchar *filename)
{
  readline *context;
  gchar *left, *right, *dir_text;
  GString *line, *error_text, *name;
  gint count = 0, index, dir_count;
  gboolean errors_present = FALSE;
  game_struct game;
  char cmd_buffer [buffer_size];
  char results [buffer_size];

  if (!filename)
  {
    g_print ("Can't load known list\n");
    return;
  }

  info->auto_games_list = NULL;

  line = g_string_new (NULL);
  error_text = g_string_new (NULL);
  name = g_string_new (NULL);
  context = read_start (filename, 256);
  zero_game_entry (&game);

  for (;;)
  {
    if (text_here (error_text))
    {
      if (errors_present == FALSE)
      {
        g_print ("gamepick: errors in %s\n", filename);
        errors_present = TRUE;
      }
      g_print ("line %d: %s\n", count, error_text->str);
      g_string_assign (error_text, "");
    }
    count++;
    if (!read_next (context, &line))
    {
      maybe_known_entry (info, &game);
      return;
    }
    if (!text_here (line))
      continue;
    if ((line->str)[0] == '#')
      continue;
    index = char_find (line->str, ':');
    if (index < 0)
      continue;
    if ((line->len < 3) || (index < 1))
      continue;
    (line->str) [index] = '\0';
    left = line->str;
    right = line->str + index + 1; /* yeah a bit iffy */
    while (*right == ' ')
      right++; /* same */
    
    if (knowns_keyword_test (left, "name"))
    {
      maybe_known_entry (info, &game);
      g_string_assign (game.name, right);
    }
    else
    if (knowns_keyword_test (left, "cmd"))
    {
      g_string_assign (game.cmd, right);
      if (known_which (&cmd_buffer[0], &results[0], game.cmd->str))
        game.cmd_found = TRUE;
    }
    else
    if (knowns_keyword_test (left, "icon dir"))
      info->icon_dir_list = g_list_append (info->icon_dir_list,
        (g_string_new (right))->str);
    else
    if (knowns_keyword_test (left, "icon sym"))
    {
      if (!text_here (game.icon) && game.cmd_found)
      {
        calculate_icon_path (&cmd_buffer[0], &results[0], right);
        if (file_readable (results))
          g_string_assign (game.icon, results);
      }
    }
    else
    if (knowns_keyword_test (left, "icon"))
    {
      if (!text_here (game.icon) && game.cmd_found)
      {
        for (dir_count = 0; dir_count <
          g_list_length (info->icon_dir_list); dir_count++)
        {
          dir_text = (gchar*)
            g_list_nth (info->icon_dir_list, dir_count)->data;
          g_string_sprintf (name, "%s/%s", dir_text, right);
          if (file_readable (name->str))
          {
            g_string_assign (game.icon, name->str);
            break;
          }
        }
      }
    }
    else
    if (knowns_keyword_test (left, "icon file"))
    {
      if (!text_here (game.icon) && game.cmd_found)
      {
        if (file_readable (right))
          g_string_assign (game.icon, right);
      }
    }
    else
      g_string_sprintf (error_text, "unknown keyword: %s", left);
  }
}


void auto_games_main (GtkWidget *widget, gamepick_info *info)
{
  game_struct *game;
  gint total, index, target_tab;

  if (!info->auto_games_list)
    return;

  total = g_list_length (info->auto_games_list);
  
  for (index = 0; index < total; index++)
  {
    game = (game_struct*) g_list_nth (info->auto_games_list, index)->data;
    target_tab = gtk_combo_box_get_active (GTK_COMBO_BOX (game->button));
    if (target_tab > 0) /* this is index of the drop down menu */
      add_game (info, game, target_tab - 1);
  }

  empty_auto_games (info);
  gtk_widget_destroy (info->auto_dialog);
  save_config (info);
}


void auto_games_run (gamepick_info *info)
{
  GtkWidget *vbox, *hbox, *button, *add_button;
  gint index;
  GString *conf_file;

  info->auto_dialog = gtk_dialog_new_with_buttons ("Auto Games",
      GTK_WINDOW (info->main_window), GTK_DIALOG_NO_SEPARATOR, NULL);
  gtk_window_set_transient_for (GTK_WINDOW (info->auto_dialog),
    GTK_WINDOW (info->main_window));
  gtk_window_set_modal (GTK_WINDOW (info->auto_dialog), TRUE);
  gtk_container_set_border_width (GTK_CONTAINER (info->auto_dialog),
    info->button_spacing);
  vbox = GTK_DIALOG (info->auto_dialog)->vbox;

  info->auto_tabs = gtk_notebook_new();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (info->auto_tabs), GTK_POS_TOP);
  gtk_container_add (GTK_CONTAINER (vbox), info->auto_tabs);
  gtk_widget_show (info->auto_tabs);

  hbox = gtk_hbox_new (FALSE, info->icon_spacing);

  add_button = gtk_button_new_with_label ("Add selected games to chosen tabs");
  g_signal_connect (G_OBJECT (add_button), "clicked",
    G_CALLBACK (auto_games_main), info); 
  gtk_box_pack_start (GTK_BOX (hbox), add_button, FALSE, FALSE, 0);

  button = gtk_button_new_with_label ("Cancel");
  g_signal_connect (G_OBJECT (button), "clicked", G_CALLBACK (close_dialog),
    info->auto_dialog); 
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  gtk_widget_show (button);

  gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, FALSE, info->button_spacing);
  gtk_widget_show (hbox);
  
  info->auto_games_total = 0; /* info->auto_games_count done by add tab */
  empty_auto_games (info);
  info->auto_games_page = 1;
  auto_add_tab (info, "Page 1");

  for (index = 0; index < g_list_length (info->auto_find_list); index++)
  {
    conf_file = (GString*) g_list_nth (info->auto_find_list, index)->data;
    load_knowns (info, conf_file->str);
  }

  conf_file = g_string_new (NULL);
  g_string_sprintf (conf_file, "%s/%s", GPDIR, global_knowns);
  if (file_readable (conf_file->str))
    load_knowns (info, conf_file->str);
  
  if (g_list_length (info->auto_games_list) > 0)
    gtk_widget_show (add_button);

  gtk_widget_show (info->auto_dialog);
}
