﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/chime/ChimeRequest.h>
#include <aws/chime/Chime_EXPORTS.h>
#include <aws/chime/model/License.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Chime {
namespace Model {

/**
 */
class UpdateAccountRequest : public ChimeRequest {
 public:
  AWS_CHIME_API UpdateAccountRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateAccount"; }

  AWS_CHIME_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon Chime account ID.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  UpdateAccountRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The new name for the specified Amazon Chime account.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateAccountRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The default license applied when you add users to an Amazon Chime
   * account.</p>
   */
  inline License GetDefaultLicense() const { return m_defaultLicense; }
  inline bool DefaultLicenseHasBeenSet() const { return m_defaultLicenseHasBeenSet; }
  inline void SetDefaultLicense(License value) {
    m_defaultLicenseHasBeenSet = true;
    m_defaultLicense = value;
  }
  inline UpdateAccountRequest& WithDefaultLicense(License value) {
    SetDefaultLicense(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_accountId;
  bool m_accountIdHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  License m_defaultLicense{License::NOT_SET};
  bool m_defaultLicenseHasBeenSet = false;
};

}  // namespace Model
}  // namespace Chime
}  // namespace Aws
