﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codeartifact/CodeArtifactRequest.h>
#include <aws/codeartifact/CodeArtifact_EXPORTS.h>
#include <aws/codeartifact/model/AllowPublish.h>
#include <aws/codeartifact/model/AllowUpstream.h>
#include <aws/codeartifact/model/PackageFormat.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace CodeArtifact {
namespace Model {

/**
 */
class ListPackagesRequest : public CodeArtifactRequest {
 public:
  AWS_CODEARTIFACT_API ListPackagesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListPackages"; }

  AWS_CODEARTIFACT_API Aws::String SerializePayload() const override;

  AWS_CODEARTIFACT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p> The name of the domain that contains the repository that contains the
   * requested packages. </p>
   */
  inline const Aws::String& GetDomain() const { return m_domain; }
  inline bool DomainHasBeenSet() const { return m_domainHasBeenSet; }
  template <typename DomainT = Aws::String>
  void SetDomain(DomainT&& value) {
    m_domainHasBeenSet = true;
    m_domain = std::forward<DomainT>(value);
  }
  template <typename DomainT = Aws::String>
  ListPackagesRequest& WithDomain(DomainT&& value) {
    SetDomain(std::forward<DomainT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The 12-digit account number of the Amazon Web Services account that owns the
   * domain. It does not include dashes or spaces. </p>
   */
  inline const Aws::String& GetDomainOwner() const { return m_domainOwner; }
  inline bool DomainOwnerHasBeenSet() const { return m_domainOwnerHasBeenSet; }
  template <typename DomainOwnerT = Aws::String>
  void SetDomainOwner(DomainOwnerT&& value) {
    m_domainOwnerHasBeenSet = true;
    m_domainOwner = std::forward<DomainOwnerT>(value);
  }
  template <typename DomainOwnerT = Aws::String>
  ListPackagesRequest& WithDomainOwner(DomainOwnerT&& value) {
    SetDomainOwner(std::forward<DomainOwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The name of the repository that contains the requested packages. </p>
   */
  inline const Aws::String& GetRepository() const { return m_repository; }
  inline bool RepositoryHasBeenSet() const { return m_repositoryHasBeenSet; }
  template <typename RepositoryT = Aws::String>
  void SetRepository(RepositoryT&& value) {
    m_repositoryHasBeenSet = true;
    m_repository = std::forward<RepositoryT>(value);
  }
  template <typename RepositoryT = Aws::String>
  ListPackagesRequest& WithRepository(RepositoryT&& value) {
    SetRepository(std::forward<RepositoryT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The format used to filter requested packages. Only packages from the provided
   * format will be returned.</p>
   */
  inline PackageFormat GetFormat() const { return m_format; }
  inline bool FormatHasBeenSet() const { return m_formatHasBeenSet; }
  inline void SetFormat(PackageFormat value) {
    m_formatHasBeenSet = true;
    m_format = value;
  }
  inline ListPackagesRequest& WithFormat(PackageFormat value) {
    SetFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The namespace prefix used to filter requested packages. Only packages with a
   * namespace that starts with the provided string value are returned. Note that
   * although this option is called <code>--namespace</code> and not
   * <code>--namespace-prefix</code>, it has prefix-matching behavior.</p> <p>Each
   * package format uses namespace as follows:</p> <ul> <li> <p> The namespace of a
   * Maven package version is its <code>groupId</code>. </p> </li> <li> <p> The
   * namespace of an npm or Swift package version is its <code>scope</code>. </p>
   * </li> <li> <p>The namespace of a generic package is its
   * <code>namespace</code>.</p> </li> <li> <p> Python, NuGet, Ruby, and Cargo
   * package versions do not contain a corresponding component, package versions of
   * those formats do not have a namespace. </p> </li> </ul>
   */
  inline const Aws::String& GetNamespace() const { return m_namespace; }
  inline bool NamespaceHasBeenSet() const { return m_namespaceHasBeenSet; }
  template <typename NamespaceT = Aws::String>
  void SetNamespace(NamespaceT&& value) {
    m_namespaceHasBeenSet = true;
    m_namespace = std::forward<NamespaceT>(value);
  }
  template <typename NamespaceT = Aws::String>
  ListPackagesRequest& WithNamespace(NamespaceT&& value) {
    SetNamespace(std::forward<NamespaceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A prefix used to filter requested packages. Only packages with names that
   * start with <code>packagePrefix</code> are returned. </p>
   */
  inline const Aws::String& GetPackagePrefix() const { return m_packagePrefix; }
  inline bool PackagePrefixHasBeenSet() const { return m_packagePrefixHasBeenSet; }
  template <typename PackagePrefixT = Aws::String>
  void SetPackagePrefix(PackagePrefixT&& value) {
    m_packagePrefixHasBeenSet = true;
    m_packagePrefix = std::forward<PackagePrefixT>(value);
  }
  template <typename PackagePrefixT = Aws::String>
  ListPackagesRequest& WithPackagePrefix(PackagePrefixT&& value) {
    SetPackagePrefix(std::forward<PackagePrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The maximum number of results to return per page. </p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListPackagesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The token for the next set of results. Use the value returned in the
   * previous response in the next request to retrieve the next set of results. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListPackagesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value of the <code>Publish</code> package origin control restriction used
   * to filter requested packages. Only packages with the provided restriction are
   * returned. For more information, see <a
   * href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageOriginRestrictions.html">PackageOriginRestrictions</a>.</p>
   */
  inline AllowPublish GetPublish() const { return m_publish; }
  inline bool PublishHasBeenSet() const { return m_publishHasBeenSet; }
  inline void SetPublish(AllowPublish value) {
    m_publishHasBeenSet = true;
    m_publish = value;
  }
  inline ListPackagesRequest& WithPublish(AllowPublish value) {
    SetPublish(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value of the <code>Upstream</code> package origin control restriction
   * used to filter requested packages. Only packages with the provided restriction
   * are returned. For more information, see <a
   * href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageOriginRestrictions.html">PackageOriginRestrictions</a>.</p>
   */
  inline AllowUpstream GetUpstream() const { return m_upstream; }
  inline bool UpstreamHasBeenSet() const { return m_upstreamHasBeenSet; }
  inline void SetUpstream(AllowUpstream value) {
    m_upstreamHasBeenSet = true;
    m_upstream = value;
  }
  inline ListPackagesRequest& WithUpstream(AllowUpstream value) {
    SetUpstream(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_domain;
  bool m_domainHasBeenSet = false;

  Aws::String m_domainOwner;
  bool m_domainOwnerHasBeenSet = false;

  Aws::String m_repository;
  bool m_repositoryHasBeenSet = false;

  PackageFormat m_format{PackageFormat::NOT_SET};
  bool m_formatHasBeenSet = false;

  Aws::String m_namespace;
  bool m_namespaceHasBeenSet = false;

  Aws::String m_packagePrefix;
  bool m_packagePrefixHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  AllowPublish m_publish{AllowPublish::NOT_SET};
  bool m_publishHasBeenSet = false;

  AllowUpstream m_upstream{AllowUpstream::NOT_SET};
  bool m_upstreamHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeArtifact
}  // namespace Aws
