﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/codeartifact/model/EndpointType.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>

using namespace Aws::Utils;

namespace Aws {
namespace CodeArtifact {
namespace Model {
namespace EndpointTypeMapper {

static const int dualstack_HASH = HashingUtils::HashString("dualstack");
static const int ipv4_HASH = HashingUtils::HashString("ipv4");

EndpointType GetEndpointTypeForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == dualstack_HASH) {
    return EndpointType::dualstack;
  } else if (hashCode == ipv4_HASH) {
    return EndpointType::ipv4;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<EndpointType>(hashCode);
  }

  return EndpointType::NOT_SET;
}

Aws::String GetNameForEndpointType(EndpointType enumValue) {
  switch (enumValue) {
    case EndpointType::NOT_SET:
      return {};
    case EndpointType::dualstack:
      return "dualstack";
    case EndpointType::ipv4:
      return "ipv4";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace EndpointTypeMapper
}  // namespace Model
}  // namespace CodeArtifact
}  // namespace Aws
