﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codeguruprofiler/CodeGuruProfilerRequest.h>
#include <aws/codeguruprofiler/CodeGuruProfiler_EXPORTS.h>
#include <aws/codeguruprofiler/model/AggregationPeriod.h>
#include <aws/codeguruprofiler/model/OrderBy.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace CodeGuruProfiler {
namespace Model {

/**
 * <p>The structure representing the listProfileTimesRequest.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/ListProfileTimesRequest">AWS
 * API Reference</a></p>
 */
class ListProfileTimesRequest : public CodeGuruProfilerRequest {
 public:
  AWS_CODEGURUPROFILER_API ListProfileTimesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListProfileTimes"; }

  AWS_CODEGURUPROFILER_API Aws::String SerializePayload() const override;

  AWS_CODEGURUPROFILER_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The end time of the time range from which to list the profiles.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ListProfileTimesRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of profile time results returned by
   * <code>ListProfileTimes</code> in paginated output. When this parameter is used,
   * <code>ListProfileTimes</code> only returns <code>maxResults</code> results in a
   * single page with a <code>nextToken</code> response element. The remaining
   * results of the initial request can be seen by sending another
   * <code>ListProfileTimes</code> request with the returned <code>nextToken</code>
   * value. </p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListProfileTimesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>nextToken</code> value returned from a previous paginated
   * <code>ListProfileTimes</code> request where <code>maxResults</code> was used and
   * the results exceeded the value of that parameter. Pagination continues from the
   * end of the previous results that returned the <code>nextToken</code> value. </p>
   *  <p>This token should be treated as an opaque identifier that is only used
   * to retrieve the next items in a list and not for other programmatic
   * purposes.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListProfileTimesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The order (ascending or descending by start time of the profile) to use when
   * listing profiles. Defaults to <code>TIMESTAMP_DESCENDING</code>. </p>
   */
  inline OrderBy GetOrderBy() const { return m_orderBy; }
  inline bool OrderByHasBeenSet() const { return m_orderByHasBeenSet; }
  inline void SetOrderBy(OrderBy value) {
    m_orderByHasBeenSet = true;
    m_orderBy = value;
  }
  inline ListProfileTimesRequest& WithOrderBy(OrderBy value) {
    SetOrderBy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The aggregation period. This specifies the period during which an
   * aggregation profile collects posted agent profiles for a profiling group. There
   * are 3 valid values. </p> <ul> <li> <p> <code>P1D</code> — 1 day </p> </li> <li>
   * <p> <code>PT1H</code> — 1 hour </p> </li> <li> <p> <code>PT5M</code> — 5 minutes
   * </p> </li> </ul>
   */
  inline AggregationPeriod GetPeriod() const { return m_period; }
  inline bool PeriodHasBeenSet() const { return m_periodHasBeenSet; }
  inline void SetPeriod(AggregationPeriod value) {
    m_periodHasBeenSet = true;
    m_period = value;
  }
  inline ListProfileTimesRequest& WithPeriod(AggregationPeriod value) {
    SetPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the profiling group.</p>
   */
  inline const Aws::String& GetProfilingGroupName() const { return m_profilingGroupName; }
  inline bool ProfilingGroupNameHasBeenSet() const { return m_profilingGroupNameHasBeenSet; }
  template <typename ProfilingGroupNameT = Aws::String>
  void SetProfilingGroupName(ProfilingGroupNameT&& value) {
    m_profilingGroupNameHasBeenSet = true;
    m_profilingGroupName = std::forward<ProfilingGroupNameT>(value);
  }
  template <typename ProfilingGroupNameT = Aws::String>
  ListProfileTimesRequest& WithProfilingGroupName(ProfilingGroupNameT&& value) {
    SetProfilingGroupName(std::forward<ProfilingGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The start time of the time range from which to list the profiles.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ListProfileTimesRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  OrderBy m_orderBy{OrderBy::NOT_SET};
  bool m_orderByHasBeenSet = false;

  AggregationPeriod m_period{AggregationPeriod::NOT_SET};
  bool m_periodHasBeenSet = false;

  Aws::String m_profilingGroupName;
  bool m_profilingGroupNameHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeGuruProfiler
}  // namespace Aws
