﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/comprehend/ComprehendRequest.h>
#include <aws/comprehend/Comprehend_EXPORTS.h>
#include <aws/comprehend/model/DatasetInputDataConfig.h>
#include <aws/comprehend/model/DatasetType.h>
#include <aws/comprehend/model/Tag.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Comprehend {
namespace Model {

/**
 */
class CreateDatasetRequest : public ComprehendRequest {
 public:
  AWS_COMPREHEND_API CreateDatasetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDataset"; }

  AWS_COMPREHEND_API Aws::String SerializePayload() const override;

  AWS_COMPREHEND_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Number (ARN) of the flywheel of the flywheel to receive
   * the data.</p>
   */
  inline const Aws::String& GetFlywheelArn() const { return m_flywheelArn; }
  inline bool FlywheelArnHasBeenSet() const { return m_flywheelArnHasBeenSet; }
  template <typename FlywheelArnT = Aws::String>
  void SetFlywheelArn(FlywheelArnT&& value) {
    m_flywheelArnHasBeenSet = true;
    m_flywheelArn = std::forward<FlywheelArnT>(value);
  }
  template <typename FlywheelArnT = Aws::String>
  CreateDatasetRequest& WithFlywheelArn(FlywheelArnT&& value) {
    SetFlywheelArn(std::forward<FlywheelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Name of the dataset.</p>
   */
  inline const Aws::String& GetDatasetName() const { return m_datasetName; }
  inline bool DatasetNameHasBeenSet() const { return m_datasetNameHasBeenSet; }
  template <typename DatasetNameT = Aws::String>
  void SetDatasetName(DatasetNameT&& value) {
    m_datasetNameHasBeenSet = true;
    m_datasetName = std::forward<DatasetNameT>(value);
  }
  template <typename DatasetNameT = Aws::String>
  CreateDatasetRequest& WithDatasetName(DatasetNameT&& value) {
    SetDatasetName(std::forward<DatasetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The dataset type. You can specify that the data in a dataset is for training
   * the model or for testing the model.</p>
   */
  inline DatasetType GetDatasetType() const { return m_datasetType; }
  inline bool DatasetTypeHasBeenSet() const { return m_datasetTypeHasBeenSet; }
  inline void SetDatasetType(DatasetType value) {
    m_datasetTypeHasBeenSet = true;
    m_datasetType = value;
  }
  inline CreateDatasetRequest& WithDatasetType(DatasetType value) {
    SetDatasetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Description of the dataset.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateDatasetRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the input data configuration. The type of input data varies
   * based on the format of the input and whether the data is for a classifier model
   * or an entity recognition model.</p>
   */
  inline const DatasetInputDataConfig& GetInputDataConfig() const { return m_inputDataConfig; }
  inline bool InputDataConfigHasBeenSet() const { return m_inputDataConfigHasBeenSet; }
  template <typename InputDataConfigT = DatasetInputDataConfig>
  void SetInputDataConfig(InputDataConfigT&& value) {
    m_inputDataConfigHasBeenSet = true;
    m_inputDataConfig = std::forward<InputDataConfigT>(value);
  }
  template <typename InputDataConfigT = DatasetInputDataConfig>
  CreateDatasetRequest& WithInputDataConfig(InputDataConfigT&& value) {
    SetInputDataConfig(std::forward<InputDataConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the request. If you don't set the client request
   * token, Amazon Comprehend generates one.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  CreateDatasetRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags for the dataset.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateDatasetRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateDatasetRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_flywheelArn;
  bool m_flywheelArnHasBeenSet = false;

  Aws::String m_datasetName;
  bool m_datasetNameHasBeenSet = false;

  DatasetType m_datasetType{DatasetType::NOT_SET};
  bool m_datasetTypeHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  DatasetInputDataConfig m_inputDataConfig;
  bool m_inputDataConfigHasBeenSet = false;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientRequestTokenHasBeenSet = true;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Comprehend
}  // namespace Aws
