﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/comprehendmedical/ComprehendMedicalRequest.h>
#include <aws/comprehendmedical/ComprehendMedical_EXPORTS.h>
#include <aws/comprehendmedical/model/InputDataConfig.h>
#include <aws/comprehendmedical/model/LanguageCode.h>
#include <aws/comprehendmedical/model/OutputDataConfig.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ComprehendMedical {
namespace Model {

/**
 */
class StartEntitiesDetectionV2JobRequest : public ComprehendMedicalRequest {
 public:
  AWS_COMPREHENDMEDICAL_API StartEntitiesDetectionV2JobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartEntitiesDetectionV2Job"; }

  AWS_COMPREHENDMEDICAL_API Aws::String SerializePayload() const override;

  AWS_COMPREHENDMEDICAL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The input configuration that specifies the format and location of the input
   * data for the job.</p>
   */
  inline const InputDataConfig& GetInputDataConfig() const { return m_inputDataConfig; }
  inline bool InputDataConfigHasBeenSet() const { return m_inputDataConfigHasBeenSet; }
  template <typename InputDataConfigT = InputDataConfig>
  void SetInputDataConfig(InputDataConfigT&& value) {
    m_inputDataConfigHasBeenSet = true;
    m_inputDataConfig = std::forward<InputDataConfigT>(value);
  }
  template <typename InputDataConfigT = InputDataConfig>
  StartEntitiesDetectionV2JobRequest& WithInputDataConfig(InputDataConfigT&& value) {
    SetInputDataConfig(std::forward<InputDataConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The output configuration that specifies where to send the output files.</p>
   */
  inline const OutputDataConfig& GetOutputDataConfig() const { return m_outputDataConfig; }
  inline bool OutputDataConfigHasBeenSet() const { return m_outputDataConfigHasBeenSet; }
  template <typename OutputDataConfigT = OutputDataConfig>
  void SetOutputDataConfig(OutputDataConfigT&& value) {
    m_outputDataConfigHasBeenSet = true;
    m_outputDataConfig = std::forward<OutputDataConfigT>(value);
  }
  template <typename OutputDataConfigT = OutputDataConfig>
  StartEntitiesDetectionV2JobRequest& WithOutputDataConfig(OutputDataConfigT&& value) {
    SetOutputDataConfig(std::forward<OutputDataConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the AWS Identity and Access Management
   * (IAM) role that grants Amazon Comprehend Medical read access to your input data.
   * For more information, see <a
   * href="https://docs.aws.amazon.com/comprehend/latest/dg/access-control-managing-permissions-med.html#auth-role-permissions-med">Role-Based
   * Permissions Required for Asynchronous Operations</a>.</p>
   */
  inline const Aws::String& GetDataAccessRoleArn() const { return m_dataAccessRoleArn; }
  inline bool DataAccessRoleArnHasBeenSet() const { return m_dataAccessRoleArnHasBeenSet; }
  template <typename DataAccessRoleArnT = Aws::String>
  void SetDataAccessRoleArn(DataAccessRoleArnT&& value) {
    m_dataAccessRoleArnHasBeenSet = true;
    m_dataAccessRoleArn = std::forward<DataAccessRoleArnT>(value);
  }
  template <typename DataAccessRoleArnT = Aws::String>
  StartEntitiesDetectionV2JobRequest& WithDataAccessRoleArn(DataAccessRoleArnT&& value) {
    SetDataAccessRoleArn(std::forward<DataAccessRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the job.</p>
   */
  inline const Aws::String& GetJobName() const { return m_jobName; }
  inline bool JobNameHasBeenSet() const { return m_jobNameHasBeenSet; }
  template <typename JobNameT = Aws::String>
  void SetJobName(JobNameT&& value) {
    m_jobNameHasBeenSet = true;
    m_jobName = std::forward<JobNameT>(value);
  }
  template <typename JobNameT = Aws::String>
  StartEntitiesDetectionV2JobRequest& WithJobName(JobNameT&& value) {
    SetJobName(std::forward<JobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the request. If you don't set the client request
   * token, Amazon Comprehend Medical generates one for you.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  StartEntitiesDetectionV2JobRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An AWS Key Management Service key to encrypt your output files. If you do not
   * specify a key, the files are written in plain text.</p>
   */
  inline const Aws::String& GetKMSKey() const { return m_kMSKey; }
  inline bool KMSKeyHasBeenSet() const { return m_kMSKeyHasBeenSet; }
  template <typename KMSKeyT = Aws::String>
  void SetKMSKey(KMSKeyT&& value) {
    m_kMSKeyHasBeenSet = true;
    m_kMSKey = std::forward<KMSKeyT>(value);
  }
  template <typename KMSKeyT = Aws::String>
  StartEntitiesDetectionV2JobRequest& WithKMSKey(KMSKeyT&& value) {
    SetKMSKey(std::forward<KMSKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The language of the input documents. All documents must be in the same
   * language. Amazon Comprehend Medical processes files in US English (en).</p>
   */
  inline LanguageCode GetLanguageCode() const { return m_languageCode; }
  inline bool LanguageCodeHasBeenSet() const { return m_languageCodeHasBeenSet; }
  inline void SetLanguageCode(LanguageCode value) {
    m_languageCodeHasBeenSet = true;
    m_languageCode = value;
  }
  inline StartEntitiesDetectionV2JobRequest& WithLanguageCode(LanguageCode value) {
    SetLanguageCode(value);
    return *this;
  }
  ///@}
 private:
  InputDataConfig m_inputDataConfig;
  bool m_inputDataConfigHasBeenSet = false;

  OutputDataConfig m_outputDataConfig;
  bool m_outputDataConfigHasBeenSet = false;

  Aws::String m_dataAccessRoleArn;
  bool m_dataAccessRoleArnHasBeenSet = false;

  Aws::String m_jobName;
  bool m_jobNameHasBeenSet = false;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientRequestTokenHasBeenSet = true;

  Aws::String m_kMSKey;
  bool m_kMSKeyHasBeenSet = false;

  LanguageCode m_languageCode{LanguageCode::NOT_SET};
  bool m_languageCodeHasBeenSet = false;
};

}  // namespace Model
}  // namespace ComprehendMedical
}  // namespace Aws
