/*
 * Utility functions
 *
 * Copyright (C) 2000-2005, Olaf Kirch <okir@suse.de>
 *
 * License: GPL 2.0 or later. See COPYING.GPL for details.
 */
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include "utils.h"


/* Older glibc versions do not have IDN support */
#ifndef AI_IDN
# define AI_IDN	0
#endif

#define AF_LEN(af)	(((af) == AF_INET)? 4 : 16)
#define SIN_FMLY(ap)	((ap)->any.sa_family)
#define SIN_LEN(ap)	AF_LEN(SIN_FMLY(ap))
#define SIN_ADDR(ap)	((SIN_FMLY(ap) == AF_INET)? \
				(caddr_t) &(ap)->sin.sin_addr : \
				(caddr_t) &(ap)->sin6.sin6_addr )

unsigned int
getnum(const char *what, const char *arg, unsigned int min, unsigned int max)
{
	unsigned int	value;
	char		*end;

	value = strtoul(arg, &end, 0);
	if (*end)
		fatal("bad %s value \"%s\"", what, arg);
	
	if (value < min)
		fatal("bad %s value %u, must be less than %u",
			what, value, min);
	if (value > max)
		fatal("bad %s value %u, must be greater than %u",
			what, value, max);
	return value;
}

int
getaddr(const char *arg, int af, sockaddr_any *ap)
{
	struct addrinfo	hints, *res = NULL, *ai;
	int		rv;

	memset(ap, 0, sizeof(*ap));
	if (af != -1) {
		SIN_FMLY(ap) = af;
		if (inet_pton(af, arg, SIN_ADDR(ap)))
			return 1;
	} else {
		if (inet_pton(AF_INET, arg, &ap->sin.sin_addr)) {
			ap->sin.sin_family = AF_INET;
			return 1;
		}
		if (inet_pton(AF_INET6, arg, &ap->sin6.sin6_addr)) {
			ap->sin6.sin6_family = AF_INET6;
			return 1;
		}
	}

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = (af == -1)? PF_UNSPEC : af;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_flags = AI_CANONNAME|AI_IDN;

	if ((rv = getaddrinfo(arg, NULL, &hints, &res)) != 0) {
		fprintf(stderr, "%s: %s\n", arg, gai_strerror(rv));
		return 0;
	}

	/* If no address family specified, prefer v4 (for now) */
	for (ai = res; af != -1 && ai && ai->ai_family != AF_INET; ai = ai->ai_next)
		;
	if (ai == NULL)
		ai = res;

	if ((af != -1 && ai->ai_family != af)
	 || ai->ai_family != ai->ai_addr->sa_family
	 || ai->ai_addrlen > sizeof(*ap)) {
		fprintf(stderr,
			"%s: inconsistent data from getaddrinfo\n",
			arg);
		return 0;
	}

	memcpy(ap, ai->ai_addr, ai->ai_addrlen);
	freeaddrinfo(res);
	return 1;
}

const char *
getfqdn(sockaddr_any *ap, int resolve)
{
	static char	namebuf[4096];
	socklen_t	salen;
	int		rv, flags;

	switch (SIN_FMLY(ap)) {
	case AF_INET:
		salen = sizeof(struct sockaddr_in);
		break;
	case AF_INET6:
		salen = sizeof(struct sockaddr_in6);
		break;
	default:
		return NULL;
	}

	flags = NI_IDN;
	if (!resolve)
		flags |= NI_NUMERICHOST;

	/* We do not set NI_NAMEREQD, so this should fall back to the
	 * numeric address in the NXDOMAIN case */
	rv = getnameinfo((struct sockaddr *) ap, salen,
			namebuf, sizeof(namebuf),
			NULL, 0, flags);
	if (rv != 0) {
		fprintf(stderr, "Unable to look up %s: %s\n",
				straddr(ap), gai_strerror(rv));
		return NULL;
	}

	return namebuf;
}

const char *
straddr(sockaddr_any *ap)
{
	static char	buffer[INET6_ADDRSTRLEN];

	return inet_ntop(ap->any.sa_family,
			SIN_ADDR(ap), buffer, sizeof(buffer));
}

int
sameaddr(sockaddr_any *ap1, sockaddr_any *ap2)
{
	return SIN_FMLY(ap1) == SIN_FMLY(ap2)
	    && !memcmp(SIN_ADDR(ap1), SIN_ADDR(ap2), SIN_LEN(ap1));
}

void
printaddr(sockaddr_any *ap, int resolve)
{
	const char	*fqdn;

	if ((fqdn = getfqdn(ap, resolve)) != NULL) {
		printf("%s (%s)", fqdn, straddr(ap));
	} else {
		printf("%s", straddr(ap));
	}
}

void
fatal(const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	if (!strchr(fmt, '\n'))
		fputs("\n", stderr);
	va_end(ap);
	exit(1);
}
