#
# Copyright 2007 Fedora Unity
#
# Jonathan Steffan <jon a fedoraunity.org>
# Jeroen van Meeuwen <kanarip a fedoraunity.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 only
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import pykickstart
import revisor
import yum
from ConfigParser import SafeConfigParser
from ConfigParser import RawConfigParser
import string
import subprocess
import time
import math
import shutil
import re
import fnmatch
import yum.misc

import revisor.cfg
import revisor.progress
from revisor import lm_optical

# Import constants
from revisor.constants import *

# Translation
from rhpl.translate import _, N_, getDefaultLangs
import rhpl.translate as translate

class RevisorBase():
    """RevisorBase holds common functions shared amongst our CLI and GUI mode"""
    def __init__(self, cfg = None, log = None):

        if cfg:
            self.cfg = cfg
        if log:
            self.log = log

        self.cfg.setup_yum()
        self.cfg.setup_ks()

    def show_help(self, keyword):
        if self.cfg.revisorUseGUI:
            self.mode.base_buttons_xml.get_widget("button_information").set_sensitive(False)
            pid = os.fork()
            if not pid:
                self.log.info(_("Opening up /usr/bin/htmlview %s%s") % (DOCS_BASEPATH, keyword))
                os.execv("/usr/bin/htmlview", ["/usr/bin/htmlview", "%s%s" % (DOCS_BASEPATH, keyword)])
            else:
#                self.mode.base_buttons_xml.get_widget("button_information").set_sensitive(True)
                self.log.error(_("Cannot fork process showing help, please use %s%s") % (DOCS_BASEPATH, keyword))
        else:
            self.log.warning(_("Cannot show Help in CLI mode, use %s%s") % (DOCS_BASEPATH, keyword))

    def setup_yum(self, pbar):
        """Setup the yum object"""
        self.cfg.yumobj.pbar = pbar

        try:
            self.cfg.yumobj.doConfigSetup(fn=self.cfg.main_conf,plugin_types=(yum.plugins.TYPE_CORE,))
        except:
            self.log.error(_("yum.YumBase.doConfigSetup failed, probably an invalid configuration file %s") % self.cfg.main_conf)

        try:
            self.cfg.reposSetup(self.cfg.yumobj.pbar)
            repo_setup = True
        except yum.Errors.LockError:
            self.cfg.yumobj.pbar.destroy()
            self.log.error(_("Another application is running which is accessing software information."))
            repo_setup = False
        except revisor.errors.RevisorDownloadError, e:
            self.cfg.yumobj.pbar.destroy()
            self.log.error(_("Unable to retrieve software information. This could be caused by not having a network connection available."))
            repo_setup = False

        if repo_setup:
            self.cfg.yumobj.doTsSetup()
            self.cfg.yumobj.doRpmDBSetup()
            self.cfg.yumobj.doRepoSetup()

            try:
                self.cfg.yumobj.doSackSetup(archlist=self.cfg.arch_list)
            except:
                pass

            self.cfg.yumobj.pbar.destroy()
            return True
        else:
            self.cfg.yumobj.pbar.destroy()
            return False

    def pkglist_from_ksdata(self):
        """Takes valid kshandler.packages.groupList, kshandler.packages.packageList
        and kshandler.packages.excludedList, and builds a nice and simple list that
        we can add our required packages to, and then start depsolving"""

        self.log.debug(_("Building a nice package list from ksdata, and adding it to the transaction"))

##
## Add packages from Groups in ksdata
##
        for ksgroup in self.cfg.kshandler.packages.groupList:
            self.log.debug(_("Found group: %s") % ksgroup.name)
            # Get group object from comps
            grp = self.cfg.yumobj.comps.return_group(ksgroup.name)
            # Add group to package set
            txmbrs_used = self.cfg.yumobj.selectGroup(grp.name)
            self.log.debug(_("From Groups: Adding %s to transaction") % grp.groupid)
            # See if this group has --nodefaults or --optional
            # Group has --optional, add all optional packages
            if ksgroup.include == pykickstart.constants.GROUP_ALL:
                self.log.debug(_("Using all optional packages for group %s") % grp.name)
                # All optional packages included, get the package names
                for pkg in grp.optional_packages.keys():
                    if pkg in self.cfg.kshandler.packages.excludedList:
                        continue
                    self.log.debug(_("Including %s") % pkg)
                    # Get the package objects
                    try:
                        pkgs = self.cfg.yumobj.pkgSack.returnNewestByName(pkg)
                        if len(pkgs) > 1:
                            pkgs = self.cfg.yumobj.bestPackagesFromList(pkgs)
                        # Add to transaction
                        for po in pkgs:
                            if not po.name in self.cfg.required_pkgs_installation and not po.name in self.cfg.required_pkgs_live:
                                self.cfg.yumobj.tsInfo.addInstall(po)
                                self.log.debug(_("From Groups --optional: Adding %s-%s:%s-%s.%s to transaction") % (po.name, po.epoch, po.version, po.release, po.arch))
                            else:
                                self.log.debug(_("Package %s not added because we don't like it being added") % po.name)
                    except yum.Errors.PackageSackError, e:
                        self.log.warning(e.value)

            # Group has --nodefaults, remove defaults
            elif ksgroup.include == pykickstart.constants.GROUP_REQUIRED:
                self.log.debug(_("Not using defaults for group %s") % grp.name)
                # All default packages excluded, get the package names
                for pkg in grp.default_packages.keys():
                    self.log.debug(_("Excluding %s") % pkg)
                    # Get the package objects
                    try:
                        pkgs = self.cfg.yumobj.pkgSack.returnNewestByName(pkg)
                        # Remove from transaction
                        for po in pkgs:
                            if not po.name in self.cfg.required_pkgs_installation and not po.name in self.cfg.required_pkgs_live:
                                self.cfg.yumobj.tsInfo.remove(po)
                                self.log.debug(_("From Group --nodefaults: Removing %s-%s:%s-%s.%s from transaction") % (po.name, po.epoch, po.version, po.release, po.arch))
                            else:
                                self.log.debug(_("Package %s not excluded because we don't like it being excluded") % po.name)
                    except yum.Errors.PackageSackError, e:
                        self.log.warning(e.value)
                        pass
##
## Pull the conditional packages in
##
            for condreq, cond in grp.conditional_packages.iteritems():
                self.log.debug(_("Testing condition: %s / %s") % (condreq, cond))
                pkgs = self.cfg.yumobj.pkgSack.searchNevra(name=condreq)
                if len(pkgs) > 1:
                    pkgs = self.cfg.yumobj.bestPackagesFromList(pkgs)
                if self.cfg.yumobj.tsInfo.conditionals.has_key(cond):
                    self.cfg.yumobj.tsInfo.conditionals[cond].extend(pkgs)
                else:
                    self.cfg.yumobj.tsInfo.conditionals[cond] = pkgs


##
## Add packages in ksdata to transaction
##
        for pkg in self.cfg.kshandler.packages.packageList:
            self.log.debug(_("From package list, including: %s") % pkg)
            if pkg in self.cfg.kshandler.packages.excludedList:
                continue
            try:
                pkgs = self.cfg.yumobj.pkgSack.returnNewestByName(pkg)
                if len(pkgs) > 1:
                    pkgs = self.cfg.yumobj.bestPackagesFromList(pkgs)

                for po in pkgs:
                    if not po.name in self.cfg.required_pkgs_installation and not po.name in self.cfg.required_pkgs_live:
                        self.cfg.yumobj.tsInfo.addInstall(po)
                        self.log.debug(_("From Packages: Adding %s-%s:%s-%s.%s to transaction") % (po.name, po.epoch, po.version, po.release, po.arch))
                    else:
                        self.log.debug(_("Package %s not excluded because we don't like it being excluded") % po.name)
            except yum.Errors.PackageSackError, e:
                # Let's see if there's some regexp, and if there is, let's search
                if "*" in pkg:
                    try:
                        pkglist = self.cfg.yumobj.pkgSack.simplePkgList()
                        matches = []
                        # anything we couldn't find a match for
                        # could mean it's not there, could mean it's a wildcard
                        if re.match('.*[\*,\[,\],\{,\},\?].*', pkg):
                            restring = fnmatch.translate(pkg)
                            regex = re.compile(restring, flags=re.I) # case insensitive
                            for item in pkglist:
                                if regex.match(item[0]):
                                    matches.append(item[0])
                                    self.log.debug(_("Found packages matching '%s': %s") % (pkg,item[0]))
                        for pkg in matches:
                            pkgs = self.cfg.yumobj.pkgSack.returnNewestByName(pkg)
                            if len(pkgs) > 1:
                                pkgs = self.cfg.yumobj.bestPackagesFromList(pkgs)

                            for po in pkgs:
                                if not po.name in self.cfg.required_pkgs_installation and not po.name in self.cfg.required_pkgs_live:
                                    self.cfg.yumobj.tsInfo.addInstall(po)
                                    self.log.debug(_("From Packages: Adding %s-%s:%s-%s.%s to transaction") % (po.name, po.epoch, po.version, po.release, po.arch))
                                else:
                                    self.log.debug(_("Package %s not excluded because we don't like it being excluded") % po.name)

                    except yum.Errors.PackageSackError, e:
                        self.log.warning(e.value)
                        pass
                else:
                    self.log.warning(e.value)
                    pass

##
## Exclude packages from ksdata
##
        for pkg in self.cfg.kshandler.packages.excludedList:
            if pkg in self.cfg.yumobj.pkgSack.simplePkgList():
                try:
                    pkgs = self.cfg.yumobj.pkgSack.returnNewestByName(pkg)
                    for po in pkgs:
                        txmbrs = self.cfg.yumobj.tsInfo.getMembers(pkgtup=po.pkgtup)
                        self.log.debug(_("From Excludes: Removing %s-%s:%s-%s.%s from transaction") % (po.name, po.epoch, po.version, po.release, po.arch))
                        self.cfg.yumobj.tsInfo.remove(po)
                except yum.Errors.PackageSackError, e:
                    self.log.warning(e.value)

        self.cfg.yumobj.tsInfo.makelists()

        self.log.debug(_("This is what was selected to be installed:"))
        installpkgs = []
        for po in self.cfg.yumobj.tsInfo.installed:
            installpkgs.append(po.pkgtup)
            self.log.debug("--> %s" % str(po.pkgtup))

    def progress_bar(self, title = "", parent = None, xml = None, callback = False):
        """This function should be used to determine the type of progress bar we need.
        There's three types:
            - GUI Dialog Progress Bar (separate dialog, window pops up)
            - GUI Nested Progress Bar (no separate dialog, progress bar is in gui.frame_xml)
            - CLI Progress Bar

        This function also determines whether we should have a Callback Progress Bar"""

# FIXME
        # Have some checks here... Are we in CLI or in GUI mode?
        # Is the parent a GtkWindow -> spawn a dialog
        # Does the parent have a widget "part_progress" -> spawn a pbar
        self.log.debug(_("Initting progress bar for ") + title)

        if self.cfg.revisorUseGUI:
            try:
                if not self.mode.BuildMedia == None:
                    self.mode.BuildMedia.show_task_list()
            except:
                self.log.debug(_("Apparently we have not yet entered the Build Media stage"))

            if callback:
                if not self.mode.frame_xml.get_widget("part_progress") == None:
                    return revisor.progress.ProgressCallbackGUI(title = title, parent = self.mode, xml = self.mode.frame_xml)
                elif not self.mode.main_window == None:
                    return revisor.progress.ProgressCallbackGUI(title = title, parent = self.mode, xml = self.mode.frame_xml)
                else:
                    return revisor.progress.ProgressCallbackGUI(title = title, parent = parent, xml = xml)
            else:
                if not self.mode.frame_xml.get_widget("part_progress") == None:
                    return revisor.progress.ProgressGUI(title = title, parent = self.mode, xml = self.mode.frame_xml)
                elif not parent.main_window == None:
                    print str(parent)
                    return revisor.progress.ProgressGUI(title = title, parent = parent, xml = xml)
                else:
                    return revisor.progress.ProgressGUI(title = title, parent = parent, xml = xml)
        else:
            # No progress bar available yet in CLI mode
            if callback:
                return revisor.progress.ProgressCallbackCLI(title = title)
            else:
                return revisor.progress.ProgressCLI(title = title)

    def set_mode(self, mode):
        self.mode = mode

    def get_package_deps(self, po, pbar):
        """Add the dependencies for a given package to the
           transaction info"""

        self.log.debug(_("Checking dependencies for %s.%s") % (po.name, po.arch))

        reqs = po.requires
        provs = po.provides

        for prov in provs:
            self.resolved_deps[prov] = po.pkgtup

        for req in reqs:
            if self.resolved_deps.has_key(req):
                continue

            (r,f,v) = req
            if r.startswith('rpmlib(') or r.startswith('config('):
                continue

            deps = self.cfg.yumobj.whatProvides(r, f, v).returnPackages()

            if not deps:
                self.log.warning(_("Unresolvable dependency %s %s %s in %s.%s") % (r, f, v, po.name, po.arch))
                continue

            for dep in deps:
                self.cfg.yumobj.tsInfo.addInstall(dep)
#                pbar.progressbar(pbar.cur_task, len(self.cfg.yumobj.tsInfo.getMembers()))
                for prov in dep.provides:
                    self.resolved_deps[prov] = dep.pkgtup

                self.log.debug(_("Added %s.%s for %s.%s") % (dep.name, dep.arch, po.name, po.arch))

    def check_dependencies_no_conflicts(self):

        pbar = self.progress_bar(_("Resolving Dependencies"), callback = True)
        pbar.num_tasks = float(len(self.cfg.yumobj.tsInfo.getMembers()))
        pbar.cur_task = 0
        self.resolved_deps = {}
        final_pkgobjs = {}

        moretoprocess = True
        while moretoprocess: # Our fun loop
            pbar.cur_task = 0
            moretoprocess = False
            for txmbr in self.cfg.yumobj.tsInfo.getMembers():
                pbar.cur_task += 1
                pbar.num_tasks = float(len(self.cfg.yumobj.tsInfo.getMembers()))
                pbar.set_fraction(pbar.cur_task/pbar.num_tasks)
                if not final_pkgobjs.has_key(txmbr.po):
                    final_pkgobjs[txmbr.po] = None # Add the pkg to our final list
                    self.get_package_deps(txmbr.po, pbar) # Get the deps of our package
                    moretoprocess = True
        pbar.destroy()


    def check_dependencies(self):
        """Check the dependencies"""
        self.log.debug(_("Checking dependencies"))
        pbar = self.progress_bar(_("Resolving Dependencies"), callback = True)
#        pbar = self.progress_bar(title = "Resolving Dependencies", xml = self.gui.frame_xml)

        dsCB = revisor.progress.dscb(pbar, self.cfg.yumobj, self.cfg)
        self.cfg.yumobj.dsCallback = dsCB

        try:
            (result, msgs) = self.cfg.yumobj.buildTransaction()
        except yum.Errors.RepoError, errmsg:
            self.cfg.yumobj.dsCallback = None
            pbar.destroy()

            self.log.error(_("""Errors where encountered while downloading package headers:

%s""" % (errmsg,)))

        self.cfg.yumobj.dsCallback = None # we only wanted this here.  blah.
        pbar.destroy()

        if result == 1:

            self.log.error(_("""Unable to resolve dependencies for some packages selected:

%s""" % (string.join(msgs, "\n"))))

        self.cfg.yumobj.tsInfo.makelists()

    def download_packages(self):
            dlpkgs = map(lambda x: x.po, filter(lambda txmbr:
                                                txmbr.ts_state in ("i", "u"),
                                                self.cfg.yumobj.tsInfo.getMembers()))

            pbar = self.progress_bar(_("Downloading Packages"))

            dlCb = revisor.progress.dlcb(pbar, dlpkgs, log = self.log, cfg = self.cfg)
            self.cfg.yumobj.repos.setProgressBar(dlCb)
            try:
                probs = self.cfg.yumobj.downloadPkgs(dlpkgs, dlCb)
            except yum.Errors.RepoError, errmsg:
                self.log.error(errmsg)
            except IndexError:
                self.log.error(_("Unable to find a suitable mirror."))

            self.cfg.yumobj.repos.setProgressBar(None)

            if len(probs.keys()) > 0:
                errstr = []
                for key in probs.keys():
                    errors = yum.misc.unique(probs[key])
                    for error in errors:
                        errstr.append("%s: %s" %(key, error))

                details_str = string.join(errstr, "\n")
                self.log.error(_("Errors were encountered while downloading packages: %s") % details_str)

    def pkglist_required_installation(self):
        """Add all packages that are required for Installation Media"""
        self.log.debug(_("Adding required packages for Installation Media"))

        for pkg in self.cfg.required_pkgs_installation:
            try:
                pkgs = self.cfg.yumobj.pkgSack.returnNewestByName(pkg)
                for po in pkgs:
                    self.cfg.yumobj.tsInfo.addInstall(po)
                    self.log.debug(_("Install Media: Adding req. pkg %s") % pkg)
            except yum.Errors.PackageSackError, e:
                self.log.error(_("%s. This is a required package.") % e.value, recoverable = False)
                return False

        return True

    def pkglist_required_live(self):
        """Add all packages that are required for Live Media"""
        self.log.debug(_("Adding required packages for Live Media"))

        for pkg in self.cfg.required_pkgs_live:
            try:
                pkgs = self.cfg.yumobj.pkgSack.returnNewestByName(pkg)
                if pkgs:
                    pkgs = self.cfg.yumobj.bestPackagesFromList(pkgs)
                for po in pkgs:
                    self.cfg.yumobj.tsInfo.addInstall(po)
                    self.log.debug(_("Live Media: Adding req. pkg %s") % pkg)
            except yum.Errors.PackageSackError, e:
                self.log.error(_("%s. This is a required package.") % e.value)
                return False
        return True

    def lift_off(self):
        """Do the actual thing. Dance the salsa."""
        self.cfg.pkglist_selected_tups = self.cfg.yumobj.tsInfo.pkgdict.keys()

        self.log.debug(_("See, this is our list of packages right now: %s") % str(self.cfg.pkglist_selected_tups))

        if self.cfg.media_installation_dvd or self.cfg.media_installation_cd or self.cfg.media_installation_tree:
            if self.pkglist_required_installation():
                self.check_dependencies()
                self.download_packages()
                self.buildInstallationMedia()
            else:
                self.log.error(_("Did not succeed in adding in all required packages"))

            # If we've done installation media, we want to remove all packages we added
            # as required packages and that were not in the original list
            for pkg in self.cfg.yumobj.tsInfo.pkgdict.keys():
                if not pkg in self.cfg.pkglist_selected_tups:
                    self.log.debug(_("Removing %s from tsInfo") % str(pkg))
                    self.cfg.yumobj.tsInfo.remove(pkg)
                else:
                    self.log.debug(_("Not removing %s from tsInfo") % str(pkg))

        if self.cfg.media_live_thumb or self.cfg.media_live_optical:
            self.log.debug(_("Let's figure out what is in the package sack still"))
            self.log.debug(_("Adding Live Media Required Packages"))

            if self.pkglist_required_live():
                self.check_dependencies()
                self.download_packages()
                self.buildLiveMedia()
            else:
                self.log.error(_("Did not succeed in adding in all required packages"))

        if self.cfg.revisorUseGUI:
            # Show Finshed Screen
            self.mode.displayFinished()

    def populate_stats(self):
        """ Populate the stats displayed on the ready screen, based on our yum transaction. """
        pkgs = self.cfg.yumobj.tsInfo.getMembers()
        self.cfg.payload_packages = len(pkgs)
        pkgsize = 0
        installsize = 0
        for pk in pkgs:
           pkgsize += pk.po.packagesize
           installsize += pk.po.installedsize
        self.cfg.payload_installmedia = pkgsize / 1024 / 1024
        pkgsize = str(self.cfg.payload_installmedia) + " MB"
        self.cfg.payload_livemedia = installsize / 1024 / 1024
        installsize = str(self.cfg.payload_livemedia) + " MB"

    def buildInstallationMedia(self):
        from revisor import pungi

#        relnotefilere = "eula.txt fedora.css GPL README-BURNING-ISOS-en_US.txt RELEASE-NOTES-en_US.html ^RPM-GPG"
#        relnotedirre = "images stylesheet-images"
#        relnotepkgs = "fedora-release fedora-release-notes"

        config = RawConfigParser()
        config.add_section('default')
        config.set('default', 'osdir', "os")
        config.set('default', 'sourcedir', "source")
        config.set('default', 'debugdir', "debug")
        config.set('default', 'isodir', "iso")
        config.set('default', 'iso_basename', self.cfg.iso_basename or self.cfg.product_name)
#        config.set('default', 'relnotefilere', relnotefilere)
#        config.set('default', 'relnotedirre', relnotedirre)
#        config.set('default', 'relnotepkgs', relnotepkgs)
        config.set('default', 'flavor', self.cfg.revisor_model)
        config.set('default', 'version', self.cfg.version)
        config.set('default', 'product_name', self.cfg.product_name)
        config.set('default', 'comps', self.cfg.comps)
        config.set('default', 'product_path', self.cfg.product_path or self.cfg.product_name)
        config.set('default', 'destdir', os.path.join(self.cfg.working_directory,"revisor-pungi"))
        config.set('default', 'arch', self.cfg.arch)

        if not os.path.exists(config.get('default','destdir')):
            try:
                os.makedirs(config.get('default', 'destdir'))
            except OSError, e:
                self.log.error(_("Error: Cannot create destination dir %s") % config.get('default', 'destdir'))
        else:
            # Empty it
            try:
                shutil.rmtree(config.get('default', 'destdir'))
            except OSError, e:
                self.log.error(_("Error: Cannot remove destination dir %s") % config.get('default', 'destdir'))
            # Recreate it
            try:
                os.makedirs(config.get('default', 'destdir'))
            except OSError, e:
                self.log.error(_("Error: Cannot recreate destination dir %s") % config.get('default', 'destdir'))

        # Actually do work.
        if not config.get('default', 'arch') == 'source':

            # init some
            mypungi = pungi.RevisorPungi(config, self.cfg, self.log)

            try:
                os.makedirs(os.path.join(config.get('default', 'destdir'),
                                         config.get('default', 'version'),
                                         config.get('default', 'flavor'),
                                         config.get('default', 'arch')))
            except:
                pass

            pkgdir = os.path.join(  config.get('default', 'destdir'),
                                    config.get('default', 'version'),
                                    config.get('default', 'flavor'),
                                    config.get('default', 'arch'),
                                    config.get('default', 'osdir'),
                                    config.get('default', 'product_path'))

            if not os.path.exists(pkgdir):
                self.log.debug(_("Creating pkgdir: %s" % (pkgdir)))
                os.makedirs(pkgdir)
            else:
                shutil.rmtree(pkgdir, ignore_errors = True)
                os.makedirs(pkgdir)

            # Now for each txmbr in

            # Link the localPkg() result into the build tree
            polist = map(lambda x: x.po, filter(lambda txmbr:
                                                txmbr.ts_state in ("i", "u"),
                                                self.cfg.yumobj.tsInfo.getMembers()))

            pbar = self.progress_bar(_("Linking in packages"))
            pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
            mypungi.setCallback(pungicallback)

            failedHardLink = False
            total = float(len(polist))
            i = 0
            for po in polist:
                try:
                    os.link(po.localPkg(), pkgdir + "/" + os.path.basename(po.localPkg()))
                except OSError, e:
                    if e.errno is 18:
                        self.log.debug(_("Package hard link failed, trying symlink for: %s") % (os.path.basename(po.localPkg())))
                        try:
                            os.symlink(po.localPkg(), pkgdir + "/" + os.path.basename(po.localPkg()))
                            self.log.debug(_("Package symlink succeeded"))
                        except OSError, e:
                            self.log.debug(_("Package link failed: %s: %s") % (e, os.path.basename(po.localPkg())))
                    else:
                        self.log.debug(_("Package link failed: %s: %s") % (e, os.path.basename(po.localPkg())))
                i += 1
                pbar.set_fraction(i/total)
            pbar.destroy()

            pbar = self.progress_bar(_("Creating Repository Information"), callback = True)
            pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
            mypungi.setCallback(pungicallback)
            mypungi.doCreateRepo()
            pbar.destroy()

            pbar = self.progress_bar(_("Building Installation Images"))
            pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
            mypungi.setCallback(pungicallback)
            mypungi.doBuildinstall()
            pbar.destroy()

            pbar = self.progress_bar(_("Linking in release notes"))
            pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
            mypungi.setCallback(pungicallback)
            mypungi.doGetRelnotes()
            pbar.destroy()

            if not self.cfg.copy_dir == "":
                if os.access(self.cfg.copy_dir, os.R_OK):
                    mypungi.doCopyDir(copy_dir = self.cfg.copy_dir)
                else:
                    self.log.warning(_("copy_dir '%s' not accessible") % self.cfg.copy_dir)

            size_cd = 685 * 1024 * 1024
            size_dvd = 4500 * 1024 * 1024
            size_dvd_dl = 2* 4500 * 1024 * 1024
            calc_size_str = subprocess.Popen(['/usr/bin/du',
                                            '-sb',
                                            '%s/os' % os.path.join(config.get('default', 'destdir'),
                                                                    config.get('default', 'version'),
                                                                    config.get('default', 'flavor'),
                                                                    config.get('default', 'arch'))],
                                stdout=subprocess.PIPE, stderr=subprocess.STDOUT).communicate()[0]
            self.log.debug(_("Str output: %s") % calc_size_str)
            calc_size_str = calc_size_str.split()[0]
            self.log.debug(_("Str output(split): %s") % calc_size_str)
            size_unified = int(calc_size_str)

            # So, what would the number of discs be?
            num_cd = int(math.ceil(size_unified / size_cd)) + 1
            num_dvd = int(math.ceil(size_unified / size_dvd)) + 1
            num_dvd_dl = int(math.ceil(size_unified / size_dvd_dl)) + 1
            num_unified = 1.0
            self.cfg.num_cd = num_cd
            self.cfg.num_dvd = num_dvd
            self.cfg.num_dvd_dl = num_dvd_dl

            self.log.debug(_("Size of CD: %s") % str(size_cd))
            self.log.debug(_("Number of CDs: %s") % str(num_cd))
            self.log.debug(_("Size of DVD: %s") % str(size_dvd))
            self.log.debug(_("Number of DVDs: %s") % str(num_dvd))
            self.log.debug(_("Size of DVD DL: %s") % str(size_dvd_dl))
            self.log.debug(_("Number of DVD DLs: %s") % str(num_dvd_dl))

            self.cfg.do_packageorder = False

            # Nice, set config.get('default','discs') or this baby poos
            config.set('default','discs',2)
            if self.cfg.media_installation_dvd_duallayer and size_unified > size_dvd_dl:
                self.cfg.do_packageorder = True
            if self.cfg.media_installation_dvd and size_unified > size_dvd:
                self.cfg.do_packageorder = True
            if self.cfg.media_installation_cd and size_unified > size_cd:
                self.cfg.do_packageorder = True

            # Now, we know everything!
            # And we lived happily ever after

            if self.cfg.do_packageorder:
                pbar = self.progress_bar(_("Running Package Order"))
                mypungi.doPackageorder()
                pbar.destroy()

            if self.cfg.kickstart_include:
                f = open(os.path.join(mypungi.topdir,"ks.cfg"), "w")
                f.write(self.cfg.kshandler.__str__())
                f.close()
                if self.cfg.kickstart_default:
                    f = open(os.path.join(mypungi.topdir,"isolinux","isolinux.cfg"),"rw+")
                    buf = f.read()
                    buf = buf.replace("default linux","default ks")
                    f.seek(0)
                    f.write(buf)
                    f.close()

# Prep for installation CD
            if self.cfg.media_installation_cd or (self.cfg.num_cd == self.cfg.num_dvd and self.cfg.media_installation_dvd):
                config.set('default','discs',str(num_cd))
                os.symlink(mypungi.topdir,"%s-cd" % mypungi.topdir)

                for i in range(2, num_cd+1):
                    try:
                        os.makedirs("%s-%s-disc%d" % (mypungi.topdir,"cd",i))
                    except:
                        # Should probably raise some error and exit
                        pass

                if num_cd > 1:
                    # Split tree
                    pbar = self.progress_bar(_("Splitting Build Tree (CD)"))
                    pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
                    mypungi.setCallback(pungicallback)
                    mypungi.doSplittree(discdir="cd")
                    pbar.destroy()

                    # Split repo
                    pbar = self.progress_bar(_("Splitting Repository (CD)"))
                    pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
                    mypungi.setCallback(pungicallback)
                    mypungi.doCreateSplitrepo(discdir="cd")
                    pbar.destroy()

                pbar = self.progress_bar(_("Creating CD ISO Images"))
                pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
                mypungi.setCallback(pungicallback)
                mypungi.doCreateIsos(discdir="cd")
                pbar.destroy()

# Prep for installation DVD
            if self.cfg.media_installation_dvd:
                # Workaround, symlink
                config.set('default','discs',str(num_dvd))
                os.symlink(mypungi.topdir,"%s-dvd" % mypungi.topdir)
                for i in range(2, num_dvd+1):
                    try:
                        os.makedirs("%s-%s-disc%d" % (mypungi.topdir,"dvd",i))
                    except:
                        # Should probably raise some error and exit
                        pass

                if num_dvd > 1:
                    # Split tree
                    pbar = self.progress_bar(_("Splitting Build Tree (DVD)"))
                    pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
                    mypungi.setCallback(pungicallback)
                    mypungi.doSplittree(media_size=4500, discdir="dvd")
                    pbar.destroy()

                    # Split repo
                    pbar = self.progress_bar(_("Splitting Repository Information (DVD)"))
                    pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
                    mypungi.setCallback(pungicallback)
                    mypungi.doCreateSplitrepo(discdir="dvd")
                    pbar.destroy()

                pbar = self.progress_bar(_("Creating DVD ISO Images"))
                pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
                mypungi.setCallback(pungicallback)
                mypungi.doCreateIsos(discdir="dvd")
                pbar.destroy()

# Prep for installation DVD Dual Layer
            if self.cfg.media_installation_dvd_duallayer:
                # Workaround, symlink
                config.set('default','discs',str(num_dvd))
                os.symlink(mypungi.topdir,"%s-dvd-dl" % mypungi.topdir)
                for i in range(2, num_dvd+1):
                    try:
                        os.makedirs("%s-%s-disc%d" % (mypungi.topdir,"dvd-dl",i))
                    except:
                        # Should probably raise some error and exit
                        pass

                if num_dvd > 1:
                    # Split tree
                    pbar = self.progress_bar(_("Splitting Build Tree (DVD Dual Layer)"))
                    pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
                    mypungi.setCallback(pungicallback)
                    mypungi.doSplittree(media_size=9000, discdir="dvd-dl")
                    pbar.destroy()

                    # Split repo
                    pbar = self.progress_bar(_("Splitting Repository Information (DVD Dual Layer)"))
                    pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
                    mypungi.setCallback(pungicallback)
                    mypungi.doCreateSplitrepo(discdir="dvd-dl")
                    pbar.destroy()

                pbar = self.progress_bar(_("Creating DVD Dual Layer ISO Images"))
                pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
                mypungi.setCallback(pungicallback)
                mypungi.doCreateIsos(discdir="dvd-dl")
                pbar.destroy()

            if self.cfg.media_installation_unified:
                # Workaround, symlink
                config.set('default','discs',1)
                pbar = self.progress_bar(_("Creating Unified ISO Image"))
                pungicallback = revisor.progress.PungiCallback(pbar, mypungi)
                mypungi.setCallback(pungicallback)
                mypungi.doCreateIsos()
                pbar.destroy()

            if len(self.cfg.delta_old_image) > 0 and self.cfg.hasDelta:
#            if len(self.cfg.delta_old_image) > 0 and self.cfg.plugins['delta']:
                from revisor.delta import RevisorDelta
                delta = RevisorDelta()
                # Generate the delta image
                pbar = self.progress_bar(_("Generating Delta ISO Images"))
                isoname = '%s-%s-%s.iso' % (config.get('default', 'iso_basename'), config.get('default', 'version'), config.get('default', 'arch'))
                for iso in self.cfg.built_iso_images:
                    self.log.debug(_("Building delta for %s" % (iso)))
                    delta.buildiso(["/usr/bin/makedeltaiso", self.cfg.delta_old_image, os.path.join(self.cfg.destination_directory,"iso","%s.delta" % isoname)])
                pbar.destroy()

            if len(self.cfg.cobbler_add_distro) > 0 and self.cfg.hasCobbler:
#            if len(self.cfg.cobbler_add_distro) > 0 and self.cfg.plugins['cobbler']:
                from revisor.cobbler import RevisorCobbler
                pbar = self.progress_bar(_("Adding Distro to Cobbler"))
                cobbler = RevisorCobbler()
                self.log.debug(_(("Using Cobbler to add distro %s") % (self.cfg.cobbler_add_distro)))
                # FIXME: Find out what we need to pass/assemble/collect from user
                #cobbler.newDistro(options)
                pbar.destroy()

            if len(self.cfg.cobbler_add_profile) > 0 and self.cfg.hasCobbler:
#            if len(self.cfg.cobbler_add_profile) > 0 and self.cfg.plugins['cobbler']:
                from revisor.cobbler import RevisorCobbler
                pbar = self.progress_bar(_("Adding Profile to Cobbler"))
                cobbler = RevisorCobbler()
                self.log.debug(_(("Using Cobbler to add profile %s") % (self.cfg.cobbler_add_profile)))
                # FIXME: Add code that will hand a render(ed|able) kickstart
                #cobbler.newProfile(kickstart)
                pbar.destroy()

    def buildLiveMedia(self):
        livemedia_targetsize = self.cfg.payload_livemedia + 1536
        self.log.debug(_("Setting Live Media Ext3 Filesystem Size to %s MB") % str(livemedia_targetsize))
        # Get our object from pilgrim
        liveImage = lm_optical.LMOptical(self)

        # Symlink our yum.installroot to
        liveImage.fs_label = "livecd-" + time.strftime("%Y%m%d")
        liveImage.bindmounts = []
        liveImage.ksparser = self.cfg.ksparser
        liveImage.ksparser.handler = self.cfg.kshandler

        pbar = self.progress_bar(_("Setting up Ext3 Filesystem"))

        if not liveImage.setup(livemedia_targetsize, build_dir="/var/tmp/revisor-livecd", yum_installroot=self.cfg.yumobj.conf.installroot, yum_cachedir=self.cfg.yumobj.conf.cachedir):
            self.log.error(_("Cannot setup installation target. Aborting."))
            sys.exit(1)
        else:
#            print "I have setup the liveImage. Go for it!"
            pass

        pbar = self.progress_bar(_("Installing Packages"), callback = True)
        liveImage.installPackages(pbar = pbar)

        self.cfg.yumobj.closeRpmDB()

        if not self.cfg.copy_dir == "":
            if os.access(self.cfg.copy_dir, os.R_OK):
                for dirpath, dirlist, filelist in os.walk(self.cfg.copy_dir):
                    for file in filelist:
                        try:
                            self.log.debug(_("Copying %s to %s") % (os.path.join(self.cfg.copy_dir,file),"%s/install_root/" % liveImage.build_dir))
                            shutil.copy(os.path.join(self.cfg.copy_dir,file),"%s/install_root/" % liveImage.build_dir)
                        except OSError, e:
                            self.log.warning(_("Copy of %s failed: %s") % (os.path.join(self.cfg.copy_dir,file), e))
                    for directory in dirlist:
                        try:
                            self.log.debug(_("Copying %s to %s") % (os.path.join(self.cfg.copy_dir,directory),"%s/install_root/" % liveImage.build_dir))
                            shutil.copytree(os.path.join(self.cfg.copy_dir,directory),"%s/install_root/" % liveImage.build_dir)
                        except OSError, e:
                            self.log.warning(_("Copy of %s failed: %s") % (os.path.join(self.cfg.copy_dir,directory), e))
            else:
                self.log.warning(_("copy_dir '%s' not accessible") % self.cfg.copy_dir)

        pbar = self.progress_bar(_("Configure System"))
        liveImage.configureSystem()

        pbar = self.progress_bar(_("Configuring Network"))
        liveImage.configureNetwork()

        pbar = self.progress_bar(_("Creating RAM Filesystem"))
        liveImage.createInitramfs()

        pbar = self.progress_bar(_("Relabel System"))
        liveImage.relabelSystem()

        pbar = self.progress_bar(_("Configure BootLoader"))

        if os.access("%s/install_root/usr/lib/syslinux/%s" % (liveImage.build_dir, "menu.c32"), os.R_OK):
            menu = "menu.c32"
        if os.access("%s/install_root/usr/lib/syslinux/%s" % (liveImage.build_dir, "vesamenu.c32"), os.R_OK):
            menu = "vesamenu.c32"

        liveImage.configureBootloader(menu = menu)

        if self.cfg.advanced_configuration:
            pbar = self.progress_bar(_("Launching Shell"))
            if self.cfg.revisorUseGUI:
                pbar.set_markup(_("Launching a Shell. Adjust the configuration from within the Chroot and logout to continue building the Live Media."))
                subprocess.call(["/usr/bin/gnome-terminal", "--window", "-x", "chroot", "%s/install_root" % (liveImage.build_dir)])
            else:
                liveImage.launchShell()

        pbar = self.progress_bar(_("Unmounting Filesystems"))
        liveImage.unmount()

        if not self.cfg.lm_skip_fs_compression:
            pbar = self.progress_bar(_("Creating Squash Filesystem"))
            pungipbar = revisor.progress.PungiCallback(pbar)
            pbar.set_markup(_("This is going to take a while and might cause the GUI to become unresponsive... it will recover after this task completes."))
            liveImage.createSquashFS(pungipbar)

        pbar = self.progress_bar(_("Creating ISO Live Image"))
        pungipbar = revisor.progress.PungiCallback(pbar)
        liveImage.createIso(pungipbar)

        pbar = self.progress_bar(_("Cleaning up build environment"))
        liveImage.teardown()

