#
# Copyright 2007 Fedora Unity
#
# Jonathan Steffan <jon a fedoraunity.org>
# Jeroen van Meeuwen <kanarip a fedoraunity.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 only
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import os
import sys
import gtk
import gtk.glade
import gobject
import gtk.gdk as gdk

from ConfigParser import SafeConfigParser

# Import constants
from revisor.constants import *

# Translation
from rhpl.translate import _, N_, getDefaultLangs
import rhpl.translate as translate

class RevisorConfiguration:
    def __init__(self, gui):
        self.gui = gui
        self.base = gui.base
        self.log = gui.log
        self.cfg = gui.cfg
        self.frame_xml = gui.frame_xml

        self.select_model = self.frame_xml.get_widget("revisor_config_select_model")

        gui.add_buttons()

        self.connect_button_signals()

        self.restore_options()

        self.header_image = gui.base_screen_xml.get_widget("header_image")
        self.header_image.set_from_file(PIXMAPS_FILES + "header_yum.png")

    # widgets:
    # repo_treeview - repository selection box (treeview)
    # revisor_config_select_model - model selection
    def connect_button_signals(self):
        sigs = { "on_button_back_clicked": self.button_back_clicked,
                 "on_button_forward_clicked": self.button_forward_clicked,
                 "on_button_cancel_clicked": self.gui.button_cancel_clicked }

        self.gui.base_buttons_xml.signal_autoconnect(sigs)

        sigs = { "on_button_revisor_config_apply_model_clicked": self.revisor_config_apply_model_clicked,
                 "on_button_revisor_config_refresh_clicked": self.revisor_config_refresh_clicked }

        self.frame_xml.signal_autoconnect(sigs)

    def revisor_config_refresh_clicked(self, button):

        widget_rc = self.frame_xml.get_widget("revisor_config")
        widget_cbox_parent = self.frame_xml.get_widget("revisor_config_table")

        models = self.revisor_config_sections(widget_rc.get_text())

        if models:
            self.select_model.hide()
            self.select_model.destroy()
            self.select_model = gtk.combo_box_new_text()

            i=0
            for model in models:
                self.select_model.append_text(model)
                if model == self.cfg.revisor_model:
                    self.select_model.set_active(i)
                i += 1

            if not self.select_model.get_active() >= 0:
                self.select_model.set_active(0)

            widget_cbox_parent.attach(self.select_model,1,2,1,2,yoptions=gtk.EXPAND,xpadding=5,ypadding=5)
            self.select_model.show()
            self.cfg.set_revisor_config(widget_rc.get_text())

            model_selected = self.select_model.get_active_text()
            self.cfg.model_options_reload_from_config()

            self.repo_treeview = self.frame_xml.get_widget("repo_treeview")
            if not model_selected == "" and not model_selected == None:
                if self.revisor_config_has_option(self.cfg.revisor_config,model_selected,"main"):
                    self.cfg.set_revisor_model(model_selected)
                    self.cfg.set_main_conf(self.revisor_parser.get(model_selected,"main"))
                    try:
                        self.repo_treeview.remove_column(self.repo_column_select)
                        self.repo_treeview.remove_column(self.repo_column_name)
                        self.repo_treeview.remove_column(self.repo_column_desc)
                    except:
                        pass

                    self.load_repositories()
        else:
            self.log.error(_("%s is not a valid Revisor configuration file") % widget_rc.get_text())

    def revisor_config_apply_model_clicked(self, button):
        model_selected = self.select_model.get_active_text()

        self.repo_treeview = self.frame_xml.get_widget("repo_treeview")
        if not model_selected == "" and not model_selected == None:
            if self.revisor_config_has_option(self.cfg.revisor_config,model_selected,"main"):
                self.cfg.set_revisor_model(model_selected)
                self.cfg.set_main_conf(self.revisor_parser.get(model_selected,"main"))
                try:
                    self.repo_treeview.remove_column(self.repo_column_select)
                    self.repo_treeview.remove_column(self.repo_column_name)
                    self.repo_treeview.remove_column(self.repo_column_desc)
                except:
                    pass

                self.cfg.model_options_reload_from_config()
                self.load_repositories()
            else:
                self.log.error(_("The configured model does not have the mandatory 'main' configuration directive."))
        else:
            self.log.error(_("Invalid model. Please choose a valid model."))

    def button_back_clicked(self, button):
        self.store_options()
        self.gui.displaySelectMedia()

    def button_forward_clicked(self, button):
        if not self.check_options():
            pass
        else:
            self.store_options()
            try: os.makedirs(self.cfg.yum_cache_dir)
            except: self.log.error("Unable to create yum cache directory %s" % self.cfg.yum_cache_dir)
            self.gui.displayLoadKickstart()

    def load_repositories(self):
        repo_parser = SafeConfigParser()
        self.repo_parser = repo_parser
        # FIXME (try except os.access)
        repo_parser.read(self.cfg.main_conf)

        # Weee, this is where we know what the install_root is, and what the cachedir is
        if repo_parser.has_option("main","installroot") and repo_parser.has_option("main","cachedir"):
            self.cfg.yum_cache_dir = os.path.join("%s/%s" % (repo_parser.get("main","installroot"),repo_parser.get("main","cachedir")))
        else:
            self.cfg.yum_cache_dir = "/var/tmp/revisor/yumcache"

        #self.log.debug("Setting yum log file to %s" % self.cfg.yum_cache_dir)

        # widget - repo_treeview
        self.repoStore = gtk.ListStore(gobject.TYPE_BOOLEAN,
                                      gobject.TYPE_STRING,
                                      gobject.TYPE_STRING)

        self.repo_column_select = gtk.TreeViewColumn(None, None)
        cbr = gtk.CellRendererToggle()
        self.repo_column_select.pack_start(cbr, False)
        self.repo_column_select.add_attribute(cbr, 'active', 0)
        cbr.connect("toggled", self._repoToggled)
        self.repo_treeview.append_column(self.repo_column_select)

        self.repo_column_name = gtk.TreeViewColumn(None, None)
        txtr = gtk.CellRendererText()
        self.repo_column_name.pack_start(txtr, False)
        self.repo_column_name.add_attribute(txtr, 'markup', 1)
        self.repo_treeview.append_column(self.repo_column_name)

        self.repo_column_desc = gtk.TreeViewColumn(None, None)
        txtr = gtk.CellRendererText()
        self.repo_column_desc.pack_start(txtr, False)
        self.repo_column_desc.add_attribute(txtr, 'markup', 2)
        self.repo_treeview.append_column(self.repo_column_desc)

        self.repoStore.set_sort_column_id(1, gtk.SORT_ASCENDING)

        for repo in repo_parser._sections:
            if not repo == "main":
                # Remove all disabled repos
                if not self.cfg.repos_enablesource and repo.find("source") >= 0:
                    continue
                if not self.cfg.repos_enabledebuginfo and repo.find("debuginfo") >= 0:
                    continue
                if not self.cfg.repos_enabletesting and repo.find("testing") >= 0:
                    continue
                if not self.cfg.repos_enabledevelopment and repo.find("development") >= 0:
                    continue

                if self.repo_parser.get(repo,"enabled") == "1":
                    repo_enabled = True
                else:
                    repo_enabled = False

                if repo in self.cfg.repos and self.cfg.repos[repo]:
                    repo_enabled = True
                elif repo in self.cfg.repos and not self.cfg.repos[repo]:
                    repo_enabled = False

                self.repoStore.append([repo_enabled,repo,"<i>" + self.repo_parser.get(repo,"name") + "</i>"])
                self.cfg.repos[repo] = repo_enabled

        self.repo_treeview.set_model(self.repoStore)

    def _repoToggled(self, widget, path):
        cfg = self.gui.cfg
        i = self.repoStore.get_iter(path)
        repo_toggled = self.repoStore.get_value(i, 1)
        if self.repoStore.get_value(i, 0):
            self.repoStore.set_value(i, 0, False)
            self.cfg.repos[repo_toggled] = False
            status = "disabled"
        else:
            self.repoStore.set_value(i, 0, True)
            self.cfg.repos[repo_toggled] = True
            status = "enabled"

        self.log.debug("%s is now %s" % (repo_toggled, status))

    def check_options(self):
        widget_rc = self.frame_xml.get_widget("revisor_config")
        config = widget_rc.get_text()

        if not os.access(config, os.R_OK):
            self.log.error(_("File %s is not accessible.") % config)
            return False

        file_main_ok = False

        if self.revisor_config_has_option(config,self.select_model.get_active_text(),"main"):
            if not os.access(self.revisor_parser.get(self.select_model.get_active_text(),"main"), os.R_OK):
                self.log.error(_("The 'main' option configuration directive has a non-accessible file: ") + self.revisor_parser.get(self.select_model.get_active_text(),"main"))
                return False
            else:
                self.log.debug(_("Configuration file's 'main' directive OK"))
                # Set the configuration file,
                self.cfg.revisor_config = config
                self.cfg.revisor_model = self.select_model.get_active_text()
                self.cfg.main_conf = self.revisor_parser.get(self.select_model.get_active_text(),"main")

                # And reload the options
                self.cfg.model_options_reload_from_config()

                # Let's check for the existance of the directories we are going to work with:
                if not self.cfg.check_working_directory():
                    return False

                # Let's check for the existance of the directories in which our products go:
                self.cfg.check_destination_directory()

                one_repo_set = False
                for repo in self.cfg.repos:
                    if self.cfg.repos[repo]:
                        one_repo_set = True

                base_repo_set = False
                if "core" in self.cfg.repos and self.cfg.repos["core"]:
                    base_repo_set = True
                elif "base" in self.cfg.repos and self.cfg.repos["base"]:
                    base_repo_set = True
                elif "fedora" in self.cfg.repos and self.cfg.repos["fedora"]:
                    base_repo_set = True
                elif "development" in self.cfg.repos and self.cfg.repos["development"]:
                    base_repo_set = True
                elif "fedora-local" in self.cfg.repos and self.cfg.repos["fedora-local"]:
                    base_repo_set = True

                if not one_repo_set or not base_repo_set:
                    self.log.warning(_("You have not selected any of the basic repositories. Please make sure that one of 'fedora', 'core', 'base' or 'development', or an equivalent repository has been configured"))

                return True
        else:
            self.log.error(_("Configuration file '%s', section '%s' does not have the mandatory 'main' option") % (config, self.select_model.get_active_text()))
            return False

    def restore_options(self):
        widget_rc = self.frame_xml.get_widget("revisor_config")

        widget_rc.set_text(self.cfg.revisor_config)

        self.load_models()

        self.load_selected_model()

    def load_selected_model(self):
        widget_cbox_parent = self.frame_xml.get_widget("revisor_config_table")
        self.repo_treeview = self.frame_xml.get_widget("repo_treeview")

        if not self.cfg.revisor_model == "" and not self.cfg.revisor_model == None:
            if self.revisor_config_has_option(self.cfg.revisor_config,self.cfg.revisor_model,"main"):
                self.cfg.main_conf = self.revisor_parser.get(self.cfg.revisor_model,"main")
                try:
                    self.repo_treeview.remove_column(self.repo_column_select)
                    self.repo_treeview.remove_column(self.repo_column_name)
                    self.repo_treeview.remove_column(self.repo_column_desc)
                except:
                    pass

                self.load_repositories()

        widget_cbox_parent.attach(self.select_model,1,2,1,2,yoptions=gtk.EXPAND,xpadding=5,ypadding=5)
        self.select_model.show()

    def load_models(self):
        widget_cbox_parent = self.frame_xml.get_widget("revisor_config_table")
        self.select_model.hide()
        self.select_model.destroy()

        models = self.revisor_config_sections(self.cfg.revisor_config)
        self.select_model = gtk.combo_box_new_text()

        i=0
        for model in models:
            self.select_model.append_text(model)
            if model == self.cfg.revisor_model:
                self.select_model.set_active(i)
            i += 1

        if not self.select_model.get_active() >= 0:
            self.select_model.set_active(0)

        self.cfg.revisor_model = self.select_model.get_active_text()

    def store_options(self):
        widget_rc = self.frame_xml.get_widget("revisor_config")

        self.cfg.revisor_config = widget_rc.get_text()
        self.cfg.revisor_model = self.select_model.get_active_text()

    def revisor_config_sections(self, config):
        parser = SafeConfigParser()
        if os.access(config, os.R_OK):
            try:
                parser.read(config)
                self.gui.log.info(_("Reading configuration file %s") % config)
                parser_sections = parser._sections.keys()
                # Omit the [revisor] section
                sections = []
                for section in parser_sections:
                    if not section == "revisor":
                        sections.append(section)

                sections.sort()
                return sections

            except:
                return False
        else:
            self.log.error(_("Could not read configuration file %s") % config)

    def revisor_config_has_option(self, config, section, option):
        self.revisor_parser = SafeConfigParser()
        # FIXME (try except os.access)
        self.revisor_parser.read(config)
        if self.revisor_parser.has_option(section,option):
            return True
        else:
            return False

