/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

/* Image Manager class */

#include "imgmanager.h"
#include "cache.h"
#include "kitaconfig.h"
#include "downloadmanager.h"

#include <kdeversion.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kconfig.h>

#include <qregexp.h>
#include <qfile.h>
#include <qdatastream.h>
#include <qmessagebox.h>
#include <qapplication.h>
#include <qimage.h>

using namespace Kita;

ImgManager* ImgManager::instance = NULL;
QMutex ImgManager::m_mutex;

/*-----------------------------------------------------------*/


ImgManager::ImgManager( QWidget* mainwidget )
{
    m_imgDatDict.clear();
    m_mainwidget = mainwidget;
    m_urlList.clear();
}


ImgManager::~ImgManager()
{}


/* call this first */ /* public */ /* static */
void ImgManager::setup( QWidget* mainwidget )
{
    instance = new ImgManager( mainwidget );
}


/* public */ /* static */
void ImgManager::deleteInstance()
{
    if ( instance ) delete instance;
}


/* public */ /* static */
ImgManager* ImgManager::getInstance()
{
    return instance;
}


/* public */ /* static */
bool ImgManager::isImgFile( const KURL& url, bool use_mimetype )
{
    return instance->isImgFilePrivate( url, use_mimetype );
}


/* public */ /* static */
bool ImgManager::isAnimationGIF( const KURL& url )
{
    QRegExp reg_ext( ".*\\.gif$" );
    if ( reg_ext.search( url.fileName().lower() ) != -1 ) return TRUE;

    return FALSE;
}


/* public */ /* static */
bool ImgManager::isBMP( const KURL& url )
{
    QRegExp reg_ext( ".*\\.bmp$" );
    if ( reg_ext.search( url.fileName().lower() ) != -1 ) return TRUE;

    return FALSE;
}


/* public */ /* static */
bool ImgManager::load( const KURL& url, const KURL& datURL )
{
    QMutexLocker locker( &m_mutex );

    return instance->loadPrivate( url, datURL );
}


/* public */ /* static */
void ImgManager::stop( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    DownloadManager::stopLoading( url );
}


/* public */ /* static */
QString ImgManager::getPath( const KURL& url )
{
    return Cache::getImgPath( url );
}


/* public */ /* static */
bool ImgManager::isLoadingNow( const KURL& url )
{

    return DownloadManager::isLoadingNow( url );
}


/* public */ /* static */
bool ImgManager::deleteCache( const KURL& url, QWidget* parent )
{
    QMutexLocker locker( &m_mutex );

    return instance->deleteCachePrivate( url, parent );
}


/* public */ /* static */
bool ImgManager::copyCache( const KURL& url, QWidget* parent )
{
    QMutexLocker locker( &m_mutex );

    return instance->copyCachePrivate( url, parent );
}


/* public */ /* static */
bool ImgManager::mosaic( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return TRUE;

    return imgdat->mosaic;
}


/* public */ /* static */
void ImgManager::setMosaic( const KURL& url, bool status )
{
    instance->setMosaicPrivate( url, status );
}


/* return -1 if IMGDAT doesn't exist. */  /* public */ /* static */
int ImgManager::code( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return -1;

    return imgdat->code;
}


/* public */ /* static */
unsigned int ImgManager::size( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return 0;

    return imgdat->size;
}


/* public */ /* static */
int ImgManager::width( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return 0;
    if ( imgdat->width == 0 ) instance->getSize( url );

    return imgdat->width;
}


/* public */ /* static */
int ImgManager::height( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return 0;
    if ( imgdat->height == 0 ) instance->getSize( url );

    return imgdat->height;
}


/* get icon pixmap */ /* public */ /* static */
QPixmap ImgManager::icon( const KURL& url )
{
    const int iconsize = 32;

    /*---------------------------*/

    QPixmap pixmap;
    QString path = Cache::getImgPath( url );
    QImage img = QImage( path );
    if ( img.isNull() ) return QPixmap();

    pixmap.convertFromImage( img.scale( iconsize, iconsize, QImage::ScaleMin ) );
    return pixmap;
}


/* public */ /* static */
KURL ImgManager::datURL( const KURL& url )
{
    instance->cacheExists( url ); /* to create IMGDAT if cache exists */

    IMGDAT* imgdat = instance->getImgDat( url );
    if ( !imgdat ) return KURL();

    return imgdat->datURL;
}


/*---------------------------------*/
/* internal functions              */


/* private */
bool ImgManager::isImgFilePrivate( const KURL& url, bool use_mimetype )
{
    /* don't use mimetype      */
    if ( !use_mimetype ) {

        QStringList extlist; // = KitaConfig::imgExtList();
        extlist = "jpg";
        extlist += "jpeg";
        extlist += "gif";
        extlist += "png";
        extlist += "bmp";

        for ( QStringList::iterator it = extlist.begin(); it != extlist.end(); ++it ) {
            QRegExp reg_ext = ".*\\." + ( *it ) + "$";
            if ( reg_ext.search( url.fileName().lower() ) != -1 ) return TRUE;

        }

        return FALSE;
    }

    /*-----------------------------------*/

    /* use mimetype ( very slow! )  */
    if ( cacheExists( url ) ) return TRUE;

#if KDE_IS_VERSION( 3, 2, 0 )
    QString mimetype = KIO::NetAccess::mimetype( url, m_mainwidget );
#else
    QString mimetype = KIO::NetAccess::mimetype( url );
#endif
    if ( mimetype.left( 6 ) == "image/" ) return TRUE;

    return FALSE;
}


/* create new loader and start loading the image       */
/* when done, ImgManager emits finishImgLoad SIGNAL.   */   /* private */
bool ImgManager::loadPrivate( const KURL& url, const KURL& datURL )
{
    const unsigned int maxload = 5; // = KitaConfig::maxLoader();

    /*------------------------------*/

    if ( DownloadManager::isLoadingNow( url ) ) return FALSE;

    /* cache exists? */
    if ( cacheExists( url ) ) {

        /* create EmitFinishEvent to emit finishImgLoad SIGNAL later. */
        /* See also customEvent().                                    */
        EmitFinishEvent * e = new EmitFinishEvent( url );
        QApplication::postEvent( this, e );  // Qt will delete it when done

        return TRUE;
    }

    if ( m_urlList.count() > maxload ) return FALSE;

    QString path = Cache::getImgPath( url );
    deleteImgDat( url );

    /* create new loader, and start loading the file */
    /* slotResult is called when done.               */
    LoaderOption option;
    option.usrurl = datURL;
    FileLoader* loader = DownloadManager::download( url, path, option );
    if ( loader == NULL ) return FALSE;

    connect( loader, SIGNAL( data( const Kita::LoaderData&, const QByteArray& ) ),
             SLOT( slotData( const Kita::LoaderData&, const QByteArray& ) ) );

    connect( loader, SIGNAL( result( const Kita::LoaderData& ) ),
             SLOT( slotResult( const Kita::LoaderData& ) ) );

    m_urlList += url.prettyURL();

    return TRUE;
}


/* delete cache */
/* when done, cacheDeleted SIGNAL emitted */ /* private */
bool ImgManager::deleteCachePrivate( const KURL& url, QWidget* parent )
{
    if ( !cacheExists( url ) ) return FALSE;

    if ( QMessageBox::warning( parent,
                               "Kita",
                               i18n( "Do you want to delete the image ?" ),
                               QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
            == QMessageBox::Ok ) {

        bool ret;;

        QString path = Cache::getImgPath( url );
        QString idxpath = Cache::getImgIdxPath( url );

#if KDE_IS_VERSION( 3, 2, 0 )
        KIO::NetAccess::del( idxpath, m_mainwidget );
        ret = KIO::NetAccess::del( path, m_mainwidget );
#else
        KIO::NetAccess::del( idxpath );
        ret = KIO::NetAccess::del( path );
#endif

        if ( ret ) {
            deleteImgDat( url );
            emit cacheDeleted( url );
        }

        return ret;
    }

    return FALSE;
}


/* copy cache */ /* public */
bool ImgManager::copyCachePrivate( const KURL& url, QWidget* parent )
{
    if ( !cacheExists( url ) ) return FALSE;

    /* make filter */
    QString filter;
    QString file = url.fileName().lower();
    int i = file.findRev( '.' );
    if ( i == -1 ) filter = "*|All files";
    else {
        QString ext = file.mid( i );
        filter = "*" + ext + "|*" + ext;
    }

    /* get filename */
    QString savefile = KFileDialog::getSaveFileName( url.fileName(), filter, parent );
    if ( savefile.isEmpty() ) return FALSE;

    /* file exists */
    if (
#if KDE_IS_VERSION( 3, 2, 0 )
        KIO::NetAccess::exists( savefile, TRUE, m_mainwidget )
#else
        KIO::NetAccess::exists( savefile )
#endif
    ) {

        if ( QMessageBox::warning( parent,
                                   "Kita",
                                   i18n( "Do you want to replace the file?" ),
                                   QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
                != QMessageBox::Ok ) return FALSE;

#if KDE_IS_VERSION( 3, 2, 0 )
        KIO::NetAccess::del( savefile, m_mainwidget );
#else
        KIO::NetAccess::del( savefile );
#endif

    }

    /* copy */
    QString src = Cache::getImgPath( url );

#if KDE_IS_VERSION( 3, 2, 0 )
    KIO::NetAccess::copy( src, savefile, m_mainwidget );
#else
    KIO::NetAccess::copy( src, savefile );
#endif

    return TRUE;
}


/* private */
void ImgManager::setMosaicPrivate( const KURL& url, bool status )
{
    IMGDAT * imgdat = getImgDat( url );
    if ( !imgdat ) return ;
    if ( imgdat->code != 200 ) return ;

    imgdat->mosaic = status;

    /* save status */
    KConfig cfg( Cache::getImgIdxPath( url ) );
    cfg.writeEntry( "Mosaic", status );
}


/* private */
void ImgManager::createImgDat( const KURL& url, int code )
{
    IMGDAT * imgdat = getImgDat( url );
    if ( !imgdat ) {
        imgdat = new IMGDAT;
        m_imgDatDict.insert( url.prettyURL(), imgdat );
    }

    KConfig cfg( Cache::getImgIdxPath( url ) );
    if ( code == 200 ) cfg.writeEntry( "URL", url.prettyURL() );

    imgdat->mosaic = cfg.readBoolEntry( "Mosaic", KitaConfig::useMosaic() );
    imgdat->height = cfg.readNumEntry( "Height", 0 );
    imgdat->width = cfg.readNumEntry( "Width", 0 );
    imgdat->size = cfg.readUnsignedNumEntry( "Size", 0 );
    imgdat->datURL = cfg.readEntry( "datURL", QString::null );
    imgdat->code = code;
}


/* private */
IMGDAT* ImgManager::getImgDat( const KURL& url )
{
    return m_imgDatDict.find( url.prettyURL() );
}


/* private */
void ImgManager::deleteImgDat( const KURL& url )
{
    IMGDAT * imgdat = getImgDat( url );
    if ( !imgdat ) return ;

    if ( m_imgDatDict.remove( url.prettyURL() ) ) {
        delete imgdat;
    }
}


/* private */
void ImgManager::getSize( const KURL& url )
{
    IMGDAT * imgdat = getImgDat( url );
    if ( !imgdat ) return ;
    if ( imgdat->code != 200 ) return ;

    QString path = Cache::getImgPath( url );
    QImage img = QImage( path );

    if ( !img.isNull() ) {

        imgdat->width = img.width();
        imgdat->height = img.height();

        /* save size */
        KConfig cfg( Cache::getImgIdxPath( url ) );
        cfg.writeEntry( "Width", imgdat->width );
        cfg.writeEntry( "Height", imgdat->height );
    }
}


/* private */
bool ImgManager::cacheExists( const KURL& url )
{
    if ( DownloadManager::isLoadingNow( url ) ) return FALSE;

    bool ret;
    QString path = Cache::getImgPath( url );

#if KDE_IS_VERSION( 3, 2, 0 )
    ret = KIO::NetAccess::exists( path, TRUE, m_mainwidget );
#else
    ret = KIO::NetAccess::exists( path );
#endif

    if ( ret && !getImgDat( url ) ) createImgDat( url, 200 );

    return ret;
}



/* This slot is called when loader received the data. */ /* private slot */
void ImgManager::slotData( const Kita::LoaderData& data, const QByteArray& barray )
{
    /* check BMP header */
    if ( barray.size() > 2 && data.size == barray.size()
            && barray[ 0 ] == 'B' && barray[ 1 ] == 'M' ) {
        DownloadManager::stopLoading( data.url );
        return ;
    }

    /* I assume that file size is smaller than 2^32 byte */
    emit receiveImgData( data.url, data.size, data.totalsize );
}


/* This slot is called when loading is done. */ /* private slot */
void ImgManager::slotResult( const Kita::LoaderData& data )
{
    m_urlList.remove( data.url.prettyURL() );
    createImgDat( data.url, data.code );

    /* save size, datURL, etc */
    if ( data.code == 200 ) {
        unsigned int totalsize = data.totalsize;
        KURL datURL = data.option.usrurl;
        KConfig cfg( Cache::getImgIdxPath( data.url ) );
        cfg.writeEntry( "Size", totalsize );
        cfg.writeEntry( "datURL", datURL.prettyURL() );
        IMGDAT* imgdat = instance->getImgDat( data.url );
        imgdat->size = totalsize;
        imgdat->datURL = datURL;
    }

    emit finishImgLoad( data.url );
}


/* protected */ /* virtual */
void ImgManager::customEvent( QCustomEvent * e )
{
    /* emit finishImgLoad SIGNAL */
    if ( e->type() == EVENT_EmitFinigh ) {
        KURL url = static_cast< EmitFinishEvent* >( e ) ->url();
        emit finishImgLoad( url );
    }
}


