/* Lyrics Plasmoid.
 * Copyright (C) 2010  Mauro E. Bender <maurobender@gmail.com>
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "lyrics_plasmoid.h"

#include <QPainter>
#include <QFontMetrics>
#include <QSizeF>
#include <KLocale>
#include <KConfigDialog>
#include <QGraphicsLinearLayout>
#include <QColor>
#include <QSizePolicy>
#include <QPixmap>
#include <QByteArray>

#include <QMessageBox>

#include <plasma/svg.h>
#include <plasma/theme.h>

#include <QFont>
#include <Plasma/Theme>
#include <QColor>

LyricsPlasmoid::LyricsPlasmoid(QObject *parent, const QVariantList &args)
    : Plasma::Applet(parent, args) {
	// this will get us the standard applet background, for free!
	setBackgroundHints(DefaultBackground);
	setHasConfigurationInterface(true);
	setAspectRatioMode(Plasma::IgnoreAspectRatio);
	
	// The lyrics cache
	m_lyricsCache = new LyricsCache;
	if(!m_lyricsCache->isValid()) {
		qDebug() << ":::Lyrics Plasmoid::: Can't create the cache for the lyrics. Deactivating Lyrics caching.";
	}
	
	// The lyrics sources
	m_lyricsSources = new LyricsSources;
	
	// The lyric getter
	m_lyricsGetter = new LyricsGetter;
	
	if(!m_lyricsSources->sources().empty()) {
		m_lyricsGetter->setLyricSource(m_lyricsSources->sources().value(m_lyricsSources->sources_names().first()));
	} else {
		qDebug() << ":::Lyrics Plasmoid::: There is no sources defined to fecth the lyrics from.";
	}
	
	m_lyrics = new Plasma::TextEdit();
	m_lyrics->setReadOnly(true);
	
	m_currentSong = Song("","");
	m_currentLyrics = "";
	
	m_songInfo = new SongInfo;
	m_titleBar = new TitleBar;
	
	connect(m_titleBar, SIGNAL(reloadLyricsButtonClicked()), this, SLOT(reloadLyrics()));
	connect(m_titleBar, SIGNAL(configurationButtonClicked()), this, SLOT(openConfigDialog()));
	
	connect(m_lyricsGetter, SIGNAL(success(bool, QString)), this, SLOT(updateLyrics(bool, QString)));
	QGraphicsLinearLayout *layout = new QGraphicsLinearLayout(Qt::Vertical);
	layout->addItem(m_songInfo);
	layout->addItem(m_titleBar);
	layout->addItem(m_lyrics);
	
	this->setLayout(layout);
	resize(250, 500);
}


LyricsPlasmoid::~LyricsPlasmoid() {
    if (hasFailedToLaunch()) {
        // Do some cleanup here
    } else {
        // Save settings
    }
}

void LyricsPlasmoid::init() {
	//Load the config
	loadConfig();
	
	// Update the lyrics template with the configuration loaded.
	updateLyricsTemplate();
	
	// Load the data engine "NowPlaying"
	m_dataEngine = dataEngine("nowplaying");
	if (m_dataEngine) {
		connect(m_dataEngine, SIGNAL(sourceAdded(QString)),
					this, SLOT(playerAdded(QString)));
		connect(m_dataEngine, SIGNAL(sourceRemoved(QString)),
					this, SLOT(playerRemoved(QString)));
		
		if(!selectPlayer(m_watchingPlayer))
			findPlayer();
	} else {
		setFailedToLaunch(true, "Can't load the 'nowplaying' Data Engine.");
	}
}

void LyricsPlasmoid::loadConfig() {
	KConfigGroup cg = config();
	
	Plasma::Theme *theme = Plasma::Theme::defaultTheme();
   QFont font = theme->font(Plasma::Theme::DefaultFont);
	QString color = theme->color(Plasma::Theme::TextColor).name();
	
	m_textColor = cg.readEntry("textColor", color);
	m_textFont = cg.readEntry("textFont", font);
	m_textAlignment = cg.readEntry("textAlignment", "center");
	
	m_watchingPlayer = cg.readEntry("watchingPlayer", "");
	
	m_priorizeMetadataLyrics = cg.readEntry("priorizeMetadataLyrics", false); 
	
	m_cacheLyrics = cg.readEntry("cacheLyrics", true);
	if(!m_lyricsCache->isValid()) {
		m_cacheLyrics = false;
	}
	
	QString lyricsSource = cg.readEntry("lyricsSource", "");
	selectLyricsSource(lyricsSource);
}

void LyricsPlasmoid::dataUpdated(const QString &name, const Plasma::DataEngine::Data &data) {
	if(name == m_watchingPlayer && data["State"].toString() != "stopped") {
		// We get the song info from the passed metadata.
		Song song = Song(data["Artist"].toString(), data["Title"].toString());
		
		if(data.contains("Lyrics")) {
			song.lyrics = data["Lyrics"].toString().trimmed().replace("\r\n", "<br />").replace("\n", "<br />");
		}
		
		if(data.contains("Album")) {
			song.album = data["Album"].toString().trimmed();
		}
		
		if(data.contains("Artwork")) {
			song.cover = data["Artwork"].value<QPixmap>();
		}
		
		loadLyrics(song);
	}
}

void LyricsPlasmoid::playerAdded(const QString &name) {
	qDebug() << ":::LyricsPlasmoid::: Player" << name << "added";
	 
	if (m_watchingPlayer.isEmpty()) {
		selectPlayer(name);
	}
}

bool LyricsPlasmoid::selectPlayer(const QString &name) {
	QStringList players = m_dataEngine->sources();
	
	// Is valid the new source player?
	if(name.isEmpty() || players.indexOf(name) == -1) {
		return false; // No, is not.
	}
	
	// Is there currently a source player selected?
	if(!m_watchingPlayer.isEmpty()) {
		// Then we disconnect the current source player.
		m_dataEngine->disconnectSource(m_watchingPlayer, this);
	}
	
	// We connect the new source player.
	qDebug() << ":::Lyrics Plasmoid::: Installing" << name << "as watched player";
	m_watchingPlayer = name;
	m_dataEngine->connectSource(m_watchingPlayer, this, 999);
	
	return true;
}

bool LyricsPlasmoid::selectLyricsSource(const QString &name) {
	QStringList lyricsSources = m_lyricsSources->sources_names();
	
	if(name.isEmpty() || lyricsSources.indexOf(name) == -1) {
		return false;
	}
	
	m_lyricsGetter->setLyricSource(m_lyricsSources->sources().value(name));
	return true;
}

void LyricsPlasmoid::playerRemoved(const QString &name)
{
	qDebug() << "Player" << name << "removed";
	
	if (m_watchingPlayer == name) {
		m_watchingPlayer = "";
		findPlayer();
	}
}

void LyricsPlasmoid::findPlayer()
{
    QStringList players = m_dataEngine->sources();
    qDebug() << "Looking for players.  Possibilities:" << players;
	 
    if (players.isEmpty()) {
        m_watchingPlayer.clear();

        update();
    } else if(m_watchingPlayer.isEmpty()) {
        m_watchingPlayer = players.first();

        qDebug() << "Installing" << m_watchingPlayer << "as watched player";
        m_dataEngine->connectSource(m_watchingPlayer, this, 999);
    }
}

void LyricsPlasmoid::updateLyricsTemplate () {
	QString default_template = 
		"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0//EN\" \"http://www.w3.org/TR/REC-html40/strict.dtd\"> "
		"<html><head><meta name=\"qrichtext\" content=\"1\" /> "
		"<style type=\"text/css\"> p.lyrics {font-family:'[%FAMILY%]'; font-size: [%SIZE%]; color: [%COLOR%]; font-weight: [%WEIGHT%]; font-style: [%STYLE%]; } p, li { white-space: pre-wrap; }</style></head> "
		"<body style=\"font-family:'Sans Serif'; font-size:9pt; font-weight:400; font-style:normal;\">"
		"<p align=\"[%ALIGN%]\" class=\"lyrics\">[%LYRIC%]</p>"
		"</body></html>";
	 
	
	m_lyricTemplate = default_template;
	m_lyricTemplate.replace("[%COLOR%]", m_textColor);
	m_lyricTemplate.replace("[%ALIGN%]", m_textAlignment);
	m_lyricTemplate.replace("[%FAMILY%]", m_textFont.family());
	
	m_lyricTemplate.replace("[%SIZE%]", QString("%1px").arg((m_textFont.pixelSize() != -1 ? m_textFont.pixelSize() : m_textFont.pointSize())));
	
	m_lyricTemplate.replace("[%WEIGHT%]", (m_textFont.bold() ? "bold" : "normal"));
	m_lyricTemplate.replace("[%STYLE%]", (m_textFont.italic() ? "italic" : "normal"));
	
	//qDebug() << "Lyric template updated: " << m_lyricTemplate;
}

void LyricsPlasmoid::updateLyrics (bool found, QString lyrics_text) {
	// If we are caching the lyrics then save this lyric to the database.
	if(!m_currentSong.artist.isEmpty() && !m_currentSong.title.isEmpty() && found && m_cacheLyrics && m_lyricsCache->isValid()) {
		m_lyricsCache->writeLyric(m_currentSong, lyrics_text);
	}
	
	QString text = m_lyricTemplate;
	m_currentLyrics = lyrics_text;
	m_lyrics->setText(text.replace("[%LYRIC%]", lyrics_text));
	//qDebug() << "Lyrics updated: " << m_currentLyrics;
}

void LyricsPlasmoid::reloadLyrics() {
	// Option 1
	// loadLyrics(m_currentSong, true);
	
	// Option 2
	m_lyricsCache->deleteLyric(m_currentSong);
	m_currentSong = Song();
	update();
}

void LyricsPlasmoid::loadLyrics(const Song &song, bool fullLoad) {
	//bool reload_lyrics_from_player = m_priorizeMetadataLyrics && !song.lyrics.isEmpty() && song.lyrics != m_currentLyrics;
	if (song != Song("", "") && (fullLoad || m_currentSong != song)) {
		m_currentSong = song;
		qDebug() << "Reloading lyrics.";
		
		QString text = m_lyricTemplate;
		m_lyrics->setText(text.replace("[%LYRIC%]", i18n("Loading...")));
		m_songInfo->setSong(m_currentSong);
		
		if(fullLoad) {
			// Delete the lyrics from cache.
			m_currentSong.lyrics = "";
			m_lyricsCache->deleteLyric(m_currentSong);
		}
		
		bool lyric_loaded = false;
		
		if(m_priorizeMetadataLyrics && !m_currentSong.lyrics.isEmpty()) {
			qDebug() << ":::LyricsPlasmoid::: Lyrics got from the metadata.";
			lyric_loaded = true;
		} else {
			qDebug() << ":::LyricsPlasmoid::: The lyrics from the metadata are empty.";
		}
		
		// If is selected the option "Cache lyrics" and the lyrics are in the cache
		// then we load the lyrics from the cache.
		QString lyrics;
		if(!lyric_loaded && m_cacheLyrics && m_lyricsCache->isValid() && !(lyrics = m_lyricsCache->readLyric(m_currentSong)).isEmpty()) {
			qDebug() << ":::LyricsPlasmoid::: Lyrics loaded from cache.";
			m_currentSong.lyrics = lyrics;
			lyric_loaded = true;
		} else {
			qDebug() << ":::LyricsPlasmoid::: Lyrics not found in the cache.";
		}
		
		// If the lyrics were loaded then we show those lyrics.
		if(lyric_loaded) {
			updateLyrics(true, m_currentSong.lyrics);
		} else {
			// Else we fetch the lyrics from the source selected.
			m_lyricsGetter->getLyric(m_currentSong.artist, m_currentSong.title);
		}
	}
}


void LyricsPlasmoid::openConfigDialog() {
	showConfigurationInterface();
}

void LyricsPlasmoid::createConfigurationInterface(KConfigDialog *parent)
{
	//Create the Appearance Config Tab
	QWidget *appearanceConfigWidget = new QWidget();
	m_appearanceConfigUi.setupUi(appearanceConfigWidget);
	 
	//Load the KDE Icons
	m_appearanceConfigUi.btn_align_left->setIcon(KIcon("format-justify-left"));
	m_appearanceConfigUi.btn_align_center->setIcon(KIcon("format-justify-center"));
	m_appearanceConfigUi.btn_align_right->setIcon(KIcon("format-justify-right"));
	
	//Selected options
	m_appearanceConfigUi.textFont->setFont(m_textFont);
	m_appearanceConfigUi.textColor->setColor(QColor(m_textColor));
	m_appearanceConfigUi.btn_align_left->setChecked(m_textAlignment == "left");
	m_appearanceConfigUi.btn_align_center->setChecked(m_textAlignment == "center");
	m_appearanceConfigUi.btn_align_right->setChecked(m_textAlignment == "right");
	
	//Create the Sources Config Tab
	QWidget *sourcesConfigWidget = new QWidget();
	m_sourcesConfigUi.setupUi(sourcesConfigWidget);
	
	QStringList playersSources = m_dataEngine->sources();
	m_sourcesConfigUi.cb_playerSource->addItems(playersSources);
	m_sourcesConfigUi.cb_playerSource->setCurrentIndex(playersSources.indexOf(m_watchingPlayer));
	
	if(!m_lyricsCache->isValid()) {
		m_sourcesConfigUi.cb_cacheLyrics->setEnabled(false);
	}
	m_sourcesConfigUi.cb_cacheLyrics->setChecked(m_cacheLyrics);
	
	m_sourcesConfigUi.cb_priorizeMetadataLyrics->setChecked(m_priorizeMetadataLyrics);
	
	QStringList lyricsSources = m_lyricsSources->sources_names();
	m_sourcesConfigUi.cb_lyricsSource->addItems(lyricsSources);
	m_sourcesConfigUi.cb_lyricsSource->setCurrentIndex(lyricsSources.indexOf(m_lyricsGetter->currentLyricSource()->name));
	
	parent->addPage(appearanceConfigWidget, i18n("Appearance"), "applications-graphics");
	parent->addPage(sourcesConfigWidget, i18n("Sources Settings"), "applications-system");
	parent->setButtons( KDialog::Ok | KDialog::Cancel);
	
	connect(parent, SIGNAL(okClicked()), this, SLOT(configAccepted()));
	parent->resize(300,350);
	
}

void LyricsPlasmoid::configAccepted() {
	bool fullReloadLyrics = false;
	//Save the configuration
	KConfigGroup cg = config();
	m_textColor = m_appearanceConfigUi.textColor->color().name();
	cg.writeEntry("textColor", m_textColor);
	
	m_textFont = m_appearanceConfigUi.textFont->font();
	cg.writeEntry("textFont", m_textFont);
	
	if(m_appearanceConfigUi.btn_align_left->isChecked()) {
			m_textAlignment = "left";
			cg.writeEntry("textAlignment", "left");
	}
	
	if(m_appearanceConfigUi.btn_align_center->isChecked()) {
			m_textAlignment = "center";
			cg.writeEntry("textAlignment", "center");
	}
	
	if(m_appearanceConfigUi.btn_align_right->isChecked()) {
			m_textAlignment = "right";
			cg.writeEntry("textAlignment", "right");
	}
	
	if(selectPlayer(m_sourcesConfigUi.cb_playerSource->currentText())) {
		cg.writeEntry("watchingPlayer", m_watchingPlayer);
	}
	
	m_cacheLyrics = m_sourcesConfigUi.cb_cacheLyrics->isChecked();
	m_priorizeMetadataLyrics = m_sourcesConfigUi.cb_priorizeMetadataLyrics->isChecked();
	cg.writeEntry("cacheLyrics", m_cacheLyrics);
	cg.writeEntry("priorizeMetadataLyrics", m_priorizeMetadataLyrics);
	
	QString oldSource = m_lyricsGetter->currentLyricSource()->name;
	if(selectLyricsSource(m_sourcesConfigUi.cb_lyricsSource->currentText())) {
		QString newSource = m_lyricsGetter->currentLyricSource()->name;
		cg.writeEntry("lyricsSource", m_lyricsGetter->currentLyricSource()->name);
		fullReloadLyrics = oldSource != newSource;
	}
	
	emit configNeedsSaving();
	
	//Update the plasmoid
	updateLyricsTemplate();
	if(fullReloadLyrics)
		reloadLyrics();
	else
		updateLyrics(true, m_currentLyrics);
}

#include "lyrics_plasmoid.moc"
