#!/usr/bin/env bash

################################################################################
# xtrfm version 1.0.1
# RFM分析
# 2003/07/22 Y.Hamuro 雛型作成
# 2004/01/21 top行 /bin/bash -> /usr/bin/env bash に変更(増田氏のご指摘)
# 2004/01/21 ヘルプでバージョン情報を表示するように変更(1.0.1)
################################################################################

#===============================================================================
#ヘルプメッセージ( -hが与えられたときに表示される )
#-------------------------------------------------------------------------------
# -hが与えられたときに表示される。
# "cat >/dev/stderr <<EOR"から"EOF"の行までのテキストがそのまま画面に表示される
# 利用法において、省略可能なパラメータは括弧("[]")で括ること。
#===============================================================================
function help {
cat >/dev/stderr <<EOF
------------------------
xtrfm.sh version 1.0.1
------------------------
利用法) xtrfm [-c 顧客ID項目名] [-d 日付項目名] [-D 基準日付] [-m 金額項目名]
              [-p期間] [-R Recencyのクラス数] [-F Frequencyのクラス数]
              [-M Monetaryのクラス数] -i入力ファイル名 -O 出力ディレクトリ名
  -c 顧客ID項目名 : 顧客IDを示す項目名を指定する(複数項目の指定可能)
                    省略時は"顧客"という項目名を利用する。
  -d 日付項目名   : 顧客の訪問日を示す項目名を指定する。
                    省略時は"日付"という項目名を利用する。
  -D 基準日付     : Recencyを計算するための基準日。YYYYMMDDの形式で指定する。
                    Recencyは、「基準日付」−「顧客の最終訪問日」で計算される
                    日数により定義される。
                    省略時は本日の日付が用いられる。
  -m 金額項目名   : 顧客の購入金額を示す項目名を指定する。
                    省略時は"金額"という項目名を利用する。
  -p 期間         : 指定された期間でデータを選択した後に、RFM分析を行う。
                    期間はYYYYMMDD_YYYYMMDDの形式で指定する。
                    省略時は入力ファイル全てのデータを用いてRFM分析を行う。
  -R              : Recencyを何個のクラスに分割するか(デフォルトは5)
  -F              : Frequencyを何個のクラスに分割するか(デフォルトは5)
  -M              : Monetaryを何個のクラスに分割するか(デフォルトは5)
  -i 入力ファイル名 : ワイルドカードが利用できる。
  -O 出力ディレクトリ名 : 結果が出力されるディレクトリ名。
                          指定したディレクトリが存在しなければ作成される。
                          省略時は、カレントディレクトリの"outfiles"に作成。

利用例) xtrfm -i dat.xt -O rsl
EOF
exit 1
}

#===============================================================================
#パラメータのセット
# 引数をとるパラメータは、その値を変数にセットし、引数をとらないパラメータは、
# 変数に"1"をセットする。
#-------------------------------------------------------------------------------
# getoptsの引数について
#   最初のコロン(:)は、不正なオプション入力時に、opt変数に"?"をセットする。
#   このことにより、次のケース文の"\?)"に引っかかり、helpを実行することになる。
#   設定したパラメータを、"アルファベット+コロン"(ex."i:")で与えると引数をとる
#   パラメータとして扱い、コロンなしの場合は、引数のないパラメータとして扱う。
#   ex. "i:a:thv" -> "-i", "-a"は引数をとるパラメータ。
#                    "-t", "-h", "-v"は引数をとらないパラメータ。
#   引数はOPTARG変数にセットされる。
#
# while & case の処理について
#   whileループ(whileからdoneまでのブロック)一回ごとに、ユーザが入力した
#   パラメータを一つづつチェックし、getoptsの引数(上記説明参照)に基づき、
#   opt,OPTARG変数をセットする。
#   opt変数には、ユーザが入力したパラメータのキーワードがセットされ、
#   OPTARG変数には、そのキーワードが引数をとるなら、その値がセットされる。
#   ユーザが不正なパラメータを入力した場合は、opt変数に"?"がセットされ、
#   "\? )"の一文が実行される。
#
# case文とは
#   "case $opt in 〜 case"は、opt変数の内容が、「 x )」のxの値と同じ行の
#   コマンドを実行する。「 x )」の後には複数のコマンドを記述可能であるが、
#   最後のコマンドの後に";;"(終端記号)をつける必要がある。
#===============================================================================
  while getopts ":c:d:D:m:p:R:F:M:i:O:h" opt; do
    case $opt in
      c  ) cFld=$OPTARG ;;
      d  ) dFld=$OPTARG ;;
      D  ) today=$OPTARG ;;
      m  ) mFld=$OPTARG ;;
      p  ) period=$OPTARG ;;
      R  ) bucketR=$OPTARG ;;
      F  ) bucketF=$OPTARG ;;
      M  ) bucketM=$OPTARG ;;
      i  ) inFile=$OPTARG ;;
      O  ) outPath=$OPTARG ;;
      h  ) help ;;
      \? ) echo xxxx; exit 0 ;help ;;
    esac
  done

#-------------------------------------------------------------------------------
# OPTINDを１減算する。->この処理により、上記パラメータを除いた引数を$nで参照
# 可能となる。
#-------------------------------------------------------------------------------
  shift $(($OPTIND -1 ))

#-------------------------------------------------------------------------------
# パラメータのチェック
#-------------------------------------------------------------------------------
  if [ "$cFld" = "" ]; then
     cFld="顧客"
  fi
  if [ "$dFld" = "" ]; then
     dFld="日付"
  fi
  if [ "$today" = "" ]; then
     today=`date +%Y%m%d`
  fi
  if [ "$mFld" = "" ]; then
     mFld="金額"
  fi
  if [ "$period" = "" ]; then
     periodFrom="0"
     periodTo="99999999"
  else
     periodFrom=${period%_*}
     periodTo=${period#*_}
  fi
  if [ "$bucketR" = "" ]; then
    bucketR=5
  fi
  if [ "$bucketF" = "" ]; then
    bucketF=5
  fi
  if [ "$bucketM" = "" ]; then
    bucketM=5
  fi
  if [ "$inFile" = "" ]; then
     help; exit 1
  fi
  if [ "$outPath" = "" ]; then
     outPath="./outfiles"
  fi

#-------------------------------------------------------------------------------
# 各種前処理
#-------------------------------------------------------------------------------
# ワークファイルのプレフィックスの設定
# $$はプロセス番号
wf=/tmp/xt##$$-xtrfm

#出力ディレクトリが存在しないなら作成する
  if [ ! -d $outPath ]; then
     mkdir -p $outPath
  fi

#-------------------------------------------------------------------------------
# メイン処理処理
#-------------------------------------------------------------------------------
# 1. 入力データを併合し、
# 2. 必要な項目(顧客,日付,金額)を抜きだし、
# 3. データを選択し、
# 4. 顧客別日別金額合計を求める。
xtcat -i "$inFile" |
xtsel -c "\$$dFld>$periodFrom && \$$dFld<$periodTo" |
xtcut -f $cFld,$dFld,$mFld |
xtagg -k $cFld,$dFld -f $mFld -c sum -o $wf-01

#来店頻度クラスを求める
xtcut -f $cFld -i $wf-01 |
xtcount -k $cFld -a 来店回数 |
xtbucket -n $bucketF -f 来店回数:Fclass -o $wf-F

#金額クラスを求める
xtcut -f $cFld,$mFld -i $wf-01 |
xtagg -k $cFld -f $mFld -c sum |
xtbucket -n $bucketM -f $mFld:Mclass -o $wf-M

#Recencyクラスを求める
xtsetchr -v "$today" -a today -i $wf-01 |
xtcut -f $cFld,$dFld,today |
xtagg -k $cFld -f $dFld -c max |
xtcal -c "day($today,\$$dFld)" -a Recency |
xtbucket -n $bucketR -f Recency:Rclass -o $wf-R

#三つのファイルを結合
xtjoin -nN -k $cFld -m $wf-F -f Fclass -i $wf-R |
xtjoin -nN -k $cFld -m $wf-M -f Mclass |
xtcut -f $cFld,Rclass,Fclass,Mclass -o $outPath/dat.xt

#-------------------------------------------------------------------------------
# 後処理
#-------------------------------------------------------------------------------
# ワークファイルの削除
  rm $wf-*

# 終了(ステータスは０)
  exit 0
