/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: externallinkbuffer.cxx,v $
 *
 *  $Revision: 1.1.2.2 $
 *
 *  last change: $Author: dr $ $Date: 2007/08/24 09:07:42 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "oox/xls/externallinkbuffer.hxx"
#include <rtl/ustrbuf.hxx>
#include <com/sun/star/container/XIndexAccess.hpp>
#include <com/sun/star/container/XNameAccess.hpp>
#include <com/sun/star/sheet/SheetLinkMode.hpp>
#include <com/sun/star/sheet/XSheetLinkable.hpp>
#include <com/sun/star/sheet/XSpreadsheet.hpp>
#include <com/sun/star/sheet/XSpreadsheetDocument.hpp>
#include <com/sun/star/sheet/XSpreadsheets.hpp>
#include <com/sun/star/table/XCell.hpp>
#include "oox/core/containerhelper.hxx"
#include "oox/core/propertyset.hxx"
#include "oox/xls/biffinputstream.hxx"
#include "oox/xls/contexthelper.hxx"

using ::rtl::OUString;
using ::rtl::OUStringBuffer;
using ::com::sun::star::uno::Reference;
using ::com::sun::star::uno::UNO_QUERY;
using ::com::sun::star::uno::UNO_QUERY_THROW;
using ::com::sun::star::container::XIndexAccess;
using ::com::sun::star::container::XNameAccess;
using ::com::sun::star::table::CellAddress;
using ::com::sun::star::table::XCell;
using ::com::sun::star::sheet::XSheetLinkable;
using ::com::sun::star::sheet::XSpreadsheet;
using ::com::sun::star::sheet::XSpreadsheetDocument;
using ::com::sun::star::sheet::XSpreadsheets;
using ::oox::core::AttributeList;
using ::oox::core::RefVector;

namespace oox {
namespace xls {

// ============================================================================

ExtRef::ExtRef()
{
}

// ============================================================================
// ============================================================================

class BiffExternName : public GlobalDataHelper
{
public:
    explicit            BiffExternName( const GlobalDataHelper& rGlobalData );

    /** Imports the EXTERNNAME record from the passed stream. */
    void                importExternName( BiffInputStream& rStrm );

    /** Returns the name of this external name. */
    const OUString&     getName() const { return maName; }

private:
    OUString            maName;
};

typedef ::boost::shared_ptr< BiffExternName > BiffExternNameRef;

// ----------------------------------------------------------------------------

BiffExternName::BiffExternName( const GlobalDataHelper& rGlobalData ) :
    GlobalDataHelper( rGlobalData )
{
}

void BiffExternName::importExternName( BiffInputStream& rStrm )
{
    /*sal_uInt16 nFlags =*/ (getBiff() >= BIFF4) ? rStrm.readuInt16() : 0;
    if( getBiff() >= BIFF5 )
        rStrm.ignore( 4 );
    maName = (getBiff() == BIFF8) ?
        rStrm.readUniString( rStrm.readuInt8() ) :
        rStrm.readByteString( false, getTextEncoding() );
}

// ============================================================================

const sal_uInt16 BIFF_SUPBOOK_INTERNAL  = 0x0401;   /// 3D-reference to own workbook.
const sal_uInt16 BIFF_SUPBOOK_ANALYSIS  = 0x3A01;   /// Analysis add-in.

// ----------------------------------------------------------------------------

class BiffExternLink : public GlobalDataHelper
{
public:
    explicit            BiffExternLink( const GlobalDataHelper& rGlobalData );

    /** Imports the EXTERNSHEET record from the passed stream. */
    void                importExternSheet( BiffInputStream& rStrm );
    /** Imports the SUPBOOK record from the passed stream. */
    void                importSupBook( BiffInputStream& rStrm );
    /** Imports the EXTERNNAME record from the passed stream. */
    void                importExternName( BiffInputStream& rStrm );

    /** Returns the type of this external link object. */
    inline ExternalLinkType getLinkType() const { return meType; }
    /** Returns the external name with the passed BIFF name index (one-based). */
    BiffExternNameRef   getExternName( sal_uInt16 nNameId ) const;

private:
    typedef RefVector< BiffExternName > ExternNameVec;

    ExternNameVec       maExtNames;
    ExternalLinkType    meType;
};

typedef ::boost::shared_ptr< BiffExternLink > BiffExternLinkRef;

// ----------------------------------------------------------------------------

BiffExternLink::BiffExternLink( const GlobalDataHelper& rGlobalData ) :
    GlobalDataHelper( rGlobalData ),
    meType( LINKTYPE_UNKNOWN )
{
}

void BiffExternLink::importExternSheet( BiffInputStream& /*rStrm*/ )
{
}

void BiffExternLink::importSupBook( BiffInputStream& rStrm )
{
    sal_uInt16 nSheetCount;
    rStrm >> nSheetCount;
    if( rStrm.getRecLeft() == 2 )
    {
        switch( rStrm.readuInt16() )
        {
            case BIFF_SUPBOOK_INTERNAL: meType = LINKTYPE_INTERNAL; break;
            case BIFF_SUPBOOK_ANALYSIS: meType = LINKTYPE_ANALYSIS; break;
        }
    }
}

void BiffExternLink::importExternName( BiffInputStream& rStrm )
{
    BiffExternNameRef xExtName( new BiffExternName( getGlobalData() ) );
    maExtNames.push_back( xExtName );
    xExtName->importExternName( rStrm );
}

BiffExternNameRef BiffExternLink::getExternName( sal_uInt16 nNameId ) const
{
    OSL_ENSURE( nNameId > 0, "BiffExternLink::getExternName - invalid name index" );
    return maExtNames.get( static_cast< sal_Int32 >( nNameId ) - 1 );
}

// ============================================================================

/** Represents a REF entry in the BIFF8 EXTERNSHEET record, maps ref identifiers
    to SUPBOOK records, and provides sheet indexes in the document. */
struct Biff8RefEntry
{
    sal_uInt16          mnSupBookId;        /// Zero-based index to SUPBOOK record.
    sal_Int16           mnTabId1;           /// Zero-based index to first sheet in SUPBOOK record.
    sal_Int16           mnTabId2;           /// Zero-based index to last sheet in SUPBOOK record.
};

// ============================================================================

class BiffExternalLinkBufferImpl : public GlobalDataHelper
{
public:
    explicit            BiffExternalLinkBufferImpl( const GlobalDataHelper& rGlobalData );

    /** Imports the EXTERNSHEET record from the passed stream. */
    void                importExternSheet( BiffInputStream& rStrm );
    /** Imports the SUPBOOK record from the passed stream. */
    void                importSupBook( BiffInputStream& rStrm );
    /** Imports the EXTERNNAME record from the passed stream. */
    void                importExternName( BiffInputStream& rStrm );

    /** Returns the link type for the passed ref identifier. */
    ExternalLinkType    getLinkType( sal_Int32 nRefId ) const;
    /** Returns the sheet range for the specified reference (BIFF5). */
    BiffLinkSheetRange  getSheetRange( sal_Int32 nRefId, sal_Int16 nTab1, sal_Int16 nTab2 ) const;
    /** Returns the sheet range for the specified reference (BIFF8). */
    BiffLinkSheetRange  getSheetRange( sal_Int32 nRefId ) const;
    /** Returns the name of the external name with the passed name index. */
    OUString            getExternName( sal_Int32 nRefId, sal_uInt16 nNameId ) const;

private:
    BiffExternLink&     createExternLink();

    const Biff8RefEntry* getRefEntry( sal_Int32 nRefId ) const;
    BiffExternLinkRef   getExternLink( sal_Int32 nRefId ) const;

private:
    typedef ::oox::core::RefVector< BiffExternLink >    BiffExternLinkVec;
    typedef ::std::vector< Biff8RefEntry >              Biff8RefEntryVec;

    BiffExternLinkVec   maExtLinks;         /// List of external documents.
    Biff8RefEntryVec    maRefEntries;       /// Contents of BIFF8 EXTERNSHEET record.
};

// ----------------------------------------------------------------------------

BiffExternalLinkBufferImpl::BiffExternalLinkBufferImpl( const GlobalDataHelper& rGlobalData ) :
    GlobalDataHelper( rGlobalData )
{
}

void BiffExternalLinkBufferImpl::importExternSheet( BiffInputStream& rStrm )
{
    switch( getBiff() )
    {
        case BIFF2:
        case BIFF3:
        case BIFF4:
        case BIFF5:
            createExternLink().importExternSheet( rStrm );
        break;
        case BIFF8:
        {
            OSL_ENSURE( maRefEntries.empty(), "BiffExternalLinkBufferImpl::importExternSheet - multiple EXTERNSHEET records" );
            sal_uInt16 nRefCount;
            rStrm >> nRefCount;
            maRefEntries.reserve( nRefCount );
            for( sal_uInt16 nRefId = 0; rStrm.isValid() && (nRefId < nRefCount); ++nRefId )
            {
                Biff8RefEntry aRefEntry;
                rStrm >> aRefEntry.mnSupBookId >> aRefEntry.mnTabId1 >> aRefEntry.mnTabId2;
                maRefEntries.push_back( aRefEntry );
            }
        }
        break;
        case BIFF_UNKNOWN: break;
    }
}

void BiffExternalLinkBufferImpl::importSupBook( BiffInputStream& rStrm )
{
    createExternLink().importSupBook( rStrm );
}

void BiffExternalLinkBufferImpl::importExternName( BiffInputStream& rStrm )
{
    if( !maExtLinks.empty() )
        maExtLinks.back()->importExternName( rStrm );
}

ExternalLinkType BiffExternalLinkBufferImpl::getLinkType( sal_Int32 nRefId ) const
{
    ExternalLinkType eLinkType = LINKTYPE_UNKNOWN;
    switch( getBiff() )
    {
        case BIFF2:
        case BIFF3:
        case BIFF4:
        break;
        case BIFF5:
            if( nRefId < 0 )
                eLinkType = LINKTYPE_INTERNAL;
            else if( const BiffExternLink* pExtLink = getExternLink( nRefId ).get() )
                eLinkType = pExtLink->getLinkType();
        break;
        case BIFF8:
            if( const BiffExternLink* pExtLink = getExternLink( nRefId ).get() )
                eLinkType = pExtLink->getLinkType();
        break;
        case BIFF_UNKNOWN: break;
    }
    return eLinkType;
}

BiffLinkSheetRange BiffExternalLinkBufferImpl::getSheetRange( sal_Int32 nRefId, sal_Int16 nTabId1, sal_Int16 nTabId2 ) const
{
    OSL_ENSURE( getBiff() == BIFF5, "BiffExternalLinkBufferImpl::getSheetRange - wrong BIFF version" );
    BiffLinkSheetRange aSheetRange;
    if( getLinkType( nRefId ) == LINKTYPE_INTERNAL )
        aSheetRange.set( nTabId1, nTabId2 );
    return aSheetRange;
}

BiffLinkSheetRange BiffExternalLinkBufferImpl::getSheetRange( sal_Int32 nRefId ) const
{
    OSL_ENSURE( getBiff() == BIFF8, "BiffExternalLinkBufferImpl::getSheetRange - wrong BIFF version" );
    BiffLinkSheetRange aSheetRange;
    if( getLinkType( nRefId ) == LINKTYPE_INTERNAL )
        if( const Biff8RefEntry* pRefEntry = getRefEntry( nRefId ) )
            aSheetRange.set( pRefEntry->mnTabId1, pRefEntry->mnTabId2 );
    return aSheetRange;
}

OUString BiffExternalLinkBufferImpl::getExternName( sal_Int32 nRefId, sal_uInt16 nNameId ) const
{
    OUString aExtName;
    if( const BiffExternLink* pExtLink = getExternLink( nRefId ).get() )
        if( const BiffExternName* pExtName = pExtLink->getExternName( nNameId ).get() )
            aExtName = pExtName->getName();
    return aExtName;
}

BiffExternLink& BiffExternalLinkBufferImpl::createExternLink()
{
    BiffExternLinkRef xExtLink( new BiffExternLink( getGlobalData() ) );
    maExtLinks.push_back( xExtLink );
    return *xExtLink;
}

const Biff8RefEntry* BiffExternalLinkBufferImpl::getRefEntry( sal_Int32 nRefId ) const
{
    return ((0 <= nRefId) && (static_cast< size_t >( nRefId ) < maRefEntries.size())) ?
        &maRefEntries[ static_cast< size_t >( nRefId ) ] : 0;
}

BiffExternLinkRef BiffExternalLinkBufferImpl::getExternLink( sal_Int32 nRefId ) const
{
    BiffExternLinkRef xExtLink;
    switch( getBiff() )
    {
        case BIFF2:
        case BIFF3:
        case BIFF4:
        break;
        case BIFF5:
            xExtLink = maExtLinks.get( nRefId );
        break;
        case BIFF8:
            if( const Biff8RefEntry* pRefEntry = getRefEntry( nRefId ) )
                xExtLink = maExtLinks.get( pRefEntry->mnSupBookId );
        break;
        case BIFF_UNKNOWN: break;
    }
    return xExtLink;
}

// ============================================================================
// ============================================================================

ExternalLinkBuffer::ExternalLinkBuffer( const GlobalDataHelper& rGlobalData ) :
    GlobalDataHelper( rGlobalData ),
    mxBiffImpl( new BiffExternalLinkBufferImpl( rGlobalData ) )
{
}

void ExternalLinkBuffer::setExternalLink( sal_Int32 nId, const OUString& aPath )
{
    ExtRef aItem;
    aItem.maPath = aPath;
    maExtRefs.insert( ExtRefMap::value_type(nId, aItem) );
}

const ExtRef* ExternalLinkBuffer::getExternalLinkById( sal_Int32 nId ) const
{
    ExtRefMap::const_iterator pos = maExtRefs.find( nId );
    if ( pos != maExtRefs.end() )
        return &pos->second;

    return NULL;
}

bool ExternalLinkBuffer::isExtRef( const OUString& aCellFormula ) const
{
    sal_Int32 nLen = aCellFormula.getLength();
    const sal_Unicode* p = aCellFormula.getStr();
    if ( p[0] != static_cast<sal_Unicode>('[') )
        // For this to be a valid external reference, the first character
        // must be '['.
        return false;

    sal_uInt8 nDigitCount = 0;
    for ( sal_Int32 i = 1; i < nLen; ++i )
    {
        if ( p[i] >= static_cast<sal_Unicode>('0') && p[i] <= static_cast<sal_Unicode>('9') )
        {
            ++nDigitCount;
            if ( nDigitCount > 10 )
                // 10 digits ought to be enough.  Let's mark any larger number invalid.
                return false;
        }
        else if ( p[i] == static_cast<sal_Unicode>(']') )
            return (nDigitCount > 0);
        else
            return false;
    }
    return false;
}

void ExternalLinkBuffer::addToDestCellList( const CellAddress& aAddr, const OUString& aFormula )
{
    ExtRefCell aCell;
    aCell.maCellAddress = aAddr;
    aCell.maFormulaString = aFormula;
    maDestCellList.push_back(aCell);
}

void ExternalLinkBuffer::importExternSheet( BiffInputStream& rStrm )
{
    mxBiffImpl->importExternSheet( rStrm );
}

void ExternalLinkBuffer::importSupBook( BiffInputStream& rStrm )
{
    mxBiffImpl->importSupBook( rStrm );
}

void ExternalLinkBuffer::importExternName( BiffInputStream& rStrm )
{
    mxBiffImpl->importExternName( rStrm );
}

void ExternalLinkBuffer::finalizeImport()
{
    setExtRefCells();
}

ExternalLinkType ExternalLinkBuffer::getLinkType( sal_Int32 nRefId ) const
{
    return mxBiffImpl->getLinkType( nRefId );
}

BiffLinkSheetRange ExternalLinkBuffer::getSheetRange( sal_Int32 nRefId, sal_Int16 nTabId1, sal_Int16 nTabId2 ) const
{
    return mxBiffImpl->getSheetRange( nRefId, nTabId1, nTabId2 );
}

BiffLinkSheetRange ExternalLinkBuffer::getSheetRange( sal_Int32 nRefId ) const
{
    return mxBiffImpl->getSheetRange( nRefId );
}

OUString ExternalLinkBuffer::getExternName( sal_Int32 nRefId, sal_uInt16 nNameId ) const
{
    return mxBiffImpl->getExternName( nRefId, nNameId );
}

void ExternalLinkBuffer::setExtRefCells()
{
    const OUString aEq( sal_Unicode( '=' ) );

    // Now, iterate through the cached destination cells, and insert the references.
    ExtRefCellList::const_iterator itr = maDestCellList.begin(), itrEnd = maDestCellList.end();
    for ( ; itr != itrEnd; ++itr )
    {
        Reference< XCell > xCell = getCell( itr->maCellAddress );
        if ( !xCell.is() )
            continue;

        OUString aFormula( aEq + itr->maFormulaString );
        expandFormula( aFormula );
        xCell->setFormula( aFormula );
    }
}

void ExternalLinkBuffer::expandFormula( OUString& rFormula )
{
    static const sal_Unicode nOpenAB('['), nCloseAB(']'), nExclam('!');
    OUStringBuffer aBuf(rFormula), aBufOut, aBufExtId;
    sal_Int32 nLen = aBuf.getLength();
    sal_uInt8 nScope = 0;
    for ( sal_Int32 i = 0; i < nLen; ++i )
    {
        sal_Unicode c = aBuf.charAt(i);
        if ( c == nOpenAB )
        {
            ++nScope;
            aBufOut.appendAscii("'");
        }
        else if ( c == nCloseAB )
        {
            --nScope;
            if ( aBufExtId.getLength() > 0 )
            {
                sal_Int32 nId = aBufExtId.makeStringAndClear().toInt32();
                const ExtRef* pExtRef = getExternalLinkById(nId);
                if ( pExtRef )
                    aBufOut.append(pExtRef->maPath);
                else
                    aBufOut.appendAscii("???");
            }
            aBufOut.appendAscii("'#");
        }
        else if ( c == nExclam )
        {
            aBufOut.appendAscii(".");
        }
        else if ( nScope > 0 )
        {
            aBufExtId.append(c);
        }
        else
        {
            aBufOut.append(c);
        }
    }
    rFormula = aBufOut.makeStringAndClear();
}

// ============================================================================

} // namespace xls
} // namespace oox

