/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#ifndef RAST_LOCAL_DB_H
#define RAST_LOCAL_DB_H

/**
 * @file local_db.h local_db
 */

#include <apr_pools.h>
#include <apr_file_io.h>

#include "rast/rast.h"
#include "rast/error.h"
#include "rast/encoding.h"
#include "rast/text_index.h"
#include "rast/property.h"
#include "rast/db.h"
#include "rast/macros.h"

RAST_EXTERN_C_BEGIN

/**
 * @defgroup local_db local_db
 * @{
 */

/** A structure that represents an index to search properties */
typedef struct {
    /** The index to search properties */
    DB *inv;
    /** The index to search text in properties */
    rast_text_index_t *text;
} rast_property_index_t;

typedef struct {
    rast_document_t base;
    rast_doc_id_t doc_id;
    rast_size_t nchars;
    rast_text_indexer_t *indexer;
    apr_hash_t *property_values;
} rast_local_document_t;

/** A structure that represents a database to search documents */
typedef struct {
    /** The super class instance */
    rast_db_t base;

    /** The path of the database */
    const char *path;
    /**
     * @param flags for rast_local_db_open()
     *          - RAST_RDRW
     *            - open for reading and writing
     *          - RAST_RDONLY
     *            - open for reading
     */
    int flags;
    /** The encoding of the database */
    const char *encoding;
    /** The encoding module of the database */
    rast_encoding_module_t *encoding_module;
    /** Whether the whole text should be preseved in the database */
    int preserve_text;
    /** Whether the numbers in the database are in the natvie byte order */
    int is_native;
    /** The property definitions */
    rast_property_t *properties;
    /** The number of properties */
    int num_properties;
    /** The block size of .pos files */
    rast_size_t pos_block_size;
    /** The index to search text */
    rast_text_index_t *text_index;
    /** The database that holds summary texts */
    DB *text_db;
    /** The database that holds properties */
    DB *properties_db;
    /** The indices to search properties */
    rast_property_index_t *property_indices;
    /** The Berkeley DB environment */
    DB_ENV *bdb_env;
    /** The Berkeley DB transaction */
    DB_TXN *bdb_txn;
    /** The lock file */
    apr_file_t *lock_file;
    /** The number of registered characters to start sync */
    int sync_threshold_chars;
    /** The number of registered characters */
    int registered_chars;
} rast_local_db_t;

/**
 * Create a database.  This function do not open the created database.
 * Use rast_local_db_open() to open the database.
 * @param name The name of the database
 * @param options The options to create a database.
 * @param pool The pool to allocate the memory out of
 * @return RAST_OK if succeeded, error otherwise
 * @see rast_db_create_option_create()
 */
rast_error_t *rast_local_db_create(const char *name,
                                   rast_db_create_option_t *options,
                                   apr_pool_t *pool);

/**
 * Please use rast_db_optimize().
 * @see rast_db_optimize()
 */
rast_error_t *rast_local_db_optimize(const char *name,
                                     const rast_db_optimize_option_t *options,
                                     apr_pool_t *pool);

/**
 * Open the specified database.
 * @param db The database opened
 * @param name The name of the database
 * @param flags How to open
 *          - RAST_RDRW
 *            - open for reading and writing
 *          - RAST_RDONLY
 *            - open for reading
 * @param options The options to open the database.
 * @param pool The pool to allocate the memory out of
 * @return RAST_OK if succeeded, error otherwise
 * @see rast_db_open(), rast_db_open_option_create()
 */
rast_error_t *rast_local_db_open(rast_db_t **db, const char *name, int flags,
                                 rast_db_open_option_t *options,
                                 apr_pool_t *pool);

/**
 * Please use rast_db_sync().
 * @see rast_db_sync()
 */
rast_error_t *rast_local_db_sync(rast_db_t *db);

/**
 * Please use rast_db_close().
 * @see rast_db_close()
 */
rast_error_t *rast_local_db_close(rast_db_t *db);

/**
 * Return the byte order of the database.
 * @param db The database
 * @return The byte order of the database
 */
rast_byte_order_e rast_local_db_byte_order(rast_db_t *db);

/**
 * Return the encoding of the database.
 * @param db The database
 * @return The encoding of the database
 */
const char *rast_local_db_encoding(rast_db_t *db);

/**
 * Please use rast_db_properties().
 * @see rast_db_properties()
 */
const rast_property_t *rast_local_db_properties(rast_db_t *db,
                                                int *num_properties);

/**
 * Return the number of registered characters to start sync.
 * @param db The database
 * @return The number of registered characters to start sync
 */
int rast_local_db_sync_threshold_chars(rast_db_t *db);

/**
 * Please use rast_db_register().
 * @see rast_db_register()
 */
rast_error_t *rast_local_db_register(rast_db_t *db,
                                     const char *text, rast_size_t nbytes,
                                     rast_value_t *properties,
                                     rast_doc_id_t *doc_id);

/**
 * Please use rast_db_create_document().
 * @see rast_db_create_document()
 */
rast_error_t *rast_local_db_create_document(rast_db_t *db,
                                            rast_document_t **doc);

/**
 * Please use rast_db_search().
 * @see rast_db_search()
 */
rast_error_t *rast_local_db_search(rast_db_t *db, const char *query,
                                   rast_search_option_t *options,
                                   rast_result_t **result, apr_pool_t *pool);

/**
 * Please use rast_db_delete().
 * @see rast_db_delete()
 */
rast_error_t *rast_local_db_delete(rast_db_t *db, rast_doc_id_t doc_id);

/**
 * Please use rast_db_update().
 * @see rast_db_update()
 */
rast_error_t *rast_local_db_update(rast_db_t *db,
                                   rast_doc_id_t old_doc_id,
                                   const char *text, rast_size_t nbytes,
                                   rast_value_t *properties,
                                   rast_doc_id_t *new_doc_id);

/**
 * Please use rast_db_get_text().
 * @see rast_db_get_text()
 */
rast_error_t *rast_local_db_get_text(rast_db_t *db,
                                     rast_doc_id_t doc_id,
                                     char **s, rast_size_t *nbytes,
                                     apr_pool_t *pool);

/**
 * Compare keys in the property index.
 * @param type The type of the property
 * @param db The Berkeley DB handle
 * @param x The left key to compare
 * @param y The right key to compare
 * @return 0 if x == y, positive value if x > y, negative value otherwise.
 */
int rast_compare_keys(rast_type_e type, DB *db, const DBT *x, const DBT *y);

/** @} */

RAST_EXTERN_C_END

#endif

/* vim: set filetype=c sw=4 expandtab : */
