/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#include <stdlib.h>
#include <stdarg.h>

#include <apr_errno.h>
#include <apr_strings.h>

#include <db.h>

#include "rast/config.h"
#include "rast/rast.h"
#include "rast/error.h"

#ifdef RAST_DEBUG
#undef rast_error_vcreate
#undef rast_error_create
#endif

static const char *
rast_strerror(int code)
{
    switch (code) {
    case RAST_ERROR_NOT_IMPLEMENTED:
        return "not implemented";
    case RAST_ERROR_CLOSED:
        return "already closed";
    case RAST_ERROR_EOF:
        return "unexpected eof";
    case RAST_ERROR_BAD_DB:
        return "bad db";
    case RAST_ERROR_INVALID_ARGUMENT:
        return "invaild argument";
    case RAST_ERROR_INVALID_QUERY:
        return "invaild query";
    case RAST_ERROR_EMPTY_QUERY:
        return "empty query";
    default:
        return "unknown error";
    }
}

rast_error_t *
rast_error_vcreate(rast_error_type_t type, int code,
                   const char *fmt, va_list ap)
{
    apr_pool_t *pool;
    rast_error_t *error;
    char *message;

    pool = rast_pool_create(NULL);
    error = (rast_error_t *) apr_palloc(pool, sizeof(rast_error_t));
    error->pool = pool;
    error->type = type;
    error->code = code;
    if (fmt != NULL) {
        error->message = apr_pvsprintf(pool, fmt, ap);
    }
    else {
        switch (error->type) {
        case RAST_ERROR_TYPE_RAST:
            error->message = rast_strerror(error->code);
            break;
        case RAST_ERROR_TYPE_APR:
            message = (char *) apr_palloc(error->pool, RAST_BUFSIZ);
            apr_strerror(error->code, message, RAST_BUFSIZ);
            error->message = message;
            break;
        case RAST_ERROR_TYPE_BDB:
            error->message = db_strerror(error->code);
            break;
        default:
            error->message = "unknown error";
            break;
        }
    }

    return error;
}

rast_error_t *
rast_error_create(rast_error_type_t type, int code, const char *fmt, ...)
{
    rast_error_t *error;
    va_list ap;

    va_start(ap, fmt);
    error = rast_error_vcreate(type, code, fmt, ap);
    va_end(ap);

    return error;
}

#ifdef RAST_DEBUG
rast_error_t *
rast_error_vcreate_debug(rast_error_type_t type, int code,
                         const char *file, int line,
                         const char *fmt, va_list ap)
{
    rast_error_t *error;

    error = rast_error_vcreate(type, code, fmt, ap);
    error->file = file;
    error->line = line;
    return error;
}

rast_error_t *
rast_error_create_debug(rast_error_type_t type, int code,
                        const char *file, int line, const char *fmt, ...)
{
    rast_error_t *error;
    va_list ap;

    va_start(ap, fmt);
    error = rast_error_vcreate_debug(type, code, file, line, fmt, ap);
    va_end(ap);
    return error;
}

#endif

void
rast_error_destroy(rast_error_t *error)
{
    if (error != RAST_OK) {
        apr_pool_destroy(error->pool);
    }
}

/* vim: set filetype=c sw=4 expandtab : */
