# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"
require "nkf"

require "rast"
require "rast/filter"

module Rast
  module Filter
    load_mime_filter("html.rb")

    class TextHtmlTest < Test::Unit::TestCase
      def test_invoke
        invoke_test_simple
        invoke_test_with_meta_tag
        invoke_test_with_address_tag
        invoke_test_with_link_tag
        invoke_test_with_encoding_convertion
      end

      def invoke_test_simple
        filter = SpoofFilter.new("UTF-8")
        brigade = Brigade.new

        bucket = TransientBucket.new("<html><head><title>天気</title></head>")
        brigade.insert_tail(bucket)
        bucket = TransientBucket.new("<body>本日は晴天なり")
        brigade.insert_tail(bucket)
        bucket = TransientBucket.new("&lt; &gt; &#97; &#61;</body></html>")
        brigade.insert_tail(bucket)
        brigade.insert_tail(EOSBucket.new)
        filter_module = TextHtml.new
        filter_module.invoke(filter, brigade, "text/html")

        buf = ""
        filter.passed[0].brigade.each do |bucket|
          buf.concat(bucket.read)
        end
        assert_equal("本日は晴天なり< > a =", buf)
        assert_equal("text/plain", filter.passed[0].mime_type)
        assert_equal(1, filter.passed.length)
        assert_equal("天気", filter.properties["title"])
        assert_equal(1, filter.properties.length)
      end

      def invoke_test_with_meta_tag
        filter = SpoofFilter.new("UTF-8")
        brigade = Brigade.new
        filename = File.join(MakeVariable::TOP_SRCDIR, "tests", "data",
                             "filter", "ja", "html-utf-8.html")
        File.open(filename) do |f|
          brigade.insert_tail(FileBucket.new(f))
          brigade.insert_tail(EOSBucket.new)
          filter_module = TextHtml.new
          filter_module.invoke(filter, brigade, "text/html")
        end

        buf = ""
        filter.passed[0].brigade.each do |bucket|
          buf.concat(bucket.read)
        end
        assert_equal("\nこれは HTML ファイルです\n\n著者情報2\n\n", buf)
        assert_equal("text/plain", filter.passed[0].mime_type)
        assert_equal(1, filter.passed.length)
        assert_equal("HTML テスト", filter.properties["title"])
        assert_equal("著者情報1 author@example.org 著者情報2",
                     filter.properties["author"])
        assert_equal("ユーザ定義のプロパティ値",
                     filter.properties["user-defined-property"])
        assert_equal(3, filter.properties.length)
      end

      def invoke_test_with_address_tag
        filter = SpoofFilter.new("UTF-8")
        brigade = Brigade.new

        html = ''
        html.concat('<html><head><title>天気</title></head><body>')
        html.concat('本日は晴天なり')
        html.concat('<address>著者</address>')
        html.concat('</body></html>')

        bucket = TransientBucket.new(html)
        brigade.insert_tail(bucket)
        brigade.insert_tail(EOSBucket.new)
        filter_module = TextHtml.new
        filter_module.invoke(filter, brigade, "text/html")
        buf = ""
        filter.passed[0].brigade.each do |bucket|
          buf.concat(bucket.read)
        end
        assert_equal("本日は晴天なり著者", buf)
        assert_equal("text/plain", filter.passed[0].mime_type)
        assert_equal(1, filter.passed.length)
        assert_equal("天気", filter.properties["title"])
        assert_equal("著者", filter.properties["author"])
        assert_equal(2, filter.properties.length)
      end

      def invoke_test_with_link_tag
        filter = SpoofFilter.new("UTF-8")
        brigade = Brigade.new

        html = ''
        html.concat('<html><head><title>天気</title></head>')
        html.concat('<LINK href="prev.html" rel="prev">')
        html.concat('<LINK href="mailto:author@somewhere.org" rev="made">')
        html.concat('<LINK href="next.html" rel="next">')
        html.concat('<body>本日は晴天なり</body></html>')

        bucket = TransientBucket.new(html)
        brigade.insert_tail(bucket)
        brigade.insert_tail(EOSBucket.new)
        filter_module = TextHtml.new
        filter_module.invoke(filter, brigade, "text/html")
        buf = ""
        filter.passed[0].brigade.each do |bucket|
          buf.concat(bucket.read)
        end
        assert_equal("本日は晴天なり", buf)
        assert_equal("text/plain", filter.passed[0].mime_type)
        assert_equal(1, filter.passed.length)
        assert_equal("天気", filter.properties["title"])
        assert_equal("author@somewhere.org", filter.properties["author"])
        assert_equal(2, filter.properties.length)
      end

      def invoke_test_with_encoding_convertion
        filter = SpoofFilter.new("UTF-8")
        brigade = Brigade.new

        filename = File.join(MakeVariable::TOP_SRCDIR, "tests", "data",
                             "filter", "ja", "html-euc-jp.html")
        File.open(filename) do |f|
          brigade.insert_tail(FileBucket.new(f))
          brigade.insert_tail(EOSBucket.new)
          filter_module = TextHtml.new
          filter_module.invoke(filter, brigade, "text/html")
        end

        buf = ""
        filter.passed[0].brigade.each do |bucket|
          buf.concat(bucket.read)
        end
        assert_equal("\nこれは HTML ファイルです\n\n著者情報2\n\n", buf)
        assert_equal("text/plain", filter.passed[0].mime_type)
        assert_equal(1, filter.passed.length)
        assert_equal("HTML テスト", filter.properties["title"])
        assert_equal("著者情報1 author@example.org 著者情報2",
                     filter.properties["author"])
        assert_equal("ユーザ定義のプロパティ値",
                     filter.properties["user-defined-property"])
        assert_equal(3, filter.properties.length)
      end
    end
  end
end
