# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"
require "socket"

require "rast/xmlrpc-server-testable"

require "rast_test"
require "rast_xmlrpc_client"

module Rast
  class XMLRPCClientTest < Test::Unit::TestCase
    include XMLRPCServerTestable

    def test_register_raw
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      kick_xmlrpc_client("--read-write", db_name) do |db|
        assert_equal(1, db.register_raw("本日は晴天なり",
                                        [
                                          "test",
                                          DateTime.new(2005, 4, 15),
                                          10
                                        ]))
      end
    end

    def test_register
      register_test_simple
      register_test_only_property
    end

    def register_test_simple
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      kick_xmlrpc_client("--read-write", db_name) do |db|
        assert_equal(1, db.register("本日は晴天なり", {}))
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(nil, result.items[0].properties)
        assert_equal(1, result.items.length)
      end
    end

    def register_test_only_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified_date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      time = DateTime.new(2005, 2, 17, 19, 4, 38)
      kick_xmlrpc_client("--read-write", db_name) do |db|
        properties = {
          "title" =>"test",
          "modified_date" => time.strftime("%F"),
          "modified" => time.strftime("%FT%T"),
          "serial" => 10,
        }
        assert_equal(1, db.register("", properties))
      end
      LocalDB.open(db_name) do |db|
        result = db.search("serial = 10",
                           {
                             "properties" => [
                               "title",
                               "modified_date",
                               "modified",
                               "serial",
                             ]
                           })
        assert_equal(1, result.items[0].doc_id)
        assert_equal("test", result.items[0].properties[0])
        assert_equal(time.strftime("%F"), result.items[0].properties[1])
        assert_equal(time.strftime("%FT%T"), result.items[0].properties[2])
        assert_equal(10, result.items[0].properties[3])
        assert_equal(4, result.items[0].properties.length)
      end
    end

    def test_search
      search_test_simple
      search_test_summary
      search_test_property_types
      search_test_sort_with_score
      search_test_calc_score
      search_test_calc_received_all_num_docs_score
      search_test_sort_with_property
      search_test_set_range
      search_test_rast_error
    end

    def search_test_simple
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり", {})
        db.register("本日は晴天なり", {})
        db.register("明日は雪っすり", {})
      end
      kick_xmlrpc_client(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.num_indices)
        assert_equal(3, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal(nil, result.items[0].properties)
        assert_equal(1, result.items.length)

        result = db.search("なり")
        assert_equal(2, result.hit_count)
        assert_equal("なり", result.terms[0].term)
        assert_equal(2, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(nil, result.items[0].properties)
        assert_equal(2, result.items[1].doc_id)
        assert_equal(nil, result.items[1].properties)
        assert_equal(2, result.items.length)
      end
    end

    def search_test_summary
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => true,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("0123456789", {"title" => "full text search property"})
        db.register("零一二三四五六七八九",
                    {"title" => "full text search property"})
        db.register("零1二3四5六7八9",
                    {"title" => "full text search property"})
      end
      kick_xmlrpc_client(db_name) do |db|
        search_options = {"need_summary" => true, "summary_nchars" => 6}
        result = db.search("5", search_options)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("345678", result.items[0].summary)
        assert_equal(3, result.items[1].doc_id)
        assert_equal("3四5六7八", result.items[1].summary)
        assert_equal(2, result.items.length)
      end
    end

    def search_test_property_types
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified_date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }

      date = Date.new(2005, 4, 16)
      datetime = DateTime.new(2005, 4, 16, 14, 25, 33)

      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("晴天",
                    {
                      "filename" => "hare.txt",
                      "serial" => 2,
                      "modified_date" => date,
                      "modified" => datetime,
                    })
      end
      kick_xmlrpc_client(db_name) do |db|
        result = db.search("晴天",
                           {
                             "properties" => [
                               "filename", "serial",
                               "modified_date", "modified"
                             ],
                             "need_summary" => true,
                           })
        assert_equal(1, result.hit_count)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("hare.txt", result.items[0].properties[0])
        assert_equal(2, result.items[0].properties[1])
        assert_equal(date.to_s, result.items[0].properties[2])
        assert_equal(datetime.strftime("%FT%T"), result.items[0].properties[3])
        assert_equal(4, result.items[0].properties.length)
        assert_equal(1, result.items.length)
      end
    end

    def search_test_sort_with_score
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("あい" * 99 + "晴天", {}) 
        db.register("晴天" * 100, {})
        db.register("あい" * 49 + "晴天", {}) 
      end
      kick_xmlrpc_client(db_name) do |db|
        # descend
        result = db.search("晴天")
        assert_equal(3, result.hit_count)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(3, result.items[1].doc_id)
        assert_equal(1, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        # ascend
        search_options = {
          "sort_order" => Rast::SORT_ORDER_ASCENDING
        }
        result = db.search("晴天", search_options)
        assert_equal(3, result.hit_count)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(3, result.items[1].doc_id)
        assert_equal(2, result.items[2].doc_id)
        assert_equal(3, result.items.length)
      end
    end

    def search_test_calc_score
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("aaa" * 100, {"filename" => "abc.txt"})
      end
      kick_xmlrpc_client(db_name) do |db|
        options = {
          "score_method" => Rast::SCORE_METHOD_NONE,
        }
        result = db.search("aaa", options)
        assert_equal(0, result.items[0].score)

        options = {
          "score_method" => Rast::SCORE_METHOD_TFIDF,
        }
        result = db.search("aaa", options)
        assert_operator(0, :<, result.items[0].score)

        result = db.search("aaa", {})
        assert_operator(0, :<, result.items[0].score)

        options = {
          "sort_method" => Rast::SORT_METHOD_PROPERTY,
          "sort_property" => "filename"
        }
        result = db.search("aaa", options)
        assert_operator(0, :<, result.items[0].score)

        options = {
          "score_method" => Rast::SCORE_METHOD_TFIDF,
          "sort_method" => Rast::SORT_METHOD_PROPERTY,
          "sort_property" => "filename"
        }
        result = db.search("aaa", options)
        assert_operator(0, :<, result.items[0].score)
      end
    end

    def search_test_calc_received_all_num_docs_score
      options = {
        "encoding" => "utf8",
        "properties" => [
        ],
      }
      db_name1 = generate_db_name
      LocalDB.create(db_name1, options)
      LocalDB.open(db_name1) do |db|
        db.register("qwerty abcdef", {})
        db.register("qwerty", {})
      end
      result1 = LocalDB.open(db_name1) do |db|
        db.search("qwe abc", {})
      end
      db_name2 = generate_db_name
      LocalDB.create(db_name2, options)
      LocalDB.open(db_name2) do |db|
        db.register("qwerty abcdef", {})
      end
      kick_xmlrpc_client(db_name2) do |db|
        terms = result1.terms.collect do |term|
          term.doc_count
        end
        result2 = db.search("qwe abc",
                            {"all_num_docs" => 2, "terms" => terms})
        assert_equal(result1.items[0].score, result2.items[0].score)
      end
    end

    def search_test_sort_with_property
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり",
                    {"filename" => "2.txt", "serial" => 10})
        db.register("本日は晴天なり",
                    {"filename" => "3.txt", "serial" => 5})
        db.register("明日は雪っすり",
                    {"filename" => "1.txt", "serial" => 15})
      end
      kick_xmlrpc_client(db_name) do |db|
        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "filename",
                             "properties" => ["filename"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(3, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(2, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "filename",
                             "sort_order" => Rast::SORT_ORDER_ASCENDING,
                             "properties" => ["filename"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(3, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(2, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "filename",
                             "sort_order" => Rast::SORT_ORDER_DESCENDING,
                             "properties" => ["filename"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(3, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "serial",
                             "properties" => ["serial"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(3, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "serial",
                             "sort_order" => Rast::SORT_ORDER_ASCENDING,
                             "properties" => ["serial"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(3, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "serial",
                             "sort_order" => Rast::SORT_ORDER_DESCENDING,
                             "properties" => ["serial"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(3, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(2, result.items[2].doc_id)
        assert_equal(3, result.items.length)
      end
    end

    def search_test_set_range
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("1", {}) 
        db.register("1", {}) 
        db.register("1", {}) 
        db.register("1", {}) 
        db.register("1", {}) 
      end
      kick_xmlrpc_client(db_name) do |db|
        search_options = {
          "start_no" => 0,
          "num_items" => 3,
        }
        result = db.search("1", search_options)
        assert_equal(5, result.hit_count)
        assert_equal(3, result.items.length)

        search_options = {
          "start_no" => 1,
          "num_items" => 3,
        }
        result = db.search("1", search_options)
        assert_equal(5, result.hit_count)
        assert_equal(3, result.items.length)

        search_options = {
          "start_no" => 4,
          "num_items" => 10,
        }
        result = db.search("1", search_options)
        assert_equal(5, result.hit_count)
        assert_equal(1, result.items.length)

        search_options = {
          "start_no" => 20,
          "num_items" => 10,
        }
        result = db.search("1", search_options)
        assert_equal(5, result.hit_count)
        assert_equal(0, result.items.length)
      end
    end

    def search_test_rast_error
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり", {})
      end
      kick_xmlrpc_client(db_name) do |db|
        assert_raise(RastError) do
          db.search("晴天", {"properties" => ["invalid_property"]})
        end
      end
    end

    def test_delete
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {})
      end
      kick_xmlrpc_client("--read-write", db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.hit_count)
        db.delete(1)
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(0, result.hit_count)
      end
    end

    def test_update
      update_test_simple
      update_test_property
    end

    def update_test_simple
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {})
      end
      kick_xmlrpc_client("--read-write", db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.hit_count)
        assert_equal(2, db.update(1, "明日は雪っす", {}))
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(0, result.hit_count)
        result = db.search("雪っ")
        assert_equal(1, result.hit_count)
      end
    end

    def update_test_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
          },
          {
            "name" => "count",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {"filename" => "a.txt", "count" => 10})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.hit_count)
      end
      kick_xmlrpc_client("--read-write", db_name) do |db|
        assert_equal(2, db.update(1, "明日は雪っす",
                                  {"filename" => "b.txt", "count" => 20}))
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(0, result.hit_count)
        result = db.search("雪っ")
        assert_equal(1, result.hit_count)
        result = db.search("count = 10")
        assert_equal(0, result.hit_count)
        result = db.search("count = 20")
        assert_equal(1, result.hit_count)
      end
    end

    def test_get_text
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      text = "本日は晴天なり"
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register(text, {})
      end
      kick_xmlrpc_client(db_name) do |db|
        assert_equal(text, db.get_text(1))
      end

      options = {
        "encoding" => "utf8",
        "preserve_text" => false,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {})
      end
      LocalDB.open(db_name) do |db|
        assert_equal("", db.get_text(1))
      end
    end

    def test_encoding
      options = {
        "encoding" => "mecab_euc_jp",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      kick_xmlrpc_client(db_name) do |db|
        assert_equal("EUC-JP", db.encoding)
      end
    end
  end
end
