[comment {-*- tcl -*- doctools manpage}]
[manpage_begin blowfish n 1.0.3]
[copyright {2003, Pat Thoyts <patthoyts@users.sourceforge.net>}]
[moddesc   {Blowfish Block Cipher}]
[titledesc {Implementation of the Blowfish block cipher}]
[require Tcl 8.4]
[require blowfish [opt 1.0.4]]
[description]
[para]

This package is an implementation in Tcl of the Blowfish algorithm
developed by Bruce Schneier [lb]1[rb]. Blowfish is a 64-bit block cipher
designed to operate quickly on 32 bit architectures and accepting a
variable key length. This implementation supports ECB and CBC mode
blowfish encryption.

[section {COMMANDS}]

[list_begin definitions]

[call [cmd "::blowfish::blowfish"] \
        [opt [arg "-mode [lb]ecb|cbc[rb]"]] \
        [opt [arg "-dir [lb]encrypt|decrypt[rb]"]] \
        [arg "-key keydata"] \
        [opt [arg "-iv vector"]] \
        [opt [arg "-out channel"]] \
        [opt [arg "-chunksize size"]] \
        [opt [arg "-pad padchar"]] \
        [lb] [arg "-in channel"] | \
        [opt [arg "--"]] [arg "data"] [rb]]

Perform the [package blowfish] algorithm on either the data provided
by the argument or on the data read from the [arg "-in"] channel. If
an [arg "-out"] channel is given then the result will be written to
this channel.

[para]

The [arg -key] option must be given. This parameter takes a binary
string of variable length and is used to generate the [package blowfish]
key schedule. You should be aware that creating a key
schedule is quite an expensive operation in blowfish so it is worth
reusing the key where possible. See [cmd Reset].

[para]

The [arg -mode] and [arg -dir] options are optional and default to cbc
mode and encrypt respectively. The initialization vector [arg -iv]
takes an 8 byte binary argument which defaults to 8 zeros.
See [sectref "MODES OF OPERATION"] for more about available modes and
their uses.

[para]

Blowfish is a 64-bit block cipher. This means that the data must be
provided in units that are a multiple of 8 bytes. The [cmd blowfish]
command will by default add nul characters to pad the input data to a
multiple of 8 bytes if necessary. The programming api commands will
never add padding and instead will raise an error if the input is not
a multiple of the block size. The [arg -pad] option can be used to
change the padding character or to disable padding if the empty string
is provided as the argument.

[list_end]

[section "PROGRAMMING INTERFACE"]

[list_begin definitions]

[call [cmd "::blowfish::Init"] [arg "mode"] [arg "keydata"] [arg "iv"]]

Construct a new blowfish key schedule using the specified key data and
the given initialization vector. The initialization vector is not used
with ECB mode but is important for CBC mode.
See [sectref "MODES OF OPERATION"] for details about cipher modes.

[call [cmd "::blowfish::Encrypt"] [arg "Key"] [arg "data"]]

Use a prepared key acquired by calling [cmd Init] to encrypt the
provided data. The data argument should be a binary array that is a
multiple of the block size of 8 bytes. The result is a binary
array the same size as the input of encrypted data.

[call [cmd "::blowfish::Decrypt"] [arg "Key"] [arg "data"]]

Decipher data using the key. Note that the same key may be used to
encrypt and decrypt data provided that the initialization vector is
reset appropriately for CBC mode.

[call [cmd "::blowfish::Reset"] [arg "Key"] [arg "iv"]]

Reset the initialization vector. This permits the programmer to re-use
a key and avoid the cost of re-generating the key schedule where the
same key data is being used multiple times.

[call [cmd "::blowfish::Final"] [arg "Key"]]

This should be called to clean up resources associated with [arg Key].
Once this function has been called the key may not be used again.

[list_end]


[section "MODES OF OPERATION"]

[list_begin definitions]

[def "Electronic Code Book (ECB)"]
ECB is the basic mode of all block ciphers. Each block is encrypted
independently and so identical plain text will produce identical
output when encrypted with the same key. Any encryption errors will
only affect a single block however this is vulnerable to known
plaintext attacks.

[def "Cipher Block Chaining (CBC)"]

CBC mode uses the output of the last block encryption to affect the
current block. An initialization vector of the same size as the cipher
block size is used to handle the first block. The initialization
vector should be chosen randomly and transmitted as the first block of
the output. Errors in encryption affect the current block and the next
block after which the cipher will correct itself. CBC is the most
commonly used mode in software encryption.

[list_end]

[section "EXAMPLES"]

[example {
% blowfish::blowfish -hex -mode ecb -dir encrypt -key secret01 "hello, world!"
d0d8f27e7a374b9e2dbd9938dd04195a
}]

[example {
 set Key [blowfish::Init cbc $eight_bytes_key_data $eight_byte_iv]
 append ciphertext [blowfish::Encrypt $Key $plaintext]
 append ciphertext [blowfish::Encrypt $Key $additional_plaintext]
 blowfish::Final $Key
}]

[section "REFERENCES"]

[list_begin enumerated]

[enum]
    Schneier, B. "Applied Cryptography, 2nd edition", 1996,
    ISBN 0-471-11709-9, pub. John Wiley & Sons.

[list_end]

[see_also des 3des rc4]

[section AUTHORS]
Frank Pilhofer, Pat Thoyts

[section {BUGS, IDEAS, FEEDBACK}]

This document, and the package it describes, will undoubtedly contain
bugs and other problems.

Please report such in the category [emph blowfish] of the
[uri {http://sourceforge.net/tracker/?group_id=12883} {Tcllib SF Trackers}].

Please also report any ideas for enhancements you may have for either
package and/or documentation.


[keywords blowfish {block cipher} security encryption cryptography]
[manpage_end]
