// The original version of this code (revision 2 of the Subversion
// archive) was released subject to the following license:

//   Copyright (c) 2006, Sun Microsystems, Inc.  All rights reserved.
//   Redistribution and use in source and binary forms, with or
//   without modification, are permitted provided that the following
//   conditions are met:

//   * Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above
//     copyright notice, this list of conditions and the following
//     disclaimer in the documentation and/or other materials provided
//     with the distribution.
//   * Neither the name of Sun Microsystems or the names of
//     contributors may be used to endorse or promote products derived
//     from this software without specific prior written permission.
 
//   THIS SOFTWARE IS PROVIDED BY SUN AND ITS LICENSORS ``AS IS'' AND
//   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
//   THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
//   PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SUN OR ITS
//   LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
//   AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
//   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//   ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
//   POSSIBILITY OF SUCH DAMAGE.

// Subsequent additions and modifications are Copyright (c) 2006 David
// Carlton, and may be used subject to the same conditions.

#include "SigHandler.hpp"

#include <iostream>
#include <stdlib.h>

#include "Private.hpp"

namespace UnitTest {

  namespace Private {

    SigHandler::SigHandler(std::ostream *stream)
      : stream_(stream), oldHandler_(handler_), jumpBuf_(0),
	oldSegvAction_(), oldFpeAction_(), oldAbrtAction_(),
	sigNum_(0) {
      handler_ = this;
    }

    SigHandler::~SigHandler() {
      handler_ = oldHandler_;
      sigaction(SIGSEGV, &oldSegvAction_, 0);
      sigaction(SIGFPE, &oldFpeAction_, 0);
      sigaction(SIGABRT, &oldAbrtAction_, 0);
    }

    void SigHandler::setUp(sigjmp_buf *buf) {
      jumpBuf_ = buf;

      struct sigaction newAction;
      newAction.sa_handler = handler;
      newAction.sa_mask = sigset_t();
      newAction.sa_flags = SA_RESETHAND;

      sigaction(SIGSEGV, &newAction, &oldSegvAction_);
      sigaction(SIGFPE, &newAction, &oldFpeAction_);
      sigaction(SIGABRT, &newAction, &oldAbrtAction_);
    }

    // Return the name of the last signal.
    const char *SigHandler::sigName() const {
      switch (sigNum_) {
      case SIGSEGV:
	return "Seg fault";
      case SIGFPE:
	return "Division by zero";
      case SIGABRT:
	return "Abort signal";
      default:
	return "Unknown signal";
      }
    }

    // NOTE (2003-08-12, carlton): Does this need to be extern "C"?  I
    // hope not; it seems to work fine.

    // NOTE (2003-12-24, carlton): Subsequent experimentation seems
    // to suggest that you can't declare members of a class to be
    // extern "C".  (Or, presumably, extern at all.)  So if I really
    // wanted to be safe, I would probably move this up to namespace
    // scope.
    void SigHandler::handler(int signum) {
      dumpStack();

      if (handler_ && handler_->jumpBuf_) {
	handler_->sigNum_ = signum;
	siglongjmp(*handler_->jumpBuf_, 1);
      }
    }

    void SigHandler::dumpStack() {
#ifdef HAVE_EXECINFO_H
      void *ptrs[12];

      int numPtrs = backtrace(ptrs, 11);
      char **trace = backtrace_symbols(ptrs, numPtrs);

      std::ostream *stream = handler_ ? handler_->stream_ : &std::cout;

      if (trace) {
	// FIXME (2006-06-09, carlton): This isn't getting tested.

	*stream << "Stack trace:" << std::endl;

	for (int i = 0; i < numPtrs; i++)
	  *stream << trace[i] << std::endl;

	*stream << "Stack ends!" << std::endl;

	free(trace);
      } else {
	*stream << "Couldn't get memory to dump stack" << std::endl;
      }
#endif
    }

    SigHandler *SigHandler::handler_ = 0;

  }

}
