// The original version of this code (revision 2 of the Subversion
// archive) was released subject to the following license:

//   Copyright (c) 2006, Sun Microsystems, Inc.  All rights reserved.
//   Redistribution and use in source and binary forms, with or
//   without modification, are permitted provided that the following
//   conditions are met:

//   * Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above
//     copyright notice, this list of conditions and the following
//     disclaimer in the documentation and/or other materials provided
//     with the distribution.
//   * Neither the name of Sun Microsystems or the names of
//     contributors may be used to endorse or promote products derived
//     from this software without specific prior written permission.
 
//   THIS SOFTWARE IS PROVIDED BY SUN AND ITS LICENSORS ``AS IS'' AND
//   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
//   THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
//   PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SUN OR ITS
//   LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
//   AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
//   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//   ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
//   POSSIBILITY OF SUCH DAMAGE.

// Subsequent additions and modifications are Copyright (c) 2006 David
// Carlton, and may be used subject to the same conditions.

#include <UnitTest.hpp>

#include <ExistingBase.hpp>

#include <iostream>

#include <signal.h>

#include "../lib/Private.hpp"
#include "unittesttest.hpp"
#include "Helper.hpp"

namespace SuiteTest {

  using namespace UnitTest;

  class Base : public HelperBase {
  protected:
    Base() {
      suite_.stream(stream_);
    }

    void add(Test *test) {
      suite_.add(TestPtr(test));
    }

    void runSuite() {
      suite_.run();
    }

    std::string output() {
      return stream_.str();
    }

    void assertOutputContains(const char *substring) {
      if (!outputContains(substring)) {
	std::cout << "Actual output: " << output()
		  << "; looking for substring: " << substring
		  << std::endl;

	ASSERT(false);
      }
    }

    bool outputContains(const char *substring) {
      return output().find(substring) != std::string::npos;
    }

    Suite &suite() {
      return suite_;
    }

    int passes() {
      return suite_.passes();
    }

    int failures() {
      return suite_.failures();
    }

  private:
    Suite suite_;
    std::ostringstream stream_;
  };

  // Test that Suite runs tests in the correct order, calling
  // setUp() and tearDown() as appropriate.
  class Order : public Base {
  public:
    void run() {
      addTest("1");
      addTest("2");

      runSuite();

      ASSERT_EQUALS("1:setup;1:run;1:teardown;2:setup;2:run;2:teardown;",
		    status_);
    }

  private:
    void addTest(const char *name) {
      Helper *test = new Helper();

      test->statusString(&status_, name);

      add(test);
    }

    std::string status_;
  };

  // Check that passes and fails of leaf classes get counted correctly.
  class PassFailLeaf : public Base {
  public:
    void run() {
      addTest(true);
      addTest(false);
      addTest(false);
      addTest(true);
      addTest(false);

      runSuite();

      ASSERT_EQUALS(2, passes());
      ASSERT_EQUALS(3, failures());
    }

  private:
    void addTest(bool pass) {
      Helper *test = new Helper();

      test->shouldFail(!pass);

      add(test);
    }
  };

  // Check that passes and failures of suite members get added correctly.
  class PassFailSuite : public Base {
  public:
    void run() {
      addTest(3, 5, true);
      addTest(10, 20, false);
      addTest(500, 700, true);

      runSuite();

      ASSERT_EQUALS(513 + 2, passes());
      ASSERT_EQUALS(725 + 1, failures());
    }

  private:
    // Add a test that reports the given number of passes and
    // failures, and that fails itself if PASS is false.
    void addTest(int passes, unsigned int failures, bool pass) {
      Helper *test = new Helper();

      test->passes(passes);
      test->failures(failures);
      test->shouldFail(!pass);

      add(test);
    }
  };

  // Check that output gets sent to the appropriate stream.
  class Stream : public Base {
  public:
    void run() {
      add(new Helper());

      Private::Debug guard(false);

      runSuite();

      // If this next assertion fails, be careful when reading the
      // output: the failure message will probably include a period
      // of its own.
      ASSERT_EQUALS(".", output());
    }

  private:
    class Helper : public Test {
    protected:
      void run() {
      }
    };
  };

  // Make sure that even error output gets sent to the right place.
  class StreamError : public Base {
  public:
    void run() {
      add(new Helper());

      runSuite();

      assertOutputContains("Assertion failure");
      assertOutputContains("StreamError::Helper\n");
    }

  private:
    class Helper : public Test {
    protected:
      void run() {
	ASSERT(false);
      }
    };
  };

  class SetUpAssertionFailure : public Base {
  public:
    void run() {
      add(new Helper());

      runSuite();

      assertOutputContains("Assertion failure");
    }

  private:
    class Helper : public Test {
    protected:
      void setUp() {
	ASSERT(false);
      }

      void run() {}
    };
  };

  // An std::exception subclass with a custom logRep implementation.
  class CustomException : public std::exception {
  public:
    const char *what() const throw() {
      return "custom exception";
    }
  };

  // Make sure that we catch exceptions in setup.
  class SetUpException : public Base {
  public:
    void run() {
      add(new Helper(false));
      add(new Helper(true));
      add(new Helper(false));

      runSuite();

      ASSERT_EQUALS(2, passes());
      ASSERT_EQUALS(1, failures());
      assertOutputContains("custom exception");
    }

  private:
    class Helper : public Test {
    public:
      Helper(bool fail)
	: fail_(fail) {
      }

    protected:
      void setUp() {
	if (fail_)
	  throw CustomException();
      }

      void run() {
      }

    private:
      bool fail_;
    };
  };

  // Make sure that we catch exceptions while running.
  class RunException : public Base {
  public:
    void run() {
      add(new Helper(false));
      add(new Helper(true));
      add(new Helper(false));

      runSuite();

      ASSERT_EQUALS(2, passes());
      ASSERT_EQUALS(1, failures());
      assertOutputContains("custom exception");
    }

  private:
    class Helper : public Test {
    public:
      Helper(bool fail)
	: fail_(fail) {
      }

    protected:
      void run() {
	if (fail_)
	  throw CustomException();
      }

    private:
      bool fail_;
    };
  };

  // Make sure that we catch exceptions in tearDown.
  class TearDownException : public Base {
  public:
    void run() {
      add(new Helper(false));
      add(new Helper(true));
      add(new Helper(false));

      runSuite();

      ASSERT_EQUALS(2, passes());
      ASSERT_EQUALS(1, failures());
      assertOutputContains("custom exception");
    }

  private:
    class Helper : public Test {
    public:
      Helper(bool fail)
	: fail_(fail) {
      }

    protected:
      void run() {
      }

      void tearDown() {
	if (fail_)
	  throw CustomException();
      }

    private:
      bool fail_;
    };
  };

  // Make sure that we catch seg faults in setup.
  class SetUpSignal : public Base {
  public:
    void run() {
      Helper *faulter = new Helper(true);

      add(faulter);
      add(new Helper(false));

      runSuite();

      ASSERT_EQUALS(1, passes());
      ASSERT_EQUALS(1, failures());
      assertOutputContains("Seg fault when setting up");
      ASSERT(!faulter->ran());
    }

  private:
    class Helper : public Test {
    public:
      Helper(bool fault)
	: fault_(fault), ran_(false) {
      }

      bool ran() const {
	return ran_;
      }

    protected:
      void setUp() {
	if (fault_)
	  raise(SIGSEGV);
      }

      void run() {
	ran_ = true;
      }

      void tearDown() {
	ran_ = true;
      }

    private:
      bool fault_;
      bool ran_;
    };
  };

  // Make sure that we catch seg faults in run.
  class RunSignal : public Base {
  public:
    void run() {
      Helper *faulter = new Helper(true);

      add(faulter);
      add(new Helper(false));

      runSuite();

      ASSERT_EQUALS(1, passes());
      ASSERT_EQUALS(1, failures());
      assertOutputContains("Seg fault when running");
      ASSERT(faulter->toreDown());
    }

  private:
    class Helper : public Test {
    public:
      Helper(bool fault)
	: fault_(fault), toreDown_(false) {
      }

      bool toreDown() const {
	return toreDown_;
      }

    protected:
      void run() {
	if (fault_)
	  raise(SIGSEGV);
      }

      void tearDown() {
	toreDown_ = true;
      }

    private:
      bool fault_;
      bool toreDown_;
    };
  };

  // Make sure that we catch seg faults in tearDown.
  class TearDownSignal : public Base {
  public:
    void run() {
      add(new Helper(true));
      add(new Helper(false));

      runSuite();

      ASSERT_EQUALS(1, passes());
      ASSERT_EQUALS(1, failures());
      assertOutputContains("Seg fault when tearing down");
    }

  private:
    class Helper : public Test {
    public:
      Helper(bool fault)
	: fault_(fault) {
      }

    protected:
      void run() {
      }

      void tearDown() {
	if (fault_)
	  raise(SIGSEGV);
      }

    private:
      bool fault_;
    };
  };

  // Make sure we catch division by 0; we'll just test it during run,
  // because that combined with the segv tests cover all code paths.
  class DivideByZero : public Base {
  public:
    void run() {
      add(new Helper(true));
      add(new Helper(false));

      runSuite();

      ASSERT_EQUALS(1, passes());
      ASSERT_EQUALS(1, failures());
      assertOutputContains("Division by zero when tearing down");
    }

  private:
    class Helper : public Test {
    public:
      Helper(bool fault)
	: fault_(fault) {
      }

    protected:
      void run() {
      }

      void tearDown() {
	if (fault_)
	  raise(SIGFPE);
      }

    private:
      bool fault_;
    };
  };

  // Make sure that we catch aborts, too.
  class Abort : public Base {
  public:
    void run() {
      add(new Helper(true));
      add(new Helper(false));

      runSuite();

      ASSERT_EQUALS(1, passes());
      ASSERT_EQUALS(1, failures());
      assertOutputContains("Abort signal when tearing down");
    }

  private:
    class Helper : public Test {
    public:
      Helper(bool fault)
	: fault_(fault) {
      }

    protected:
      void run() {
      }

      void tearDown() {
	if (fault_) {
	  raise(SIGABRT);
	}
      }

    private:
      bool fault_;
    };
  };

  // Test our signal handler for signals outside of the tests
  // themselves.  (See Bug 3395.)
  class BadAssertion : public Base {
  public:
    void run() {
      Helper *first(new Helper());
      Helper *second(new BadHelper());
      Helper *third(new Helper());

      add(first);
      add(second);
      add(third);

      runSuite();

      assertOutputContains("Fatal error when running tests.\n"
			   "Error probably occurred near test: "
			   "SuiteTest::BadAssertion::"
			   "BadHelper\n");
      ASSERT_EQUALS(1, passes());
      ASSERT(failures() >= 1);

      ASSERT(first->ran());
      ASSERT(second->ran());
      ASSERT(!third->ran());
    }

  private:
    class Helper : public Test {
    public:
      Helper() : ran_(false) {}

      bool ran() const { return ran_; }

    protected:
      void run() { ran_ = true; }

    private:
      bool ran_;
    };

    class BadHelper : public Helper {
    protected:
      void run() {
	Helper::run();

	throw BadAssertion();
      }

    private:
      class BadAssertion : public AssertionFailure {
	void printMe(std::ostream &stream, const std::string &testName)
	  const {
	  raise(SIGSEGV);
	}
      };
    };
  };

  class TemplatedNewTest : public Base {
  public:
    void run() {
      suite().add<FailHelper>();
      suite().add<PassHelper>();
      suite().add<FailHelper>();

      Private::Debug guard(false);

      runSuite();

      ASSERT_EQUALS(1, passes());
      ASSERT_EQUALS(2, failures());

      assertOutputContains("SuiteTest::TemplatedNewTest::FailHelper");
      ASSERT(!outputContains("SuiteTest::TemplatedNewTest::PassHelper"));
    }

  private:
    class FailHelper {
    public:
      void run() {
	ASSERT(false);
      }
    };

    class PassHelper {
    public:
      void run() {
	ASSERT(true);
      }
    };
  };

  class TemplatedNewTestArgs : public Base {
  public:
    void run() {
      suite().add<Helper>(true);
      suite().add<Helper>(false);
      suite().add<Helper>(true);

      runSuite();

      ASSERT_EQUALS(2, passes());
      ASSERT_EQUALS(1, failures());
    }

  private:
    class Helper {
    public:
      explicit Helper(bool toAssert) : toAssert_(toAssert) {}

      void run() {
	ASSERT(toAssert_);
      }

    private:
      const bool toAssert_;
    };
  };

  class TemplatedNewTestManyArgs : public Base {
  public:
    void run() {
      suite().add<Helper>(1, 2);
      suite().add<Helper>(3, 4, 5);
      suite().add<Helper>(6, 7, 8, 9);

      runSuite();

      ASSERT_EQUALS("1 2 0 0\n3 4 5 0\n6 7 8 9\n", args_.str());
    }

  private:
    class Helper {
    public:
      explicit Helper(int first, int second, int third = 0, int fourth = 0) {
	args_ << first << ' ' << second << ' '
	      << third << ' ' << fourth << '\n';
      }

      void run() {
      }
    };

    static std::ostringstream args_;
  };

  std::ostringstream TemplatedNewTestManyArgs::args_;

  // class RunOtherThreadFailure : public Base {
  //   public:
  //     void run() {
  //         suite().add<Helper>(true);
  //         suite().add<Helper>(false);
  //         suite().add<Helper>(true);

  //         runSuite();

  //         ASSERT_EQUALS(1, passes());
  //         ASSERT_EQUALS(2, failures());

  //         assertOutputContains
  //             ("Failure in test "
  //               "SuiteTest::RunOtherThreadFailure::Helper");
  //         assertOutputContains
  //             ("within function "
  //               "virtual void SuiteTest::RunOtherThreadFailure::Helper::"
  //               "FailingNotifiee::onEvent()");
  //         assertOutputContains("Helper failed!");
  //     }

  //   private:
  //     class Helper {
  //       public:
  //         Helper(bool shouldFail) : shouldFail_(shouldFail) {}

  //         void run() {
  //             FailingNotifiee activity(shouldFail_);

  //             activity.nextTime(0);

  //             NEW_WAIT_FOR_ASSERTION(activity.hasRun());
  //         }

  //       private:
  //         class FailingNotifiee : public ActivityNotifiee {
  //           public:
  //             FailingNotifiee(bool shouldFail)
  //                     : ActivityNotifiee("FailingNotifiee"),
  //                       shouldFail_(shouldFail),
  //                       hasRun_(false) {
  //             }

  //             void onEvent() {
  //                 if (shouldFail_)
  //                     FAIL("Helper failed!");

  //                 hasRun_ = true;
  //             }

  //             bool hasRun() const { return hasRun_; }

  //           private:
  //             const bool shouldFail_;
  //             bool hasRun_;
  //         };

  //         const bool shouldFail_;
  //     };
  // };

  class SetUpFailure : public Base {
  public:
    void run() {
      suite().add<Helper>(true);
      suite().add<Helper>(false);
      suite().add<Helper>(true);

      runSuite();

      ASSERT_EQUALS(1, passes());
      ASSERT_EQUALS(2, failures());

      assertOutputContains("Failure in test "
			   "SuiteTest::SetUpFailure::Helper");
      assertOutputContains("Helper failed!");

      ASSERT_EQUALS(3, Helper::setUpsFinished());
    }

  private:
    class Helper {
    public:
      Helper(bool shouldFail) {
	if (shouldFail)
	  FAIL("Helper failed!");

	++setUpsFinished_;
      }

      void run() {}

      static int setUpsFinished() { return setUpsFinished_; }

    private:
      static int setUpsFinished_;
    };
  };

  int SetUpFailure::Helper::setUpsFinished_ = 0;

  class RunFailure : public Base {
  public:
    void run() {
      suite().add<Helper>(true);
      suite().add<Helper>(false);
      suite().add<Helper>(true);

      runSuite();

      ASSERT_EQUALS(1, passes());
      ASSERT_EQUALS(2, failures());

      assertOutputContains("Failure in test "
			   "SuiteTest::RunFailure::Helper");
      assertOutputContains("Helper failed!");

      ASSERT_EQUALS(3, Helper::runsFinished());
    }

  private:
    class Helper {
    public:
      Helper(bool shouldFail) : shouldFail_(shouldFail) {}

      void run() {
	if (shouldFail_)
	  FAIL("Helper failed!");

	++runsFinished_;
      }

      static int runsFinished() { return runsFinished_; }

    private:
      static int runsFinished_;

      const bool shouldFail_;
    };
  };

  int RunFailure::Helper::runsFinished_ = 0;

  class TearDownFailure : public Base {
  public:
    void run() {
      suite().add<Helper>(true);
      suite().add<Helper>(false);
      suite().add<Helper>(true);

      runSuite();

      ASSERT_EQUALS(1, passes());
      ASSERT_EQUALS(2, failures());

      assertOutputContains("Failure in test "
			   "SuiteTest::TearDownFailure::Helper");
      assertOutputContains("Helper failed!");

      ASSERT_EQUALS(3, Helper::testsFinished());
    }

  private:
    class Helper {
    public:
      Helper(bool shouldFail) : shouldFail_(shouldFail) {}

      ~Helper() {
	if (shouldFail_)
	  FAIL("Helper failed!");

	++testsFinished_;
      }

      void run() {}

      static int testsFinished() { return testsFinished_; }

    private:
      static int testsFinished_;

      const bool shouldFail_;
    };
  };

  int TearDownFailure::Helper::testsFinished_ = 0;

  class RunFailureNoDot : public Base {
  public:
    void run() {
      suite().add<Helper>();

      runSuite();

      assertOutputContains("\nFailure in test");
      ASSERT(!outputContains(".\n"));
    }

  private:
    class Helper {
    public:
      void run() {
	FAIL("oops");
      }
    };
  };

  // class DebugSignalBacktrace : public LogBase, public Base {
  //   public:
  //     void run() {
  //         bool oldDebug = Private::debug;
  //         Private::debug = true;

  //         add(new Helper());

  //         runSuite();

  //         ASSERT(numLogs());

  //         Private::debug = oldDebug;
  //     }

  //   private:
  //     class Helper : public Test {
  //       protected:
  //         void run() {
  //             raise(SIGSEGV);
  //         }
  //     };

  //     bool oldDebug_;
  // };

  class All : public Suite {
  public:
    All() {
      add<Order>();
      add<PassFailLeaf>();
      add<PassFailSuite>();
      add<Stream>();
      add<StreamError>();
      add<SetUpAssertionFailure>();
      add<SetUpException>();
      add<RunException>();
      add<TearDownException>();
      add<SetUpSignal>();
      add<RunSignal>();
      add<TearDownSignal>();
      add<DivideByZero>();
      add<Abort>();
      add<BadAssertion>();
      add<TemplatedNewTest>();
      add<TemplatedNewTestArgs>();
      add<TemplatedNewTestManyArgs>();
      //         add<RunOtherThreadFailure>();
      add<SetUpFailure>();
      add<RunFailure>();
      add<TearDownFailure>();
      add<RunFailureNoDot>();
      //         add<DebugSignalBacktrace>();
    }
  };

}

UnitTest::TestPtr suiteTests() {
  return UnitTest::createSuite<SuiteTest::All >();
}
