// The original version of this code (revision 2 of the Subversion
// archive) was released subject to the following license:

//   Copyright (c) 2006, Sun Microsystems, Inc.  All rights reserved.
//   Redistribution and use in source and binary forms, with or
//   without modification, are permitted provided that the following
//   conditions are met:

//   * Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above
//     copyright notice, this list of conditions and the following
//     disclaimer in the documentation and/or other materials provided
//     with the distribution.
//   * Neither the name of Sun Microsystems or the names of
//     contributors may be used to endorse or promote products derived
//     from this software without specific prior written permission.
 
//   THIS SOFTWARE IS PROVIDED BY SUN AND ITS LICENSORS ``AS IS'' AND
//   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
//   THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
//   PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SUN OR ITS
//   LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
//   AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
//   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//   ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
//   POSSIBILITY OF SUCH DAMAGE.

// Subsequent additions and modifications are Copyright (c) 2006 David
// Carlton, and may be used subject to the same conditions.

#include <UnitTest.hpp>

#include <stdexcept>

#include "unittesttest.hpp"

using UnitTest::Private::TestHolder;

namespace TestHolderTest {

  class Base {};

  class Name : public Base {
  public:
    void run() {
      TestHolder<Helper> holder;

      ASSERT_EQUALS("TestHolderTest::Name::Helper", holder.name());
    }

  private:
    class Helper {
    public:
      void run() {}
    };
  };

  class Stages : public Base {
  public:
    void run() {
      TestHolder<Helper> holder;

      ASSERT(!Helper::created());

      holder.setUp();

      ASSERT(Helper::created());
      ASSERT(!Helper::ran());

      holder.run();
      ASSERT(Helper::ran());
      ASSERT(!Helper::destroyed());

      holder.tearDown();
      ASSERT(Helper::destroyed());
    }

    class Helper {
    public:
      Helper() { created_ = true; }
      ~Helper() { destroyed_ = true; }

      void run() { ran_ = true; }

      static bool created() { return created_; }
      static bool ran() { return ran_; }
      static bool destroyed() { return destroyed_; }

    private:
      static bool created_;
      static bool ran_;
      static bool destroyed_;
    };
  };

  bool Stages::Helper::created_ = false;
  bool Stages::Helper::ran_ = false;
  bool Stages::Helper::destroyed_ = false;

  class Assertions : public Base {
  public:
    void run() {
      TestHolder<PassHelper> passHolder;

      passHolder.setUp();
      passHolder.run();
      passHolder.tearDown();

      TestHolder<FailHelper> failHolder;

      failHolder.setUp();
      bool success = false;
      try {
	failHolder.run();
      } catch (const UnitTest::AssertionFailure &) {
	success = true;
      }
      failHolder.tearDown();

      ASSERT(success);
    }

  private:
    class PassHelper {
    public:
      void run() {
	ASSERT(true);
      }
    };

    class FailHelper {
    public:
      void run() {
	ASSERT(false);
      }
    };
  };

  // Make sure that we can throw some exceptions on tearDown.  (If we
  // throw too many exceptions, though, the program will terminate.)
  // This also makes sure that the destructor isn't called twice...
  class ExceptionOnTearDown : public Base {
  public:
    void run() {
      TestHolder<Helper> holder;

      holder.setUp();
      holder.run();

      ASSERT_EXCEPTION(holder.tearDown(), std::runtime_error);
    }

  private:
    class Helper {
    public:
      void run() {}

      ~Helper() { throw std::runtime_error("Helper destructor"); }
    };
  };

  class Args : public Base {
  public:
    void run() {
      int val = 0;
      TestHolder<Helper, int &> holder(val);
      holder.setUp();
      holder.run();
      holder.tearDown();

      ASSERT_EQUALS(17, val);

      TestHolder<Helper, int &, int> holder2(val, 12);
      holder2.setUp();
      holder2.run();
      holder2.tearDown();

      ASSERT_EQUALS(12, val);
    }

  private:
    class Helper {
    public:
      Helper(int &toSet, int val = 17) : toSet_(toSet), val_(val) {}

      void run() {
	toSet_ = val_;
      }

    private:
      int &toSet_;
      int val_;
    };
  };

  class All : public UnitTest::Suite {
  public:
    All() {
      add<Name>();
      add<Stages>();
      add<Assertions>();
      add<ExceptionOnTearDown>();
      add<Args>();
    }
  };

}

UnitTest::TestPtr testHolderTests() {
  return UnitTest::createSuite<TestHolderTest::All>();
}
