// The original version of this code (revision 2 of the Subversion
// archive) was released subject to the following license:

//   Copyright (c) 2006, Sun Microsystems, Inc.  All rights reserved.
//   Redistribution and use in source and binary forms, with or
//   without modification, are permitted provided that the following
//   conditions are met:

//   * Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above
//     copyright notice, this list of conditions and the following
//     disclaimer in the documentation and/or other materials provided
//     with the distribution.
//   * Neither the name of Sun Microsystems or the names of
//     contributors may be used to endorse or promote products derived
//     from this software without specific prior written permission.
 
//   THIS SOFTWARE IS PROVIDED BY SUN AND ITS LICENSORS ``AS IS'' AND
//   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
//   THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
//   PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SUN OR ITS
//   LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
//   AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
//   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//   ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
//   POSSIBILITY OF SUCH DAMAGE.

// Subsequent additions and modifications are Copyright (c) 2006 David
// Carlton, and may be used subject to the same conditions.

#include <UnitTest.hpp>

#include <sstream>
#include <stdexcept>

#include "unittesttest.hpp"

namespace TestTest {
  using namespace UnitTest;

  // A helper class for testing ASSERT_EQUALS: it supports != and
  // output via <<, and can be implicitly constructed from ints.

  class CompareHelper {
  public:
    CompareHelper(int data)
      : data_(data) {
    }

    int data() const {
      return data_;
    }

    bool operator!=(const CompareHelper &other) const {
      return data_ != other.data_;
    }

  private:
    const int data_;
  };

  std::ostream &
  operator<<(std::ostream &stream, const CompareHelper &helper) {
    stream << "CompareHelper[" << helper.data() << "]";

    return stream;
  }

  // A base class to make it easier to test ASSERT and variants.
  class AssertBase {
  public:
    virtual ~AssertBase() {}

    void run() {
      compareValid();

      try {
	compareInvalid();
      } catch (const AssertionFailure &expected) {
	lookForString(expected);
	return;
      }

      // If we reach here, the above 'catch' failed.
      ASSERT(false);
    }

  protected:
    // If you want to do some assertions that should work, stick it
    // here.
    virtual void compareValid() {}

    // If you want to do an assertion that should fail, stick it here.
    virtual void compareInvalid() { ASSERT(false); }

    // If you want to look at the message given by a failed assertion,
    // return the string you're looking for here.
    virtual std::string expectedString() const { return ""; }

    void lookForString(const AssertionFailure &expected) {
      std::ostringstream output;

      expected.printMe(output, name());

      ASSERT(output.str().find(expectedString()) != std::string::npos);
    }

  private:
    std::string name() const {
      return Private::demangledName(typeid(*this));
    }
  };

  // Test ASSERT.
  class Assert : public AssertBase {
  protected:
    void compareValid() {
      ASSERT(true);
    }

    void compareInvalid() {
      ASSERT(false);
    }
  };

  // Next come tests for ASSERT_EQUALS for various kinds of arguments.

  // Test ASSERT_EQUALS for numbers.
  class AssertEquals : public AssertBase {
  protected:
    void compareValid() {
      ASSERT_EQUALS(1, 1);
    }

    void compareInvalid() {
      ASSERT_EQUALS(1, 2);
    }

    std::string expectedString() const {
      return "Expected 1, got 2.";
    }
  };

  // Test ASSERT_EQUALS for pointers.
  class AssertEqualsPtr : public AssertBase  {
  protected:
    void compareValid() {
      int c;
      ASSERT_EQUALS(&c, &c);
    }

    void compareInvalid() {
      int c;
      ASSERT_EQUALS(&c, &c + 1);
    }
  };

  // A base class for comparing string-like objects.
  class AssertEqualsStringBase : public AssertBase {
  protected:
    void compareValid() {
      compareWords("yes", "yes");
    }

    void compareInvalid() {
      compareWords("no", "know");
    }

    std::string expectedString() const {
      return "Expected no, got know.";
    }

    virtual void compareWords(const char *first, const char *second) = 0;

    std::string createString(const char *text) {
      return text;
    }
  };

  // Test ASSERT_EQUALS for std::strings vs. const char *'s.
  class AssertEqualsStringChar :
    public AssertEqualsStringBase {
  protected:
    void compareWords(const char *first, const char *second) {
      ASSERT_EQUALS(createString(first), second);
    }
  };

  // Test ASSERT_EQUALS for pairs of std::strings.
  class AssertEqualsStringString :
    public AssertEqualsStringBase {
  protected:
    void compareWords(const char *first, const char *second) {
      ASSERT_EQUALS(createString(first), createString(second));
    }
  };

  // Test ASSERT_EQUALS for const char * vs. std::string.
  class AssertEqualsCharString :
    public AssertEqualsStringBase {
  protected:
    void compareWords(const char *first, const char *second) {
      ASSERT_EQUALS(first, createString(second));
    }
  };

  // Test ASSERT_EQUALS for CompareHelper vs. CompareHelper.
  class AssertEqualsCompareHelper : public AssertBase {
  protected:
    void compareValid() {
      ASSERT_EQUALS(CompareHelper(1), CompareHelper(1));
    }

    void compareInvalid() {
      ASSERT_EQUALS(CompareHelper(1), CompareHelper(2));
    }

    std::string expectedString() const {
      return "Expected CompareHelper[1], got CompareHelper[2].";
    }
  };

  // Test ASSERT_EQUALS for CompareHelper vs. int.
  class AssertEqualsCompareHelperInt : public AssertBase {
  protected:
    void compareValid() {
      ASSERT_EQUALS(CompareHelper(1), 1);
    }

    void compareInvalid() {
      ASSERT_EQUALS(CompareHelper(1), 2);
    }

    std::string expectedString() const {
      return "Expected CompareHelper[1], got 2.";
    }
  };

  // Make sure we print for a space after the line number at the end of
  // an assertion message, to make appropriate Emacs versions happy.
  class ExtraSpace : public AssertBase {
  protected:
    void compareInvalid() {
      ASSERT(false);
    }

    std::string expectedString() const {
      return ": \n";
    }
  };

  class Name : public Test {
  protected:
    void run() {
      ASSERT_EQUALS("TestTest::Name", name());
    }
  };

  class RadixBase : public AssertBase {
  protected:
    explicit RadixBase(Radix radix)
      : radix_(UnitTest::radix()) {
      UnitTest::radix(radix);
    }

    ~RadixBase() {
      UnitTest::radix(radix_);
    }

  protected:
    void compareInvalid() {
      ASSERT_EQUALS(10, 20);
    }

  private:
    Radix radix_;
  };

  class RadixDecimal : public RadixBase {
  public:
    RadixDecimal() : RadixBase(DECIMAL) {}

  protected:
    std::string expectedString() const {
      return "Expected 10, got 20.";
    }
  };

  class RadixHex : public RadixBase {
  public:
    RadixHex() : RadixBase(HEX) {}

  protected:
    std::string expectedString() const {
      return "Expected 0xa, got 0x14.";
    }
  };

  class AssertException : public AssertBase {
  protected:
    void compareValid() {
      ASSERT_EXCEPTION(throw 1, int);

      // Make sure that exceptions of the wrong type are let through.
      try {
	ASSERT_EXCEPTION(throw 'a', int);
      } catch (const char &expected) {
      }
    }

    void compareInvalid() {
      ASSERT_EXCEPTION(doNothing(), int);
    }

    std::string expectedString() const {
      return "Expected int to be thrown, but none was.";
    }

  private:
    // NOTE (2004-09-03, carlton): We have a separate function here to
    // avoid compiler warnings about statements with no effect.
    virtual void doNothing() const {
    }
  };

  class AssertExceptionMessage : public AssertBase {
  protected:
    void compareValid() {
      ASSERT_EXCEPTION_MESSAGE(throw std::runtime_error("foo"), 
			       std::runtime_error, "foo");

      // Make sure that exceptions of the wrong type are let through.
      try {
	ASSERT_EXCEPTION_MESSAGE(throw std::runtime_error("foo"), 
				 std::logic_error, "foo");
      } catch (const std::runtime_error &expected) {
      }
    }

    void compareInvalid() {
      expectedString_ = "Expected std::runtime_error to be thrown, "
	"regarding bar; actual exception was foo";
      try {
	ASSERT_EXCEPTION_MESSAGE(throw std::runtime_error("foo"), 
				 std::runtime_error, "bar");
      } catch (const AssertionFailure &expected) {
	lookForString(expected);
      }

      expectedString_ =
	"Expected std::runtime_error to be thrown, but none was.";
      ASSERT_EXCEPTION_MESSAGE(doNothing(), std::runtime_error,
			       "foo");
    }

    std::string expectedString() const {
      return expectedString_;
    }

  private:
    // NOTE (2004-09-03, carlton): We have a separate function here to
    // avoid compiler warnings about statements with no effect.
    virtual void doNothing() const {
    }

    std::string expectedString_;
  };

  class AssertionInOtherClassHelper {
  public:
    void doAssert(bool val) {
      ASSERT(val);
    }
  };

  class AssertionInOtherClass : public AssertBase {
  protected:
    void compareValid() {
      AssertionInOtherClassHelper helper;
      helper.doAssert(true);
    }

    void compareInvalid() {
      AssertionInOtherClassHelper helper;
      helper.doAssert(false);
    }
  };

  class All : public Suite {
  public:
    All() {
      add<Assert>();
      add<AssertEquals>();
      add<AssertEqualsPtr>();
      add<AssertEqualsStringChar>();
      add<AssertEqualsStringString>();
      add<AssertEqualsCharString>();
      add<AssertEqualsCompareHelper>();
      add<AssertEqualsCompareHelperInt>();
      add<ExtraSpace>();
      add(TestPtr(new Name()));
      add<RadixDecimal>();
      add<RadixHex>();
      add<AssertException>();
      add<AssertExceptionMessage>();
      add<AssertionInOtherClass>();
    }
  };

}

UnitTest::TestPtr testTests() {
  return UnitTest::createSuite<TestTest::All>();
}
