/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: charttyp.cxx,v $
 *
 *  $Revision: 1.8 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/08 23:43:54 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "charttyp.hxx"

#include "schattr.hxx"

/*------------------------------------------------------------------------

  Prioritaetenliste:
  -----------------


  3D      > 2D
	 >
  Symbols,Splines > no Symbols,no Splines
	 >
  Lines   > no Lines
	 >
  Percent > Stacked > Normal


  Kompatiblitaetsklassen:

  1 XY-Chart       Hat X-Werte-Spalte
  2 Pie-Chart      keine Achse
  3 Line,Area,Bar  Achse, Symbol,Linie,Splines, ... (fast alles!)
  4 NetChart       ???

		3D   Line  Spline  Symbols Stacked Percent  Deep3D vertikal Errors
--------------------------------------------------------------------------
  1      x     X     X       X          - (x?) -      -      x        X
  2      X     -     -       -          ?      F      -      -        -
  3      x     X     ?       x          X      X      x      x        x
  4      -     F     -       X          X      X      -      -        -?


  X = schaltbar,vorhanden
  x = schaltbar, nicht (vollst.) vrhanden
  F = immer
  ? = weiss noch nicht
  - = gibts nicht
------------------------------------------------------------------------*/

void ChartType::Init()
{
    bHasLines = FALSE;
    bIsDonut = FALSE;
    bIsPercent = FALSE;
    bIs3D = FALSE;
    bIsDeep3D = FALSE;
    bIsVertical = FALSE;
    bIsStacked = FALSE;
    bHasVolume = FALSE;
    bHasUpDown = FALSE;

    nSymbolType = -2;
	nShapeType = -1;

    nSplineType = SPLINE_NONE;
    nSpecialType = 0;

	nBaseType = CHSTYLE_2D_COLUMN;
}

/* ************************************************************************
|*
|* SetType initialisiert die ganze Klasse aus einem SvxChartStyle-enum
|* ( = Konvertierung SvxChartStyle -> ChartType )
|*
\*********************************************************************** */
void ChartType::SetType(const ChartModel* pModel)
{
	SetType(pModel->ChartStyle());
}
void ChartType::SetType(const SfxItemSet *pAttr)
{

	const SfxPoolItem *pPoolItem = NULL;

	if (pAttr->GetItemState(SCHATTR_STYLE_3D, TRUE, &pPoolItem) == SFX_ITEM_SET)
		bIs3D=((const SfxBoolItem*) pPoolItem)->GetValue();

	if (pAttr->GetItemState(SCHATTR_STYLE_DEEP, TRUE, &pPoolItem) == SFX_ITEM_SET)
		bIsDeep3D=((const SfxBoolItem*) pPoolItem)->GetValue();

	if (pAttr->GetItemState(SCHATTR_STYLE_VERTICAL, TRUE, &pPoolItem) == SFX_ITEM_SET)
		bIsVertical=((const SfxBoolItem*) pPoolItem)->GetValue();

	if (pAttr->GetItemState(SCHATTR_STYLE_LINES, TRUE, &pPoolItem) == SFX_ITEM_SET)
		bHasLines=((const SfxBoolItem*) pPoolItem)->GetValue();

	if (pAttr->GetItemState(SCHATTR_STYLE_PERCENT, TRUE, &pPoolItem) == SFX_ITEM_SET)
		bIsPercent=((const SfxBoolItem*) pPoolItem)->GetValue();

	if (pAttr->GetItemState(SCHATTR_STYLE_STACKED, TRUE, &pPoolItem) == SFX_ITEM_SET)
		bIsStacked=((const SfxBoolItem*) pPoolItem)->GetValue();

	if (pAttr->GetItemState(SCHATTR_STYLE_SPLINES, TRUE, &pPoolItem) == SFX_ITEM_SET)
		nSplineType=((const SfxInt32Item*) pPoolItem)->GetValue();

	if (pAttr->GetItemState(SCHATTR_STYLE_SYMBOL, TRUE, &pPoolItem) == SFX_ITEM_SET)
		nSymbolType=((const SfxInt32Item*) pPoolItem)->GetValue();

	if (pAttr->GetItemState(SCHATTR_STYLE_SHAPE, TRUE, &pPoolItem) == SFX_ITEM_SET)
		nShapeType=((const SfxInt32Item*) pPoolItem)->GetValue();

	if (pAttr->GetItemState(SCHATTR_STOCK_VOLUME, TRUE, &pPoolItem) == SFX_ITEM_SET)
		bHasVolume=((const SfxBoolItem*) pPoolItem)->GetValue();

	if (pAttr->GetItemState(SCHATTR_STOCK_UPDOWN, TRUE, &pPoolItem) == SFX_ITEM_SET)
		bHasUpDown=((const SfxBoolItem*) pPoolItem)->GetValue();

    // handle special types
    //   4 : CHSTYLE_2D_LINE_COLUMN
    //   5 : CHSTYLE_2D_LINE_STACKEDCOLUMN

    if( 4 == nSpecialType )
    {
        // CHSTYLE_2D_LINE_COLUMN
        if( bIsStacked )
        {
            // set to CHSTYLE_2D_LINE_STACKEDCOLUMN
            nSpecialType = 5;
        }
    }
    else if( 5 == nSpecialType )
    {
        // CHSTYLE_2D_LINE_STACKEDCOLUMN
        if( ! bIsStacked )
        {
            // set to CHSTYLE_2D_LINE_COLUMN
            nSpecialType = 4;
        }
    }

    //Todo: extend list (?)
}

void ChartType::GetAttrSet(SfxItemSet *pAttr)
{
	pAttr->Put(SfxBoolItem(SCHATTR_STOCK_VOLUME     ,bHasVolume));
	pAttr->Put(SfxBoolItem(SCHATTR_STOCK_UPDOWN     ,bHasUpDown));
	pAttr->Put(SfxBoolItem(SCHATTR_STYLE_3D         ,bIs3D));
	pAttr->Put(SfxBoolItem(SCHATTR_STYLE_DEEP       ,bIsDeep3D));
	pAttr->Put(SfxBoolItem(SCHATTR_STYLE_VERTICAL   ,bIsVertical));
	pAttr->Put(SfxBoolItem(SCHATTR_STYLE_LINES      ,bHasLines));
	pAttr->Put(SfxBoolItem(SCHATTR_STYLE_PERCENT    ,bIsPercent));
	pAttr->Put(SfxBoolItem(SCHATTR_STYLE_STACKED    ,bIsStacked));
	pAttr->Put(SfxInt32Item(SCHATTR_STYLE_SPLINES    ,nSplineType));
	pAttr->Put(SfxInt32Item(SCHATTR_STYLE_SYMBOL     ,nSymbolType));
	if(nShapeType!=-1)
		pAttr->Put(SfxInt32Item(SCHATTR_STYLE_SHAPE      ,nShapeType));
	//Todo: erweitern!
}
void ChartType::SetType(const SvxChartStyle eStyle)
{
	nSymbolType = HasSymbols(eStyle) ? SVX_SYMBOLTYPE_AUTO : SVX_SYMBOLTYPE_NONE;
	bIs3D       = Is3D(eStyle);
	bIsStacked  = IsStacked(eStyle);
	bIsPercent  = IsPercent(eStyle);
	nBaseType   = GetBaseType(eStyle);
	bIsDeep3D   = IsDeep3D(eStyle);
	bIsVertical = IsVertical(eStyle);
	nSplineType = GetSplineType(eStyle);
	bIsDonut    = IsDonut(eStyle);
	bHasLines	= HasLines(eStyle);



	switch(eStyle)
	{
		case CHSTYLE_2D_PIE_SEGOF1:
			nSpecialType=1;
			break;
		case CHSTYLE_2D_PIE_SEGOFALL:
			nSpecialType=2;
			break;
		case CHSTYLE_2D_DONUT2:
			nSpecialType=3;
			break;
		case CHSTYLE_2D_LINE_COLUMN:
			nSpecialType=4;
			break;
		case CHSTYLE_2D_LINE_STACKEDCOLUMN:
			nSpecialType=5;
			break;
		case CHSTYLE_2D_STOCK_1:
			bHasVolume=FALSE;
			bHasUpDown=FALSE;
			break;
		case CHSTYLE_2D_STOCK_2:
			bHasVolume=FALSE;
			bHasUpDown=TRUE;
			break;
		case CHSTYLE_2D_STOCK_3:
			bHasVolume=TRUE;
			bHasUpDown=FALSE;
			break;
		case CHSTYLE_2D_STOCK_4:
			bHasVolume=TRUE;
			bHasUpDown=TRUE;
			break;
		default:
			nSpecialType=0;
			break;
	}
}
/*************************************************************************
|*
|* Konvertierung ChartType -> SvxChartStyle
|*
\************************************************************************/
SvxChartStyle ChartType::GetChartStyle() const
{
	SvxChartStyle aResult = CHSTYLE_2D_COLUMN; // in case of error return default

	switch(nSpecialType)
	{
	case 1:
	   return CHSTYLE_2D_PIE_SEGOF1;
	case 2:
	   return CHSTYLE_2D_PIE_SEGOFALL;
	case 3:
	   return CHSTYLE_2D_DONUT2;
	case 4:
	   return CHSTYLE_2D_LINE_COLUMN;
	case 5 :
	   return CHSTYLE_2D_LINE_STACKEDCOLUMN;
	default:
	   break;
	}

	switch(nBaseType)
	{
		case CHTYPE_DONUT:
			return CHSTYLE_2D_DONUT1;

		case CHTYPE_LINE:
		{

			if(bIs3D)
				return CHSTYLE_3D_STRIPE; //default 3d

			if (nSymbolType!=SVX_SYMBOLTYPE_NONE)
			{

				if(nSplineType==SPLINE_CUBIC)
					return CHSTYLE_2D_CUBIC_SPLINE_SYMBOL;

				if(nSplineType==SPLINE_B)
					return CHSTYLE_2D_B_SPLINE_SYMBOL; //default spline&symbol&line:

				if(bIsPercent) //MUSS vor stacked, da percent auch stacked
					return CHSTYLE_2D_PERCENTLINESYM;

				if(bIsStacked)
					return 	CHSTYLE_2D_STACKEDLINESYM;

				return CHSTYLE_2D_LINESYMBOLS;  //default Line&Symbols


			}

			if(nSplineType==SPLINE_CUBIC)
				return CHSTYLE_2D_CUBIC_SPLINE;

			if(nSplineType==SPLINE_B)
				return CHSTYLE_2D_B_SPLINE;

			if(bIsPercent) //MUSS vor stacked, da percent auch stacked
				return CHSTYLE_2D_PERCENTLINE;

			if(bIsStacked)
				return CHSTYLE_2D_STACKEDLINE;

			return CHSTYLE_2D_LINE; //default Line
		 }
		 break;

		case CHTYPE_AREA:
			{
				if(bIs3D)
				{
					if(bIsPercent)
						return CHSTYLE_3D_PERCENTAREA;
					if(bIsStacked)
						return CHSTYLE_3D_STACKEDAREA;

					return CHSTYLE_3D_AREA;
				}

				if(bIsPercent)
					return CHSTYLE_2D_PERCENTAREA;

				if(bIsStacked)
					return CHSTYLE_2D_STACKEDAREA;

				return CHSTYLE_2D_AREA;
			}
			break;

		case CHTYPE_CIRCLE:
			{
				if(bIs3D)
					return CHSTYLE_3D_PIE;

				if(bIsDonut || bIsStacked)
					return CHSTYLE_2D_DONUT1;
				//case CHSTYLE_2D_PIE_SEGOF1:
				//case CHSTYLE_2D_PIE_SEGOFALL:
				//case CHSTYLE_2D_DONUT2:
				return CHSTYLE_2D_PIE;
			}
			break;

		case CHTYPE_XY:
			{
				if(nSymbolType!=SVX_SYMBOLTYPE_NONE)
				{
					if(nSplineType==SPLINE_CUBIC)
						return CHSTYLE_2D_CUBIC_SPLINE_SYMBOL_XY;

					if(nSplineType==SPLINE_B)
						return CHSTYLE_2D_B_SPLINE_SYMBOL_XY;
				}

				if(nSplineType==SPLINE_CUBIC)
					return CHSTYLE_2D_CUBIC_SPLINE_XY;

				if(nSplineType==SPLINE_B)
					return CHSTYLE_2D_B_SPLINE_XY;

				if(bHasLines && (nSymbolType!=SVX_SYMBOLTYPE_NONE))//fehlte! XY=Symbol+Line
					return CHSTYLE_2D_XY;

				if(bHasLines)
					return CHSTYLE_2D_XY_LINE;

				if(nSymbolType!=SVX_SYMBOLTYPE_NONE)
					return CHSTYLE_2D_XYSYMBOLS;

				DBG_ERROR( "Unknown chart type" );

				return CHSTYLE_2D_XY;
			}
			break;

		case CHTYPE_NET:
			{
				if(nSymbolType!=SVX_SYMBOLTYPE_NONE)
				{
					if(bIsPercent)
						return CHSTYLE_2D_NET_SYMBOLS_PERCENT;

					if(bIsStacked)
						return CHSTYLE_2D_NET_SYMBOLS_STACK;

					return CHSTYLE_2D_NET_SYMBOLS;
				}

				if(bIsPercent)
					return CHSTYLE_2D_NET_PERCENT;

				if(bIsStacked)
					return CHSTYLE_2D_NET_STACK;

				return CHSTYLE_2D_NET;
			}
			break;

		case CHTYPE_COLUMN: //==BAR
		case CHTYPE_BAR:
			{
				if(bIsVertical) //Bar = vertical Column
				{
					if(bIs3D)
					{
						if(bIsPercent)
							return CHSTYLE_3D_PERCENTFLATBAR;
						if(bIsStacked)
							return CHSTYLE_3D_STACKEDFLATBAR;
						if(bIsDeep3D)
							return CHSTYLE_3D_BAR;
						return CHSTYLE_3D_FLATBAR;
					}
					if(bIsPercent)
						return CHSTYLE_2D_PERCENTBAR;
					if(bIsStacked)
						return CHSTYLE_2D_STACKEDBAR;

					return CHSTYLE_2D_BAR;
				}
				if(bIs3D)
				{
					if(bIsPercent)
						return CHSTYLE_3D_PERCENTFLATCOLUMN;
					if(bIsStacked)
						return CHSTYLE_3D_STACKEDFLATCOLUMN;
					if(bIsDeep3D)
						return CHSTYLE_3D_COLUMN;
					return CHSTYLE_3D_FLATCOLUMN;
				}

				if(bIsPercent)
					return CHSTYLE_2D_PERCENTCOLUMN;
				if(bIsStacked)
					return CHSTYLE_2D_STACKEDCOLUMN;

				//case CHSTYLE_2D_LINE_COLUMN:
				//case CHSTYLE_2D_LINE_STACKEDCOLUMN:

				return CHSTYLE_2D_COLUMN;
			}
			break;

		case CHTYPE_STOCK:
			if( bHasVolume )
				aResult = bHasUpDown
					? CHSTYLE_2D_STOCK_4
					: CHSTYLE_2D_STOCK_3;
			else
				aResult = bHasUpDown
					? CHSTYLE_2D_STOCK_2
					: CHSTYLE_2D_STOCK_1;
			break;

		case CHTYPE_ADDIN:
			aResult = CHSTYLE_ADDIN;
			break;

		default:
			DBG_ERROR( "ChartModel::GetBaseType: invalid type!" );
			break;
	}

	return aResult;
}

BOOL ChartType::HasLines(const SvxChartStyle eChartStyle) const
{
    return
        ( GetBaseType( eChartStyle ) == CHTYPE_LINE ) ||
        ( ( GetBaseType( eChartStyle ) == CHTYPE_XY ) &&
          eChartStyle != CHSTYLE_2D_XYSYMBOLS );
}
/*************************************************************************
|*
|* Chart-Typ mit Symbolen
|*
\************************************************************************/
BOOL ChartType::IsDeep3D(const SvxChartStyle eChartStyle) const
{
	switch(eChartStyle)
	{
	case CHSTYLE_3D_COLUMN:
	case CHSTYLE_3D_BAR:
		return TRUE;
	default:
		break;
	}
	return FALSE;
}
/*************************************************************************
|*
|* Chart-Typ mit Splines
|*
\************************************************************************/
BOOL ChartType::HasSplines(const SvxChartStyle eChartStyle) const
{
	switch (eChartStyle)
	{
		case CHSTYLE_2D_CUBIC_SPLINE :
		case CHSTYLE_2D_CUBIC_SPLINE_SYMBOL :
		case CHSTYLE_2D_B_SPLINE :
		case CHSTYLE_2D_B_SPLINE_SYMBOL :
		case CHSTYLE_2D_CUBIC_SPLINE_XY :
		case CHSTYLE_2D_CUBIC_SPLINE_SYMBOL_XY :
		case CHSTYLE_2D_B_SPLINE_XY :
		case CHSTYLE_2D_B_SPLINE_SYMBOL_XY :
			return TRUE;

		default :
			return FALSE;
	}
}
long ChartType::GetSplineType(const SvxChartStyle eChartStyle) const
{
	switch (eChartStyle)
	{
		case CHSTYLE_2D_CUBIC_SPLINE :
		case CHSTYLE_2D_CUBIC_SPLINE_SYMBOL :
		case CHSTYLE_2D_CUBIC_SPLINE_XY :
		case CHSTYLE_2D_CUBIC_SPLINE_SYMBOL_XY :
			return SPLINE_CUBIC;

		case CHSTYLE_2D_B_SPLINE :
		case CHSTYLE_2D_B_SPLINE_SYMBOL :
		case CHSTYLE_2D_B_SPLINE_XY :
		case CHSTYLE_2D_B_SPLINE_SYMBOL_XY :
			return SPLINE_B;

		default :
			return SPLINE_NONE;
	}
}
/*************************************************************************
|*
|* Chart-Typ mit Symbolen
|*
\************************************************************************/
BOOL ChartType::HasSymbols(const SvxChartStyle eChartStyle) const
{
	switch(eChartStyle)
	{
		case CHSTYLE_2D_LINESYMBOLS:
		case CHSTYLE_2D_STACKEDLINESYM:
		case CHSTYLE_2D_PERCENTLINESYM:
		case CHSTYLE_2D_XYSYMBOLS:
		case CHSTYLE_2D_XY://fehlte! XY=Symbol+Line
		case CHSTYLE_3D_XYZSYMBOLS:
		case CHSTYLE_2D_NET_SYMBOLS_STACK:
		case CHSTYLE_2D_NET_SYMBOLS:
		case CHSTYLE_2D_NET_SYMBOLS_PERCENT:
		case CHSTYLE_2D_CUBIC_SPLINE_SYMBOL:
		case CHSTYLE_2D_B_SPLINE_SYMBOL:
		case CHSTYLE_2D_CUBIC_SPLINE_SYMBOL_XY:
		case CHSTYLE_2D_B_SPLINE_SYMBOL_XY:

        case CHSTYLE_2D_STOCK_1:
        case CHSTYLE_2D_STOCK_2:
        case CHSTYLE_2D_STOCK_3:
        case CHSTYLE_2D_STOCK_4:

        case CHSTYLE_2D_LINE_COLUMN:
        case CHSTYLE_2D_LINE_STACKEDCOLUMN:

			return TRUE;
			break;

		default:
			break;
	}
	return FALSE;
}
/*************************************************************************
|*
|* 3D-Chart-Typ
|*
\************************************************************************/
BOOL ChartType::Is3D(const SvxChartStyle eChartStyle) const
{
	switch (eChartStyle)
	{
		case CHSTYLE_3D_STRIPE:
		case CHSTYLE_3D_COLUMN:
		case CHSTYLE_3D_BAR:
		case CHSTYLE_3D_FLATCOLUMN:
		case CHSTYLE_3D_FLATBAR:
		case CHSTYLE_3D_STACKEDFLATCOLUMN:
		case CHSTYLE_3D_STACKEDFLATBAR:
		case CHSTYLE_3D_PERCENTFLATCOLUMN:
		case CHSTYLE_3D_PERCENTFLATBAR:
		case CHSTYLE_3D_AREA:
		case CHSTYLE_3D_STACKEDAREA:
		case CHSTYLE_3D_PERCENTAREA:
		case CHSTYLE_3D_SURFACE:
		case CHSTYLE_3D_PIE:
			//neu (aber bisher (380 Build 1502) nicht benutzt):
		case CHSTYLE_3D_XYZ:
		case CHSTYLE_3D_XYZSYMBOLS:
			return TRUE;

		default:
			return FALSE;
	}
}

/*************************************************************************
|*
|* Stacked-Chart-Typ (vollstaendig, d.h. percent => stacked
|*
\************************************************************************/
BOOL ChartType::IsStacked(const SvxChartStyle eChartStyle) const
{
	if(IsPercent(eChartStyle))  //Percent ist immer Stacked!!!
		return TRUE;

	switch(eChartStyle)
	{
		case CHSTYLE_2D_STACKEDCOLUMN:
		case CHSTYLE_2D_STACKEDBAR:
		case CHSTYLE_2D_STACKEDLINE:
		case CHSTYLE_2D_STACKEDAREA:
		case CHSTYLE_3D_STACKEDFLATCOLUMN:
		case CHSTYLE_3D_STACKEDFLATBAR:
		case CHSTYLE_3D_STACKEDAREA:
		//neu, (siehe auch IsPercent()):
		case CHSTYLE_2D_STACKEDLINESYM:
		case CHSTYLE_2D_NET_STACK:
		case CHSTYLE_2D_NET_SYMBOLS_STACK:
		case CHSTYLE_2D_LINE_STACKEDCOLUMN:

			return TRUE;
			break;

		default:
			break;
	}
	return FALSE;
}
/*************************************************************************
|*
|* Percent-Chart-Typ
|*
\************************************************************************/
BOOL ChartType::IsPercent(const SvxChartStyle eChartStyle) const
{
	switch(eChartStyle)
	{
		case CHSTYLE_2D_PERCENTCOLUMN:
		case CHSTYLE_2D_PERCENTBAR:
		case CHSTYLE_2D_PERCENTLINE:
		case CHSTYLE_2D_PERCENTAREA:
		case CHSTYLE_3D_PERCENTFLATCOLUMN:
		case CHSTYLE_3D_PERCENTAREA:
		// Neu:
		case CHSTYLE_2D_NET_PERCENT:
		case CHSTYLE_2D_NET_SYMBOLS_PERCENT:
		case CHSTYLE_2D_PERCENTLINESYM:
		case CHSTYLE_3D_PERCENTFLATBAR:

			return TRUE;
			break;

		default:
			break;
	}
	return FALSE;
}

/*************************************************************************
|*
|* Basistyp ermitteln
|*
\************************************************************************/

long ChartType::GetBaseType(const SvxChartStyle eChartStyle) const
{
	long nResult = CHTYPE_INVALID;

	switch( eChartStyle )
	{
		case CHSTYLE_2D_B_SPLINE:
		case CHSTYLE_2D_B_SPLINE_SYMBOL:
		case CHSTYLE_2D_LINE:
		case CHSTYLE_2D_STACKEDLINE:
		case CHSTYLE_2D_PERCENTLINE:
		case CHSTYLE_2D_LINESYMBOLS:
		case CHSTYLE_2D_STACKEDLINESYM:
		case CHSTYLE_2D_PERCENTLINESYM:
		case CHSTYLE_2D_CUBIC_SPLINE:
		case CHSTYLE_2D_CUBIC_SPLINE_SYMBOL:
		case CHSTYLE_3D_STRIPE:
			nResult = CHTYPE_LINE;
			break;

		case CHSTYLE_2D_AREA:
		case CHSTYLE_2D_STACKEDAREA:
		case CHSTYLE_2D_PERCENTAREA:
		case CHSTYLE_3D_AREA:
		case CHSTYLE_3D_STACKEDAREA:
		case CHSTYLE_3D_PERCENTAREA:
			nResult = CHTYPE_AREA;
			break;

		case CHSTYLE_2D_PIE:
		case CHSTYLE_2D_PIE_SEGOF1:
		case CHSTYLE_2D_PIE_SEGOFALL:
		case CHSTYLE_2D_DONUT1:
		case CHSTYLE_2D_DONUT2:
		case CHSTYLE_3D_PIE:
			nResult = CHTYPE_CIRCLE;
			break;

		case CHSTYLE_2D_B_SPLINE_XY:
		case CHSTYLE_2D_XY_LINE:
		case CHSTYLE_2D_B_SPLINE_SYMBOL_XY:
		case CHSTYLE_2D_XYSYMBOLS:
		case CHSTYLE_2D_XY:
		case CHSTYLE_2D_CUBIC_SPLINE_XY:
		case CHSTYLE_2D_CUBIC_SPLINE_SYMBOL_XY:
			nResult = CHTYPE_XY;
			break;

		case CHSTYLE_2D_NET:
		case CHSTYLE_2D_NET_SYMBOLS:
		case CHSTYLE_2D_NET_STACK:
		case CHSTYLE_2D_NET_SYMBOLS_STACK:
		case CHSTYLE_2D_NET_PERCENT:
		case CHSTYLE_2D_NET_SYMBOLS_PERCENT:
			nResult = CHTYPE_NET;
			break;

		case CHSTYLE_2D_COLUMN:
		case CHSTYLE_2D_STACKEDCOLUMN:
		case CHSTYLE_2D_PERCENTCOLUMN:
		case CHSTYLE_2D_LINE_COLUMN:
		case CHSTYLE_2D_LINE_STACKEDCOLUMN:
		case CHSTYLE_3D_COLUMN:
		case CHSTYLE_3D_FLATCOLUMN:
		case CHSTYLE_3D_STACKEDFLATCOLUMN:
		case CHSTYLE_3D_PERCENTFLATCOLUMN:
			nResult = CHTYPE_COLUMN;
			break;

		case CHSTYLE_2D_BAR:
		case CHSTYLE_2D_STACKEDBAR:
		case CHSTYLE_2D_PERCENTBAR:
		case CHSTYLE_3D_BAR:
		case CHSTYLE_3D_FLATBAR:
		case CHSTYLE_3D_STACKEDFLATBAR:
		case CHSTYLE_3D_PERCENTFLATBAR:
			nResult = CHTYPE_BAR;
			break;

		case CHSTYLE_2D_STOCK_1:
		case CHSTYLE_2D_STOCK_2:
		case CHSTYLE_2D_STOCK_3:
		case CHSTYLE_2D_STOCK_4:
			nResult = CHTYPE_STOCK;
			break;

		case CHSTYLE_ADDIN:
			nResult = CHTYPE_ADDIN;
			break;
			
		default:
			DBG_ERROR( "Invalid chart style given!" );
			break;
	}

	return nResult;
}

/*************************************************************************
|*
|* Vertikales Chart
|*
\************************************************************************/

BOOL ChartType::IsVertical(const SvxChartStyle eChartStyle) const
{
	switch(eChartStyle)
	{
			case CHSTYLE_2D_BAR:
			case CHSTYLE_2D_STACKEDBAR:
			case CHSTYLE_2D_PERCENTBAR:

			case CHSTYLE_3D_BAR:
			case CHSTYLE_3D_FLATBAR:
			case CHSTYLE_3D_STACKEDFLATBAR:
			case CHSTYLE_3D_PERCENTFLATBAR:

				return TRUE;
				break;

			default:
				break;
		}
		return FALSE;
}




BOOL ChartType::IsDonut(const SvxChartStyle eChartStyle) const
{
	switch (eChartStyle)
	{
		case CHSTYLE_2D_DONUT1:
		case CHSTYLE_2D_DONUT2:

			return TRUE;
			break;

		default:
			break;
	}
	return FALSE;
}
/*************************************************************************
|*
|* CleanUp sorgt nach dem Setzen bestimmter Eigenschaften dafuer, das
|* alle uebrigen Eigenschaften, die jetzt nicht mehr verfuegbar sind,
|* passend gesetzt werden. Beispiel:
|* Type = Percent, stacked              oder    Typ = Column
|* Percent wird auf FALSE gesetzt               Typ auf Pie
|* nach CleanUp ist auch Stacked = FALSE;       Typ = Pie,Percent,Stacked
|*                                              Typ auf Column
|*                                              Type = Column,Percent,Stacked
\************************************************************************/

BOOL ChartType::CleanUp()
{
   SvxChartStyle eOldStyle = GetChartStyle();
   SetType(eOldStyle);
   SvxChartStyle eNewStyle = GetChartStyle();

   return eNewStyle == eOldStyle;
}
//Wie SetBaseType, jedoch werden alle Properties auf defaults gesetz
BOOL ChartType::SetDefaultBaseType(long nDefault)
{
	if(nDefault==-1)
		nDefault=GetBaseType();

	SvxChartStyle eOldStyle = GetChartStyle();

	SvxChartStyle eNewStyle=CHSTYLE_2D_COLUMN;

	switch(nDefault)
	{
		case CHTYPE_LINE:
			eNewStyle = (Is3D()) ? CHSTYLE_3D_STRIPE : CHSTYLE_2D_LINE;
			break;
		case CHTYPE_AREA:
			eNewStyle = (Is3D()) ?  CHSTYLE_3D_AREA: CHSTYLE_2D_AREA;
			break;
		case CHTYPE_DONUT:
			eNewStyle = CHSTYLE_2D_DONUT1;
			break;
		case CHTYPE_CIRCLE:
			eNewStyle = (Is3D()) ?  CHSTYLE_3D_PIE: CHSTYLE_2D_PIE;
			break;
		case CHTYPE_XY:
			eNewStyle = (Is3D()) ?  CHSTYLE_2D_XY: CHSTYLE_2D_XY;
			break;
		case CHTYPE_NET:
			eNewStyle = (Is3D()) ?  CHSTYLE_2D_NET: CHSTYLE_2D_NET ;
			break;
		case CHTYPE_COLUMN:
		case CHTYPE_BAR:
			eNewStyle = (Is3D()) ?  CHSTYLE_3D_COLUMN: CHSTYLE_2D_COLUMN;
			break;
		case CHTYPE_STOCK:
			eNewStyle = CHSTYLE_2D_STOCK_1;

			break;
	}

	SetType(eNewStyle);

	return eNewStyle == eOldStyle;
}
