/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <glib/gi18n.h>

#include "util.h"
#include "gpass/password.h"

static GPassEntryClass *parent_class = NULL;

static const gchar *
password_class_icon_id(void)
{
    return "gnome-stock-authentication";
}

static void
password_class_attributes(GPassEntryClass *entry_class,
                          GPassAttributeList *attributes)
{
    GObjectClass *g_object_class = G_OBJECT_CLASS(entry_class);
    GPassAttribute *attr;
    GParamSpec *spec;

    parent_class->attributes(entry_class, attributes);
    
    spec = g_object_class_find_property(g_object_class, "username");
    attr = gpass_param_spec_to_attribute(spec, GPASS_ATTRIBUTE_TYPE_STRING);
    gpass_attribute_list_append(attributes, attr);
    g_object_unref(attr);
    
    spec = g_object_class_find_property(g_object_class, "password");
    attr = gpass_param_spec_to_attribute(spec, GPASS_ATTRIBUTE_TYPE_PASSWORD);
    gpass_attribute_list_append(attributes, attr);
    g_object_unref(attr);
}

static void
password_instance_init(GTypeInstance *instance, gpointer g_class)
{
    GPassPassword *self = GPASS_PASSWORD(instance);
    
    GPASS_ENTRY(self)->type = "password";
    self->username = g_strdup("");
    self->password = g_strdup("");
}

enum {
    PROP_0,
    PROP_USERNAME,
    PROP_PASSWORD,
    PROP_ID,
};

static void
password_instance_set_property(GObject *object, guint prop_id,
                               const GValue *value, GParamSpec *pspec)
{
    GPassPassword *self = GPASS_PASSWORD(object);
  
    switch (prop_id) {
    case PROP_USERNAME:
        g_free(self->username);
        self->username = g_value_dup_string(value);
        break;
    case PROP_PASSWORD:
        g_free(self->password);
        self->password = g_value_dup_string(value);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
password_instance_get_property(GObject *object, guint prop_id,
                               GValue *value, GParamSpec *pspec)
{
    GPassPassword *self = GPASS_PASSWORD(object);

    switch (prop_id) {
    case PROP_USERNAME:
        g_value_set_static_string(value, self->username);
        break;
    case PROP_PASSWORD:
        g_value_set_static_string(value, self->password);
        break;
    case PROP_ID:
        g_value_set_static_string(value, self->username);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
password_instance_finalize(GObject *object)
{
    GPassPassword *self = GPASS_PASSWORD(object);

    g_free(self->username);
    g_free(self->password);
    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
password_instance_set_attributes(GPassEntry *entry,
                                 GPassAttributeList *attributes)
{
    GPassPassword *self = GPASS_PASSWORD(entry);
    GPassAttribute *attr;

    parent_class->set_attributes(entry, attributes);
    
    attr = gpass_attribute_list_lookup(attributes, "username");
    if (attr != NULL) {
        g_object_set_property(G_OBJECT(self), "username", attr->value);
    }
    attr = gpass_attribute_list_lookup(attributes, "password");
    if (attr != NULL) {
        g_object_set_property(G_OBJECT(self), "password", attr->value);
    }
}

static void
password_instance_get_attributes(GPassEntry *entry,
                                 GPassAttributeList *attributes)
{
    GPassPassword *self = GPASS_PASSWORD(entry);
    GPassAttribute *attr;

    parent_class->get_attributes(entry, attributes);
    
    attr = gpass_attribute_list_lookup(attributes, "username");
    if (attr != NULL) {
        gpass_attribute_set(attr, self->username);
    }
    attr = gpass_attribute_list_lookup(attributes, "password");
    if (attr != NULL) {
        gpass_attribute_set(attr, self->password);
    }
}

static gboolean
password_instance_include(GPassEntry *entry, const gchar *string)
{
    GPassPassword *self = GPASS_PASSWORD(entry);
    
    if (parent_class->include(entry, string)) {
        return TRUE;
    }
    if (strstr(self->username, string)) {
        return TRUE;
    }
    return FALSE;
}

static gboolean
password_instance_equal(GPassEntry *entry, GPassEntry *target)
{
    GPassPassword *self = GPASS_PASSWORD(entry);
    GPassPassword *password;
    
    if (!parent_class->equal(entry, target)) {
        return FALSE;
    }
    password = GPASS_PASSWORD(target);
    if (gpass_strcmp(self->username, password->username) != 0) {
        return FALSE;
    }
    if (gpass_strcmp(self->password, password->password) != 0) {
        return FALSE;
    }
    return TRUE;
}

static void
password_class_init(gpointer g_class, gpointer g_class_data)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(g_class);
    GPassEntryClass *entry_class = GPASS_ENTRY_CLASS(g_class);

    parent_class = g_type_class_peek_parent(g_class);
    
    entry_class->icon_id = password_class_icon_id;
    entry_class->attributes = password_class_attributes;

    gobject_class->set_property = password_instance_set_property;
    gobject_class->get_property = password_instance_get_property;
    gobject_class->finalize = password_instance_finalize;
    entry_class->set_attributes = password_instance_set_attributes;
    entry_class->get_attributes = password_instance_get_attributes;
    entry_class->include = password_instance_include;
    entry_class->equal = password_instance_equal;

    g_object_class_install_property
        (gobject_class, PROP_USERNAME,
         g_param_spec_string("username", _("Username"),
                             _("Username"),
                             NULL, G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, PROP_PASSWORD,
         g_param_spec_string("password", _("Password"),
                             _("Password"),
                             NULL, G_PARAM_READWRITE));
    g_object_class_override_property(gobject_class, PROP_ID, "id");
}

GType
gpass_password_get_type(void)
{
    static GType type = 0;
    
    if (type == 0) {
        static const GTypeInfo info = {
            sizeof(GPassPasswordClass),
            NULL,
            NULL,
            password_class_init,
            NULL,
            NULL,
            sizeof(GPassPassword),
            0,
            password_instance_init
        };
        
        type = g_type_register_static(GPASS_TYPE_ENTRY, "GPassPassword",
                                      &info, G_TYPE_FLAG_ABSTRACT);
    }
    return type;
}

/***********************************************************
 *
 * generate password
 *
 ***********************************************************/
GType
gpass_password_generate_type_get_type(void)
{
    static GType etype = 0;
  
    if (etype == 0) {
        static const GEnumValue values[] = {
            { GPASS_PASSWORD_GENERATE_TYPE_RANDOM,
              "GPASS_PASSWORD_GENERATE_TYPE_RANDOM", N_("Random")},
            { 0, NULL, NULL }
        };
        
        etype = g_enum_register_static("GPassPasswordGenerateType", values);
    }
    return etype;
}

#define LOWERCASE_CHARS "abcdefgkijklmnopqrstuvwxyz"
#define UPPERCASE_CHARS "ABCDEFGKIJKLMNOPQRSTUVWXYZ"
#define NUMBER_CHARS    "0123456789"
#define SPECIAL_CHARS   "!@#$%^&*()_+=-'\"\\{}[]?><,.|"

GError *
random_password_generator(gint length, GString *chars, GString **password)
{
    gchar *result = NULL;
    int fd;
    ssize_t read_len;
    guint index;
    int i;
    GError *error = NULL;

    if ((fd = open("/dev/urandom", O_RDONLY, 0)) == -1) {
        if ((fd = open("/dev/random", O_RDONLY | O_NONBLOCK, 0)) == -1) {
            g_set_error(&error, 0, errno,
                        _("could not open random device"));
            return error;
        }
    }
    result = g_malloc(sizeof(gchar) * length + 1);
    read_len = read(fd, result, length);
    close(fd);
    if (read_len != length) {
        g_set_error(&error, 0, errno, _("could not read random device"));
        g_free(result);
        return error;
    }
    for (i = 0; i < length; i++) {
        index = ((unsigned char) result[i]) % chars->len;
        result[i] = chars->str[index];
    }
    result[length] = '\0';
    *password = g_string_assign(*password, result);
    return NULL;
}

GError *
gpass_password_generate(GPassPasswordGenerateType type,
                        gint use_flags, gint length, GString **password)
{
    GString *chars;
    GError *error = NULL;

    if (length == 0) {
        g_set_error(&error, 0, 0, _("must set length"));
        return error;
    }
    if (use_flags == 0) {
        g_set_error(&error, 0, 0, _("must set use_flags"));
        return error;
    }
    chars = g_string_new(NULL);
    if (use_flags & GPASS_PASSWORD_GENERATE_USE_LOWERCASE) {
        chars = g_string_append(chars, LOWERCASE_CHARS);
    }
    if (use_flags & GPASS_PASSWORD_GENERATE_USE_UPPERCASE) {
        chars = g_string_append(chars, UPPERCASE_CHARS);
    }
    if (use_flags & GPASS_PASSWORD_GENERATE_USE_NUMBER) {
        chars = g_string_append(chars, NUMBER_CHARS);
    }
    if (use_flags & GPASS_PASSWORD_GENERATE_USE_SPECIAL) {
        chars = g_string_append(chars, SPECIAL_CHARS);
    }
    if (chars->len == 0) {
        g_set_error(&error, 0, 0, _("must set use_flags"));
        goto end;
    }
    switch (type) {
    case GPASS_PASSWORD_GENERATE_TYPE_RANDOM:
        error = random_password_generator(length, chars, password);
        break;
    default:
        g_set_error(&error, 0, 0,
                    _("not supported password generate type: %d"), type);
    }
 end:
    g_string_free(chars, TRUE);
    return error;
}
