/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "entry-tree.h"
#include "summary.h"
#include "application.h"

static GObjectClass *parent_class = NULL;

#define DEFAULT_FILE_PATH "%s/.gpass/passwords.gps"

static void
gpass_application_instance_init(GTypeInstance *instance, gpointer g_class)
{
    GPassApplication *self = (GPassApplication *) instance;
    
    self->file_path = g_strdup_printf(DEFAULT_FILE_PATH, g_get_home_dir());
    self->file = NULL;
    self->passwords = NULL;
    self->entry_factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);
    self->view_factory = NULL;
    self->command_stack = g_object_new(GPASS_TYPE_COMMAND_STACK, NULL);
    self->changed_master_password = FALSE;
}

static void
gpass_application_instance_finalize(GObject *object)
{
    GPassApplication *self = (GPassApplication *) GPASS_APPLICATION(object);

    g_free(self->file_path);
    if (self->file != NULL) {
        g_object_unref(self->file);
    }
    if (self->passwords != NULL) {
        g_object_unref(self->passwords);
    }
    g_object_unref(self->entry_factory);
    g_object_unref(self->command_stack);
    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
gpass_application_class_init (gpointer g_class, gpointer g_class_data)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(g_class);

    parent_class = g_type_class_peek_parent(g_class);
    
    gobject_class->finalize = gpass_application_instance_finalize;
}

GType
gpass_application_get_type(void)
{
    static GType type = 0;
    if (type == 0) {
        static const GTypeInfo info = {
            sizeof(GPassApplicationClass),
            NULL,
            NULL,
            gpass_application_class_init,
            NULL,
            NULL,
            sizeof(GPassApplication),
            0,
            gpass_application_instance_init
        };
        
        type = g_type_register_static(G_TYPE_OBJECT,
                                      "GPassApplicationType", &info, 0);
    }
    return type;
}

#define GPASS_DAT_PATH "%s/.gpass/gpass.dat"

static GError *
gpass_application_intergradation(GPassApplication *self, gboolean *result)
{
    GPassView *view;
    GPassViewResult view_result;
    GError *error;
    
    error = gpass_view_factory_create_view(self->view_factory,
                                           "intergradation", self, &view);
    if (error != NULL) {
        return error;
    }
    error = gpass_view_run(view, &view_result);
    if (error != NULL) {
        return error;
    }
    if (view_result == GPASS_VIEW_RESULT_SUCCEED) {
        *result = TRUE;
    }
    else {
        *result = FALSE;
    }
    return gpass_view_factory_destroy_view(self->view_factory,
                                           "intergradation");
}

GError *
gpass_application_welcome(GPassApplication *self, gboolean *result)
{
    const gchar *template;
    GPassView *view;
    GPassViewResult view_result;
    GError *error = NULL;
    
    if (g_file_test(self->file_path, G_FILE_TEST_EXISTS)) {
        template = "authentication";
    }
    else {
        gchar *gpass_dat_path =
            g_strdup_printf(GPASS_DAT_PATH, g_get_home_dir());
        
        if (g_file_test(gpass_dat_path, G_FILE_TEST_EXISTS)) {
            gboolean intergradation_result;
            
            error = gpass_application_intergradation(self,
                                                     &intergradation_result);
            if (error != NULL) {
                g_free(gpass_dat_path);
                return error;
            }
            if (intergradation_result) {
                *result = TRUE;
                return NULL;
            }
        }
        g_free(gpass_dat_path);
        template = "welcome";
    }
    error = gpass_view_factory_create_view(self->view_factory,
                                           template, self, &view);
    if (error != NULL) {
        return error;
    }
    error = gpass_view_run(view, &view_result);
    if (error != NULL) {
        return error;
    }
    if (view_result == GPASS_VIEW_RESULT_SUCCEED) {
        *result = TRUE;
    }
    else {
        *result = FALSE;
    }
    return gpass_view_factory_destroy_view(self->view_factory, template);
}

GError *
gpass_application_run(GPassApplication *self, gboolean *result)
{
    const gchar *template = "window";
    GPassView *view;
    GPassViewResult view_result;
    GError *error = NULL;

    error = gpass_file_read(self->file, self->entry_factory, &self->passwords);
    if (error != NULL) {
        return error;
    }
    error = gpass_view_factory_create_view(self->view_factory,
                                           template, self, &view);
    if (error != NULL) {
        return error;
    }
    error = gpass_view_run(view, &view_result);
    if (error != NULL) {
        return error;
    }
    if (view_result == GPASS_VIEW_RESULT_SUCCEED) {
        *result = TRUE;
    }
    else {
        *result = FALSE;
    }
    error = gpass_view_factory_destroy_view(self->view_factory, template);
    return error;
}

void
gpass_application_change_master_password(GPassApplication *self,
                                         const gchar *master_password)
{
    g_object_set(self->file, "master_password", master_password, NULL);
    self->changed_master_password = TRUE;
}

gboolean
gpass_application_passwords_is_changed(GPassApplication *self)
{
    if (self->changed_master_password) {
        return TRUE;
    }
    return !gpass_command_stack_at_base(self->command_stack);
}

GError *
gpass_application_passwords_save(GPassApplication *self, gboolean *result)
{
    GError *error = NULL;
    
    if (gpass_application_passwords_is_changed(self)) {
        error = gpass_file_write(self->file, self->passwords);
        if (error == NULL) {
            self->changed_master_password = FALSE;
            gpass_command_stack_set_base(self->command_stack);
            *result = TRUE;
        }
    }
    else {
        *result = FALSE;
    }
    return error;
}

GError *
gpass_application_passwords_edit(GPassApplication *self, GPassEntry *target,
                                 GPassEntry *edited)
{
    GPassEntry *parent = gpass_entry_parent(target);
    GPassEntry *sibling = gpass_entry_next_sibling(target);
    GPassView *view;
    GError *error;
    
    error = gpass_view_factory_create_view(self->view_factory,
                                           "window", self, &view);
    if (error != NULL) {
        return error;
    }
    gpass_entry_tree_change_entry(GPASS_WINDOW(view), target, edited);
    if (gpass_entry_has_child(target)) {
        GPassEntry *p = gpass_entry_first_child(target);

        while (p != NULL) {
            GPassEntry *next = gpass_entry_next_sibling(p);

            gpass_entry_unlink(p);
            gpass_entry_append(edited, p);
            p = next;
        }
    }
    gpass_entry_unlink(target);
    gpass_entry_insert_before(parent, sibling, edited);
    if (GPASS_WINDOW(view)->current == target) {
        GPASS_WINDOW(view)->current = edited;
        gpass_summary_update(GPASS_WINDOW(view));
    }
    return NULL;
}

GError *
gpass_application_passwords_insert(GPassApplication *self, GPassEntry *target,
                                   GPassEntry *parent, GPassEntry *sibling)
{
    GPassView *view;
    GError *error;
    
    error = gpass_view_factory_create_view(self->view_factory,
                                           "window", self, &view);
    if (error != NULL) {
        return error;
    }
    gpass_entry_tree_insert(GPASS_WINDOW(view), target, parent, sibling);
    gpass_entry_insert_before(parent, sibling, target);
    return NULL;
}

GError *
gpass_application_passwords_unlink(GPassApplication *self, GPassEntry *target,
                                   GPassEntry *parent, GPassEntry *sibling)
{
    GPassView *view;
    GError *error;
    
    error = gpass_view_factory_create_view(self->view_factory,
                                           "window", self, &view);
    if (error != NULL) {
        return error;
    }
    gpass_entry_tree_unlink(GPASS_WINDOW(view), target, parent, sibling);
    gpass_entry_unlink(target);
    return NULL;
}

GError *
gpass_application_passwords_move_after(GPassApplication *self,
                                       GPassEntry *target,
                                       GPassEntry *parent,
                                       GPassEntry *previous)
{
    GPassView *view;
    GError *error;
    
    error = gpass_view_factory_create_view(self->view_factory,
                                           "window", self, &view);
    if (error != NULL) {
        return error;
    }
    gpass_entry_tree_move_after(GPASS_WINDOW(view), target, parent, previous);
    gpass_entry_unlink(target);
    gpass_entry_insert_after(parent, previous, target);
    return NULL;
}
