/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n.h>
#include <gtk/gtklabel.h>

#include "helper.h"
#include "view.h"

/***********************************************************
 *
 * GPassViewResult
 *
 ***********************************************************/
GType
gpass_view_result_get_type(void)
{
    static GType etype = 0;
  
    if (etype == 0) {
        static const GEnumValue values[] = {
            { GPASS_VIEW_RESULT_FAILED,
              "GPASS_VIEW_RESULT_FAILED", "failed" },
            { GPASS_VIEW_RESULT_SUCCEED,
              "GPASS_VIEW_RESULT_SUCCEED", "succeed" },
            { 0, NULL, NULL }
        };
        etype = g_enum_register_static("GPassViewResult", values);
    }
    return etype;
}

/***********************************************************
 *
 * GPassView
 *
 ***********************************************************/
static GObjectClass *parent_class = NULL;

static void
gpass_view_instance_init(GTypeInstance *instance, gpointer g_class)
{
    GPassView *self = (GPassView *) instance;

    self->model = NULL;
    self->template = NULL;
    self->xml = NULL;
    self->window = NULL;
    self->first_widget = NULL;
    self->result = GPASS_VIEW_RESULT_SUCCEED;
}

enum {
  PROP_0,
  PROP_MODEL,
  PROP_TEMPLATE,
};

static GError *
set_template(GPassView *self, const gchar *template)
{
    gchar *path = NULL;
    GError *error = NULL;
    
    g_free(self->template);
    self->template = g_strdup(template);
    if (self->xml != NULL) {
        g_object_unref(self->xml);
    }
    path = g_strdup_printf(PACKAGE_DATA_DIR "/%s.glade", template);
    self->xml = glade_xml_new(path, NULL, NULL);
    if (self->xml == NULL) {
        g_set_error(&error, 0, 0, _("Could not find template: %s"), path);
        goto end;
    }
    self->window = glade_xml_get_widget(self->xml, template);
    if (self->window == NULL) {
        g_set_error(&error, 0, 0, _("Could not find widget: %s in %s"),
                    template, path);
        goto end;
    }
    g_object_set_data(G_OBJECT(self->xml), "self", self);
    glade_xml_signal_autoconnect(self->xml);
    error = GPASS_VIEW_GET_CLASS(self)->loaded_glade_xml(self);
 end:
    g_free(path);
    return error;
}

static void
gpass_view_set_property(GObject *object, guint prop_id, const GValue *value,
                        GParamSpec *pspec)
{
    GPassView *self = GPASS_VIEW(object);
    GError *error;
  
    switch (prop_id) {
    case PROP_MODEL:
        self->model = g_value_get_pointer(value);
        break;
    case PROP_TEMPLATE:
        error = set_template(self, g_value_get_string(value));
        if (error != NULL) {
            gpass_helper_error(error);
        }
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
gpass_view_get_property(GObject *object, guint prop_id, GValue *value,
                        GParamSpec *pspec)
{
    GPassView *self = GPASS_VIEW(object);

    switch (prop_id) {
    case PROP_MODEL:
        g_value_set_pointer(value, self->model);
        break;
    case PROP_TEMPLATE:
        g_value_set_string(value, self->template);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static GError *
gpass_view_instance_loaded_glade_xml(GPassView *self)
{
    return NULL;
}

static GError *
gpass_view_instance_run(GPassView *self, GPassViewResult *result)
{
    gtk_widget_show_all(self->window);
    if (self->first_widget != NULL) {
        gtk_window_set_focus(GTK_WINDOW(self->window), self->first_widget);
    }
    self->main_loop = g_main_loop_new(NULL, FALSE);
    GDK_THREADS_LEAVE();
    g_main_loop_run(self->main_loop);
    GDK_THREADS_ENTER();
    g_main_loop_unref(self->main_loop);
    gtk_widget_hide(self->window);
    *result = self->result;
    return NULL;
}

static void
gpass_view_instance_finalize(GObject *object)
{
    GPassView *self = GPASS_VIEW(object);

    g_free(self->template);
    if (self->xml != NULL) {
        g_object_unref(self->xml);
    }
    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
gpass_view_class_init(gpointer g_class, gpointer g_class_data)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(g_class);
    GPassViewClass *klass = GPASS_VIEW_CLASS(g_class);

    parent_class = g_type_class_peek_parent(g_class);
    
    gobject_class->set_property = gpass_view_set_property;
    gobject_class->get_property = gpass_view_get_property;
    gobject_class->finalize = gpass_view_instance_finalize;

    klass->loaded_glade_xml = gpass_view_instance_loaded_glade_xml;
    klass->run = gpass_view_instance_run;
    
    g_object_class_install_property
        (gobject_class, PROP_MODEL,
         g_param_spec_pointer("model", _("Model"),
                              _("The data model"), G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, PROP_TEMPLATE,
         g_param_spec_string("template", _("Template"),
                             _("The name of template"),
                             NULL, G_PARAM_READWRITE));
}

GType
gpass_view_get_type(void)
{
    static GType type = 0;

    if (type == 0) {
        static const GTypeInfo info = {
            sizeof(GPassViewClass),
            NULL,
            NULL,
            gpass_view_class_init,
            NULL,
            NULL,
            sizeof(GPassView),
            0,
            gpass_view_instance_init
        };

        type = g_type_register_static(G_TYPE_OBJECT, "GPassViewType",
                                      &info, G_TYPE_FLAG_ABSTRACT);
    }
    return type;
}

GError *
gpass_view_run(GPassView *self, GPassViewResult *result)
{
    return GPASS_VIEW_GET_CLASS(self)->run(self, result);
}

void
gpass_view_shutdown_main_loop(GPassView *self)
{
    if (g_main_loop_is_running(self->main_loop)) {
        g_main_loop_quit(self->main_loop);
    }
}

void
gpass_view_self_from_widget(GtkWidget *widget, gpointer **self)
{
    gpointer result;
    GladeXML *xml;
    GError *error = NULL;
    
    xml = glade_get_widget_tree(GTK_WIDGET(widget));
    if (xml == NULL) {
        g_set_error(&error, 0, 0, _("could not get glade xml"));
        gpass_helper_error(error);
    }
    result = g_object_get_data(G_OBJECT(xml), "self");
    if (result == NULL) {
        g_set_error(&error, 0, 0, _("could not get self"));
        gpass_helper_error(error);
    }
    *self = result;
}

void
gpass_view_show_error_message(GPassView *self, const gchar *msg,
                              GtkWidget *focus_widget)
{
    GtkWidget *widget;
    gchar *str;
    
    widget = glade_xml_get_widget(self->xml, "message");
    str = g_strdup_printf("<span foreground=\"red\">%s</span>", msg);
    gtk_label_set_markup(GTK_LABEL(widget), str);
    g_free(str);
    gtk_widget_grab_focus(focus_widget);
}
