/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib/gi18n.h>

#include "assert-macros.h"

#include "gpass/configuration.h"
#include "gpass/entry-factory.h"

/***********************************************************
 *
 * initialize/terminate,  setup/teardown
 *
 ***********************************************************/
static void
initialize(void)
{
    g_type_init();
}

static void
terminate(void)
{
}

static void
setup(void)
{
}

static void
teardown(void)
{
    gpass_configuration_finalize();
}

/***********************************************************
 *
 * test case
 *
 ***********************************************************/
START_TEST(test_create_entry)
{
    GPassEntryFactory *factory;
    GPassEntry *entry;
    const gchar *type;
    GError *error;
    
    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);

    error = gpass_entry_factory_create_entry(factory, "general", &entry);
    ASSERT_NULL(error);
    g_object_get(entry, "type", &type, NULL);
    ASSERT_EQUAL_STRING("general", type);
    g_object_unref(entry);

    error = gpass_entry_factory_create_entry(factory, "folder", &entry);
    ASSERT_NULL(error);
    g_object_get(entry, "type", &type, NULL);
    ASSERT_EQUAL_STRING("folder", type);
    g_object_unref(entry);

    g_object_unref(factory);
}
END_TEST

START_TEST(test_create_default_entry)
{
    GPassEntryFactory *factory;
    GPassEntry *entry;
    const gchar *type;
    GError *error;
    
    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);

    error = gpass_entry_factory_create_default_entry(factory, &entry);
    ASSERT_NULL(error);
    g_object_get(entry, "type", &type, NULL);
    ASSERT_EQUAL_STRING("general", type);
    g_object_unref(entry);

    g_object_unref(factory);
}
END_TEST

START_TEST(test_create_cursor)
{
    GPassEntryFactory *factory;
    GPassEntryFactoryCursor *cursor;
    
    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);

    cursor = gpass_entry_factory_create_cursor(factory);
    ASSERT_TRUE(GPASS_IS_ENTRY_FACTORY_CURSOR(cursor));
    g_object_unref(cursor);

    g_object_unref(factory);
}
END_TEST

START_TEST(test_cursor__get)
{
    GPassEntryFactory *factory;
    GPassEntryFactoryCursor *cursor;
    gint index, max_index;
    const gchar *type, *nick, *launcher, *icon_id;
    gboolean can_have_child;
    GPassEntryClass *entry_class;
    
    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);
    cursor = gpass_entry_factory_create_cursor(factory);

    gpass_entry_factory_cursor_seek(cursor, "general");
    g_object_get(cursor,
                 "index", &index, "max_index", &max_index,
                 "type", &type, "nick", &nick, "launcher", &launcher,
                 "icon_id", &icon_id, "can_have_child", &can_have_child,
                 "entry_class", &entry_class, NULL);
    ASSERT_EQUAL_INT(1, index);
    ASSERT_EQUAL_INT(2, max_index);
    ASSERT_EQUAL_STRING("general", type);
    ASSERT_EQUAL_STRING(_("General"), nick);
    ASSERT_EQUAL_STRING("gnome-open @hostname@", launcher);
    ASSERT_EQUAL_STRING("gnome-stock-authentication", icon_id);
    ASSERT_FALSE(can_have_child);
    ASSERT_TRUE(GPASS_IS_ENTRY_CLASS(entry_class));
    
    gpass_entry_factory_cursor_seek(cursor, "folder");
    g_object_get(cursor,
                 "index", &index, "max_index", &max_index,
                 "type", &type, "nick", &nick, "launcher", &launcher,
                 "icon_id", &icon_id, "can_have_child", &can_have_child,
                 "entry_class", &entry_class, NULL);
    ASSERT_EQUAL_INT(0, index);
    ASSERT_EQUAL_INT(2, max_index);
    ASSERT_EQUAL_STRING("folder", type);
    ASSERT_EQUAL_STRING(_("Folder"), nick);
    ASSERT_EQUAL_STRING("", launcher);
    ASSERT_EQUAL_STRING("gtk-directory", icon_id);
    ASSERT_TRUE(can_have_child);
    ASSERT_TRUE(GPASS_IS_ENTRY_CLASS(entry_class));

    g_object_unref(cursor);
    g_object_unref(factory);
}
END_TEST

START_TEST(test_cursor__movement)
{
    GPassEntryFactory *factory;
    GPassEntryFactoryCursor *cursor;
    GError *error;
    
    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);
    cursor = gpass_entry_factory_create_cursor(factory);

    ASSERT_FALSE(gpass_entry_factory_cursor_is_done(cursor));
    gpass_entry_factory_cursor_next(cursor);
    ASSERT_FALSE(gpass_entry_factory_cursor_is_done(cursor));
    gpass_entry_factory_cursor_next(cursor);
    ASSERT_TRUE(gpass_entry_factory_cursor_is_done(cursor));

    error = gpass_entry_factory_cursor_seek(cursor, "bar");
    ASSERT_NOT_NULL(error);
    g_error_free(error);

    error = gpass_entry_factory_cursor_seek(cursor, "folder");
    ASSERT_NULL(error);
    ASSERT_FALSE(gpass_entry_factory_cursor_is_done(cursor));

    error = gpass_entry_factory_cursor_seek(cursor, "general");
    ASSERT_NULL(error);
    ASSERT_FALSE(gpass_entry_factory_cursor_is_done(cursor));
    
    g_object_unref(cursor);
    g_object_unref(factory);
}
END_TEST

/***********************************************************
 *
 * suite / main
 *
 ***********************************************************/
static Suite *
test_suite(void)
{
    Suite *s = suite_create("GPassEntryFactory");
    TCase *tc;
    
    tc = tcase_create("functions");
    suite_add_tcase(s, tc);
    tcase_add_checked_fixture(tc, setup, teardown);
    
    tcase_add_test(tc, test_create_entry);
    tcase_add_test(tc, test_create_default_entry);
    tcase_add_test(tc, test_create_cursor);
    tcase_add_test(tc, test_cursor__get);
    tcase_add_test(tc, test_cursor__movement);
    return s;
}

int
main(int argc, char *argv[])
{
    Suite *s;
    SRunner *sr;
    int nf;

    initialize();
    
    s = test_suite();
    sr = srunner_create(s);
    srunner_run_all(sr, CK_ENV);
    nf = srunner_ntests_failed(sr);
    srunner_free(sr);
    
    terminate();
    return (nf == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}
