/***************************************************************************
 *   Copyright (C) 2005 Novell, Inc.                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA           *
 ***************************************************************************/

#include <qstring.h>
#include <qstringlist.h>
#include <qdir.h>
#include <qclipboard.h>
#include <kfiledialog.h>
#include <kcmdlineargs.h>
#include <fcntl.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include <khelpmenu.h>
#include <kprocess.h>

#include <qtooltip.h>
#include <qdom.h>
#include <qtimer.h>

#include "searchdlg.h"
#include "kerryapp.h"
#include "configdialog.h"
#include <kaction.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <kprocess.h>
#include <kpassivepopup.h>
#include <kcombobox.h>

#define HISTORY_ITEMS_START_ID 100


KerryApplication::KerryApplication() : KUniqueApplication(),
    hitListWindow(0), sysTrayIcon(0)
{
}

KerryApplication::~KerryApplication()
{
  if (hitListWindow) {
    KConfig *config = KGlobal::config();
    config->writeEntry("History",hitListWindow->editSearch->historyItems());
    config->writeEntry("DialogSize",hitListWindow->size());
    config->sync();

    delete hitListWindow;
  }
}

int KerryApplication::newInstance()
{
    if (!hitListWindow)
        init(KGlobal::instance()->aboutData());

        KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
        if (args->isSet("show-searchdialog")) {
            hitListWindow->showSearchDialog();
        }

        if (args->count()==1)
           search(args->arg(0));

        args->clear();

    return KUniqueApplication::newInstance();
}

void KerryApplication::init(const KAboutData* /*about*/)
{
  if (hitListWindow)
      return;

  KConfig *config = KGlobal::config();
  config->setGroup("General");

  hitListWindow = new SearchDlg();
  QSize *defaultSize;
  defaultSize = new QSize(750, 650);
  hitListWindow->resize(config->readSizeEntry("DialogSize", defaultSize));
  delete defaultSize;
  connect(hitListWindow,SIGNAL(configure()),SLOT(configure()));

  hitListWindow->setDisplayAmount(config->readNumEntry("DisplayAmount", 5));
  hitListWindow->setSortOrder(config->readNumEntry("DefaultSortOrder",0));
  hitListWindow->editSearch->setHistoryItems(config->readListEntry("History"));

  sysTrayIcon = new KSystemTray(hitListWindow);
  KPopupMenu *menu = sysTrayIcon->contextMenu();
  connect(menu,SIGNAL(aboutToShow()),SLOT(aboutToShowSysTrayMenu()));
  connect(menu,SIGNAL(activated(int)),SLOT(historySelected(int)));

  menu->insertSeparator();
  menu->insertItem(SmallIconSet("history_clear"),
                   i18n("Clear Search History"), this,
                   SLOT(clearHistory()));

  menu->insertItem(SmallIconSet("configure"),
                   i18n("Configure Kerry..."), this,
                   SLOT(configure()));
/*
  KHelpMenu *helpmenu = new KHelpMenu(hitListWindow, about, false);
  menu->insertItem( SmallIconSet("help"), KStdGuiItem::help().text(), helpmenu->menu() );
*/
  globalKeys = new KGlobalAccel(this);
  KGlobalAccel* keys = globalKeys;
  keys->insert( "Program:kerry", i18n("Kerry Beagle Search") );

  KShortcut showDialogShortcut = KShortcut(ALT+Key_Space);
//  showDialogShortcut.append( KKey( Key_F12 ) );
  keys->insert( "Show Kerry Dialog", i18n("Show Search Dialog"), QString::null, showDialogShortcut, showDialogShortcut, hitListWindow, SLOT(showSearchDialog()) );
  keys->insert( "Search Primary Selection with Kerry", i18n("Search Primary Selection"), QString::null, CTRL+ALT+Key_Space, CTRL+ALT+Key_Space, this, SLOT(searchPrimarySelection()) );
  globalKeys->readSettings();
  globalKeys->updateConnections();

  sysTrayIcon->setPixmap(sysTrayIcon->loadIcon("kerry_systemtray"));
  QToolTip::add(sysTrayIcon, i18n("Kerry Beagle Search (%1)").arg(globalKeys->shortcut("Show Kerry Dialog").seq(0).toString()));

  sysTrayIcon->show();

  sysTrayIcon->actionCollection()->action("file_quit")->setShortcut(KShortcut());
  disconnect(sysTrayIcon->actionCollection()->action("file_quit"), SIGNAL(activated()), sysTrayIcon, SLOT(maybeQuit()));
  connect(sysTrayIcon->actionCollection()->action("file_quit"), SIGNAL(activated()), this, SLOT(quitKerry()));

  QTimer::singleShot( 1000, this, SLOT(checkBeagleBuildIndex()));
}

void KerryApplication::search(const QString& text)
{
  if (hitListWindow)
    hitListWindow->search(text);
}

void KerryApplication::aboutToShowSysTrayMenu()
{
  KPopupMenu *menu = sysTrayIcon->contextMenu();

  for (int id=HISTORY_ITEMS_START_ID;id<=HISTORY_ITEMS_START_ID+MAX_HISTORY_ITEMS;id++)
    menu->removeItem(id);

  QStringList searches = hitListWindow->editSearch->historyItems();
  if (searches.count()==0) {
    menu->insertItem(i18n("<No Recent Searches>"),HISTORY_ITEMS_START_ID,1);
    menu->setItemEnabled(HISTORY_ITEMS_START_ID,false);
    return;
  }

  for (int i=0;i<(int)searches.count();i++)
    menu->insertItem(searches[i],i+HISTORY_ITEMS_START_ID,i+1);
}

void KerryApplication::historySelected(int id)
{
  if (id<HISTORY_ITEMS_START_ID)
    return;

  if (hitListWindow)
    hitListWindow->search(sysTrayIcon->contextMenu()->text(id));
}

void KerryApplication::searchPrimarySelection()
{
  QApplication::clipboard()->setSelectionMode( true );
  QString text = QApplication::clipboard()->text();
  if (!text.isEmpty() && hitListWindow)
    hitListWindow->search(text);
}

void KerryApplication::quitKerry()
{
    int autoStart = KMessageBox::questionYesNoCancel( 0L, i18n("Should Kerry start automatically\nwhen you login?"), i18n("Automatically Start Kerry?"), i18n("&Start"), i18n("&Do Not Start") );

  KConfig *config = KGlobal::config();
  config->setGroup("General");
  if ( autoStart == KMessageBox::Yes ) {
    config->writeEntry("AutoStart", true);
  } else if ( autoStart == KMessageBox::No) {
    config->writeEntry("AutoStart", false);
  } else  // cancel chosen don't quit
    return;
  config->writeEntry("History",hitListWindow->editSearch->historyItems());
  config->sync();

  if (hitListWindow)
  	hitListWindow->hide();

  KProcess *proc = new KProcess;
  *proc << "beagle-shutdown";
  if (!proc->start())
      kdDebug() << "Couldn't run beagle-shutdown." << endl;

  qApp->closeAllWindows();
  qApp->quit();
}

void KerryApplication::clearHistory()
{
   hitListWindow->editSearch->clearHistory();
   KConfig *config = KGlobal::config();
   config->writeEntry("History",QString::null);
   config->sync();
}

void KerryApplication::readIndexConfig(bool &indexHomeDir, bool &indexOnBattery, QStringList &roots, QStringList &excludeTypes, QStringList &excludeValues)
{
    indexHomeDir = true;
    indexOnBattery = true;
    roots.clear();
    excludeTypes.clear();
    excludeValues.clear();

    QFile file( QDir::home().absPath()+"/.beagle/config/indexing.xml" );
    if ( !file.open( IO_ReadOnly ) )
        return;

    QDomDocument doc( "mydocument" );
    if ( !doc.setContent( &file ) ) {
        file.close();
        return;
    }
    file.close();

    QDomElement docElem = doc.documentElement();

    QDomNode n = docElem.firstChild();
    while( !n.isNull() ) {
        QDomElement e = n.toElement();
        if( !e.isNull() ) {
            if (e.tagName()=="IndexHomeDir")
               indexHomeDir = e.text()=="true";
            if (e.tagName()=="IndexOnBattery")
               indexOnBattery = e.text()=="true";
            else if (e.tagName()=="Roots") {
               QDomNode ro = n.firstChild();
               while( !ro.isNull() ) {
                  QDomElement exel = ro.toElement();
                  if( !exel.isNull() )
                    roots << exel.text();
                  ro = ro.nextSibling();
                }
            }
            else if (e.tagName()=="Excludes") {
                QDomNode ex = n.firstChild();
                while( !ex.isNull() ) {
                  QDomElement exel = ex.toElement();
                  if( !exel.isNull() ) {
                    excludeTypes << exel.attribute("Type");
		    excludeValues << exel.attribute("Value");
                  }
                  ex = ex.nextSibling();
                }
            }
        }
        n = n.nextSibling();
    }
}

bool KerryApplication::saveIndexConfig(bool indexHomeDir, bool indexOnBattery, QStringList roots, QStringList excludeTypes, QStringList excludeValues)
{
  QDir beagleDir(QDir::home().absPath()+"/.beagle");
  if (!beagleDir.exists())
    beagleDir.mkdir(QDir::home().absPath()+"/.beagle");

  QDir beagleConfigDir(QDir::home().absPath()+"/.beagle/config");
  if (!beagleConfigDir.exists())
    beagleConfigDir.mkdir(QDir::home().absPath()+"/.beagle/config");

  QFile configFile( QDir::home().absPath()+"/.beagle/config/indexing.xml" );
  if ( !configFile.open( IO_WriteOnly ) )
        return false;

  QDomDocument doc( QString::null );
  doc.appendChild( doc.createProcessingInstruction(
                   "xml", "version=\"1.0\" encoding=\"UTF-8\"" ) );

  QDomElement root = doc.createElement( "IndexingConfig" );
  root.setAttribute("xmlns:xsd","http://www.w3.org/2001/XMLSchema");
  root.setAttribute("xmlns:xsi","http://www.w3.org/2001/XMLSchema-instance");
  doc.appendChild( root );

  QDomElement rootselement = doc.createElement( "Roots" );
  root.appendChild(rootselement);

  for ( QStringList::Iterator it = roots.begin(); it != roots.end(); ++it ) {
    QDomElement tag = doc.createElement( "Root" );
    rootselement.appendChild( tag );

    QDomText t = doc.createTextNode( *it );
    tag.appendChild( t );
  }

  QDomElement tag = doc.createElement( "IndexHomeDir" );
  root.appendChild( tag );

  QDomText t = doc.createTextNode( indexHomeDir ? "true" : "false" );
  tag.appendChild( t );

  tag = doc.createElement( "IndexOnBattery" );
  root.appendChild( tag );

  t = doc.createTextNode( indexOnBattery ? "true" : "false" );
  tag.appendChild( t );

  QDomElement excludeselement = doc.createElement( "Excludes" );
  root.appendChild(excludeselement);

  QStringList::Iterator it_types = excludeTypes.begin();
  for ( QStringList::Iterator it = excludeValues.begin(); it != excludeValues.end(); ++it ) {
    QDomElement t = doc.createElement( "ExcludeItem" );
    t.setAttribute("Type", *it_types);
    t.setAttribute("Value", *it);
    excludeselement.appendChild( t );

    ++it_types;
  }

  QTextStream stream( &configFile );
  stream << doc.toString();
  configFile.close();

  return true;
}

QStringList KerryApplication::readDisabledBackends()
{
    QStringList disabledBackends;

    QDomDocument doc( "mydocument" );
    QFile file( QDir::home().absPath()+"/.beagle/config/daemon.xml" );
    if ( !file.open( IO_ReadOnly ) )
        return disabledBackends;

    if ( !doc.setContent( &file ) ) {
        file.close();
        return disabledBackends;
    }
    file.close();

    QDomElement docElem = doc.documentElement();

    QDomNode n = docElem.firstChild();
    while( !n.isNull() ) {
        QDomElement e = n.toElement();
        if( !e.isNull() ) {
            if (e.tagName()=="DeniedBackends") {
               QDomNode ro = n.firstChild();
               while( !ro.isNull() ) {
                  QDomElement exel = ro.toElement();
                  if( !exel.isNull() )
                    disabledBackends << exel.text();
                  ro = ro.nextSibling();
                }
            }
        }
        n = n.nextSibling();
    }
    return disabledBackends;
}

bool KerryApplication::saveDisabledBackends(QStringList disabledBackends)
{
  QDir beagleDir(QDir::home().absPath()+"/.beagle");
  if (!beagleDir.exists())
    beagleDir.mkdir(QDir::home().absPath()+"/.beagle");

  QDir beagleConfigDir(QDir::home().absPath()+"/.beagle/config");
  if (!beagleConfigDir.exists())
    beagleConfigDir.mkdir(QDir::home().absPath()+"/.beagle/config");

  QFile configFile( QDir::home().absPath()+"/.beagle/config/daemon.xml" );

  QDomDocument doc( QString::null );
  QDomElement root;

  if (configFile.exists()) {
    if ( !configFile.open( IO_ReadOnly ) )
      return false;

    if ( !doc.setContent( &configFile ) ) {
        configFile.close();
        return false;
    }
    configFile.close();

    root = doc.documentElement();

    QDomNode n = root.firstChild();
    while( !n.isNull() ) {
        QDomElement e = n.toElement();
        if( !e.isNull() )
            if (e.tagName()=="DeniedBackends")
              root.removeChild( e );
        n = n.nextSibling();
    }
  }
  else {
    doc.appendChild( doc.createProcessingInstruction(
                     "xml", "version=\"1.0\" encoding=\"UTF-8\"" ) );

    root = doc.createElement( "DaemonConfig" );
    root.setAttribute("xmlns:xsd","http://www.w3.org/2001/XMLSchema");
    root.setAttribute("xmlns:xsi","http://www.w3.org/2001/XMLSchema-instance");
    doc.appendChild( root );
  }

  QDomElement rootselement = doc.createElement( "DeniedBackends" );
  root.appendChild(rootselement);

  for ( QStringList::Iterator it = disabledBackends.begin(); it != disabledBackends.end(); ++it ) {
    QDomElement tag = doc.createElement( "anyType" );
    tag.setAttribute("xsi:type","xsd:string");
    rootselement.appendChild( tag );

    QDomText t = doc.createTextNode( *it );
    tag.appendChild( t );
  }

  configFile.remove();
  if ( !configFile.open( IO_WriteOnly ) )
    return false;

  QTextStream stream( &configFile );
  stream << doc.toString();
  configFile.close();

  return true;
}

void KerryApplication::configure()
{
  ConfigDialog *dlg = new ConfigDialog( globalKeys );

  KConfig *config = KGlobal::config();
  config->setGroup("Beagle");
  dlg->setStartBeagle(config->readBoolEntry("AutoStart",false));
  config->setGroup("General");
  dlg->setDefaultSortOrder(config->readNumEntry("DefaultSortOrder",0));

  dlg->setMaxResultsDisplayed(hitListWindow->getDisplayAmount());

  bool indexHomeDir, indexOnBattery;
  QStringList roots, excludeTypes, excludeValues;
  readIndexConfig(indexHomeDir, indexOnBattery, roots, excludeTypes, excludeValues);

  dlg->setIndexHome(indexHomeDir);
  dlg->setIndexOnBattery(indexOnBattery);
  dlg->setRoots(roots);
  dlg->setExcludes(excludeTypes, excludeValues);

  m_availableBackends.clear();
  KProcess *proc = new KProcess;
  connect(proc, SIGNAL(receivedStdout(KProcess *, char *, int)), SLOT(gotAvailableBackends(KProcess *, char *, int)));
  *proc << "beagled" << "--list-backends";
  if (!proc->start(KProcess::Block,KProcess::Stdout))
    kdError("Could not ask Beagle daemon for available backends.");
  dlg->addAvailableBackends(m_availableBackends);

  dlg->setDisabledBackends(readDisabledBackends());

  if ( dlg->exec() == QDialog::Accepted ) {
        dlg->commitShortcuts();
        // the keys need to be written to kdeglobals, not kickerrc --ellis, 22/9/02
        globalKeys->writeSettings(0, true);
        globalKeys->updateConnections();

	QToolTip::remove(sysTrayIcon);
	QToolTip::add(sysTrayIcon, i18n("Kerry Beagle Search (%1)").arg(globalKeys->shortcut("Show Kerry Dialog").seq(0).toString()));

        config->setGroup("General");
        const int sortOrder = dlg->getDefaultSortOrder();
        config->writeEntry("DefaultSortOrder",sortOrder);
        hitListWindow->setSortOrder(sortOrder);

	const int maxresults = dlg->getMaxResultsDisplayed();
        hitListWindow->setDisplayAmount(maxresults);
        config->writeEntry("DisplayAmount",maxresults);

	config->setGroup("Beagle");
	config->writeEntry("AutoStart",dlg->getStartBeagle());
	config->sync();

        saveIndexConfig(dlg->getIndexHome(), dlg->getIndexOnBattery(), dlg->getRoots(), dlg->getTypes(), dlg->getValues());
        saveDisabledBackends(dlg->getDisabledBackends());

        KProcess *proc = new KProcess;
        *proc << "beagle-config";
        *proc << "--beagled-reload-config";
        if (!proc->start())
          kdError("Could not make Beagle reload its config.");
  }

  delete dlg;
}

void KerryApplication::gotAvailableBackends(KProcess*, char *buffer, int len)
{
    QString myBuf = QString::fromLatin1(buffer, len);
    if (myBuf.startsWith("User:")) {
      QStringList list = QStringList::split('\n',myBuf);
      for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it ) 
        if ((*it).startsWith(" - "))
          m_availableBackends << (*it).mid(3);
    }
}

void KerryApplication::checkBeagleBuildIndex()
{
    QDir dir("/tmp", ".beagleindexwapi*");
    dir.setFilter(QDir::Dirs|QDir::Hidden);

    if (dir.entryList().isEmpty())
        return;

    KPassivePopup::message(KPassivePopup::Boxed, i18n("System May Be Slower Than Usual"), i18n("The daily running process for updating the system\nwide Beagle documentation index was detected."), BarIcon("kerry"), sysTrayIcon); 
}

#include "kerryapp.moc"
