/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#ifndef KFTPSERVERMANAGEMENTKFTPSERVERMANAGEMENT_H
#define KFTPSERVERMANAGEMENTKFTPSERVERMANAGEMENT_H

#define KFTP_SERVERADMINISTRATION_VERSION 2

#include <qobject.h>
#include <qdom.h>

class KListViewItem;

namespace KFTPWidgets {
namespace ServerManagement {
  class ListView;
  class ListViewItem;
}
}

namespace KFTPServerManagement {

/**
 * @author Blacknator
 *
 * @brief contains the xml data for Server Management
 *
 * This class is a non gui class which read and writes the Server Management data from/to a xml file.<br>
 * Also it fills a server list view with all commands and sub categories
 */
class Manager : public QObject
{
Q_OBJECT
public:
    Manager(QObject *parent = 0, const char *name = 0);
    ~Manager();

    /** read the xml file
     * @param filename xml file to read
     */
    void load(const QString& filename);

    /** saves the xml file */
    void save();

    /** sets default data, overwrites current xml data/file */
    void setDefault();

    /** Fills a server list view with all items and sub categories
     * @param tree ListView to fill
     * @param parent parent node, needed to fill sub categories, is called from this method itself
     * @param item parent item, needed to fill sub categories, is called from this method itself
     */
    void guiPopulateServerManagementTree(KFTPWidgets::ServerManagement::ListView *tree, QDomNode parent = QDomNode(), KFTPWidgets::ServerManagement::ListViewItem *item = 0);

    /** the global xml file for server management */
    QDomDocument m_document;
protected:
    /**
     * add a new category to the parent element
     * @param parent the parent element
     * @param categoryText the name of the new category
     * @return the new category which was created in this method
     *
     * \code
     * QDomDocument doc;
     * addCategory(doc, "statistic");
     * \endcode
     * This is similar to:
     * \code
     * QDomDocument doc;
     * QDomElement category;
     * category = doc.createElement("category");
     * category.setAttribute("statistic");
     * doc.appendChild(category);
     * \endcode
     */
    inline QDomElement addCategory(QDomElement &parent, QString categoryText);

    /** add a new element to the element parent
     * @param parent the parent element
     * @param elementText the name of the new element
     * @return the new element which was created in this method
     *
     * \code
     * QDomDocument doc;
     * addCommand(doc, "site idle"); // <command name="site idle" >
     * \endcode
     * This is similar to:
     * \code
     * QDomDocument doc;
     * QDomElement element;
     * element = doc.createElement("command");
     * element.setAttribute("name", "site idle");
     * doc.appendChild(element);
     * \endcode
     */
    inline QDomElement addCommand(QDomElement &parent, QString elementText);

    /** add a text to the element parent
     * @param parent the parent element
     * @param elementText the name of the new QDomElement
     * @param textNodetext the text inside of QDomElement
     *
     * \code
     * addTextToElement(element, "Description", "Display your idle time"); // <Description>Display your idle time</Description>
     * \endcode
     * This is similar to
     * \code
     * QDomDocument doc;
     * QDomElement tag;
     * QDomText text;
     * tag = doc.createElement("Description");
     * text = doc.createTextNode("Display your idle time");
     * tag.appendChild(text);
     * doc.appendChild(tag);
     * \endcode
     */
    inline void addTextToElement(QDomElement &parent, QString elementText, QString textNodetext);
private:
    /** xml file name */
    QString m_filename;
};

} // namespace KFTPServerManagement

extern KFTPServerManagement::Manager *FTPServerManagement;

#endif
