/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#ifndef KFTPFILEDIRLISTVIEW_H
#define KFTPFILEDIRLISTVIEW_H

#include "engine/thread.h"

#include "widgets/listview.h"

#include <klistview.h>
#include <kdirlister.h>
#include <kfileitem.h>
#include <kurl.h>
#include <kurldrag.h>

#include <qvaluelist.h>

namespace KFTPSession {
  class Session;
}

namespace KFTPWidgets {

namespace Browser {

class History
{
public:
    History();
    
    void newEntry(const KURL &url);
    KURL getCurrent();
    QString getSelected();

    void goBack();
    void goForward();

    QValueList<QPair<KURL, QString> > m_history;
    QValueList<QPair<KURL, QString> >::iterator m_historyPos;
};

class TreeView;
class ListViewItem;

typedef QPtrList<ListViewItem> ListItems;

/**
 * This class represents a list view that can show either some local or
 * remote URL. For retrieval of remote URLs the underlying session is used
 * and for local ones, KDirLister is used.
 *
 * @author Jernej Kos
 */
class ListView : public KFTPWidgets::ListView
{
Q_OBJECT
friend class Actions;
friend class View;
friend class KFTPSession::Session;
public:
    /**
     * Location info specification.
     */
    enum LocationInfo {
      Up,
      Back,
      Forward,
      Reload,
      Home
    };
    
    /**
     * Class constructor.
     *
     * @param parent The parent widget
     * @param session The session associated with this browser view
     */
    ListView(QWidget *parent, KFTPSession::Session *session);
    
    /**
     * Class destructor.
     */
    ~ListView();
    
    /**
     * Open an URL in the list view.
     *
     * @param url The URL to open (note that it has to be accessible trough the given session)
     * @param addHistory Should this URL be added to history combo or not
     */
    void openUrl(const KURL &url, bool addHistory = true);
    
    /**
     * Get the currently open URL.
     *
     * @return The currently open URL
     */
    KURL getCurrentUrl() { return m_curURL; }
    
    /**
     * Set this list view's "companion" (aka the list view on the other
     * side - in another session).
     *
     * @param companion A valid ListView
     */
    void setCompanion(ListView *companion) { m_companion = companion; }
    
    /**
     * Set the tree view that we can use.
     *
     * @param tree A valid TreeView pointer
     */
    void setTreeView(TreeView *tree);
    
    /**
     * Set the URL that is considered home. This URL is accessed when the user
     * clicks the home button.
     *
     * @param url The URL to be considered home
     */
    void setHomeUrl(const KURL &url) { m_homeURL = url; }
    
    /**
     * Toggle visibility of hidden files. Note that this will not affect the view
     * until reload.
     *
     * @param value True if the hidden files should be shown, false otherwise
     */
    void setShowHidden(bool value) { m_showHiddenFiles = value; }

    /**
     * Reset this view using a specified URL. You should only call this method
     * when chaning from one URL to the other (like from local to remote).
     *
     * @param url The new URL
     * @param rootText Text of the root item in the tree view
     */
    void resetView(const KURL &url, const QString &rootText = 0);
    
    /**
     * Move in some manner.
     *
     * @param location A valid LocationInfo
     */
    void moveTo(LocationInfo location);
    
    /**
     * Check if we can move in some manner.
     *
     * @param location A valid LocationInfo
     * @return True if a call to moveTo using location can be made
     */
    bool canMoveTo(LocationInfo location);

    /**
     * Get a current selection of URLs.
     *
     * @return A valid KURL::List of currently selected items
     */
    KURL::List getCurrentSelection();
    
    /**
     * Get a current selection of list items.
     */
    ListItems getCurrentSelectionLI();
    
    /**
     * Can this list view execute local actions.
     */
    bool canLocalActions();
    
    /**
     * Get the drag object used for drag and drop, copying and pasting.
     *
     * @return A valid QDragObject instance
     */
    QDragObject *dragObject();
private:
    KFTPSession::Session *m_session;
    KFTPEngine::Thread *m_ftpClient;
    KDirLister *m_dirLister;

    ListView *m_companion;
    TreeView *m_tree;

    // Position information
    KURL m_homeURL;
    KURL m_curURL;
    KURL m_reqURL;
    History m_history;

    // Options
    bool m_addHistory;
    bool m_showHiddenFiles;
    bool m_shouldDisableResize;
    bool m_autoResizeEnabled;

    /* Drag & drop support */
    QTimer *m_dragOpenTimer;
    QListViewItem *m_dropItem;
    QDragObject *m_dragObject;
    
    bool canTransfer(ListView *caller = 0);
    int transferMode();
    
    /**
     * Reselects the folder that was previously entered by scrolling the current
     * listview.
     */
    void smartFolderReselect();
    
    /**
     * Calculates new widths for all the columns.
     */
    void updateColumnWidths();
    
    /**
     * This method gets called when the widget is resized. It is used for automatic
     * resize handling.
     *
     * @param width The new width
     * @param height The new height
     */
    void resizeContents(int width, int height);
    
    void startDrag();
    bool acceptDrag(QDropEvent *e);
    void contentsDragEnterEvent(QDragEnterEvent *e);
    void contentsDragMoveEvent(QDragMoveEvent *e);
    void contentsDragLeaveEvent(QDragLeaveEvent*);
    void contentsDropEvent(QDropEvent *e);
private slots:
    void slotDirListerNewItems(const KFileItemList &items);
    void slotDirListerDeleteItem(KFileItem *item);
    void slotDirListerClear();
    void slotDirListerRefreshItems(const KFileItemList &items);

    void slotFtpEngineEvent(KFTPEngine::Event *event);

    void slotListClicked();
    void slotTreeClicked(const KURL url);
    void slotItemRenamed(QListViewItem *item, const QString &str, int col);

    void slotContextMenu(QListViewItem*, const QPoint&);

    void slotDragTimeout();
    
    /**
     * This slot gets called when a section in listview's header is resized. This
     * is needed for proper automatic resize handling.
     *
     * @param section The section that was resized
     */
    void slotHeaderResized(int section);
signals:
    /**
     * This signal is emitted when items change.
     */
    void itemsChanged();
};

/**
 * This class represents a single item inside the list view. It is associated
 * with a specific DirectoryEntry that holds all it's properties. You can also create
 * instances of this class directly from local KFileItems.
 *
 * @author Jernej Kos
 */
class ListViewItem : public KListViewItem
{
friend class ListView;
friend class Actions;
public:
    /**
     * Class constructor.
     *
     * @param parent A parent list view
     */
    ListViewItem(ListView *parent);
    
    /**
     * Class constructor.
     *
     * @param parent A parent list view item
     */
    ListViewItem(ListViewItem *parent);

    /**
     * Class constructor.
     *
     * @param parent A parent list view
     * @param item A KFileItem representing the local file
     */
    ListViewItem(ListView *parent, KFileItem *item);
    
    /**
     * Class constructor.
     *
     * @param parent A parent list view item
     * @param item A KFileItem representing the local file
     */
    ListViewItem(ListViewItem *parent, KFileItem *item);

    /**
     * Class constructor.
     *
     * @param parent A parent list view
     * @param item A DirectoryEntry representing the remote file
     * @param url Full URL to this file
     */
    ListViewItem(ListView *parent, const KFTPEngine::DirectoryEntry &item, const KURL &url);
    
    /**
     * Get the directory entry associated with this item.
     *
     * @return A valid KFTPEngine::DirectoryEntry
     */
    KFTPEngine::DirectoryEntry getDirectoryEntry() const { return m_dirEntry; }
    
    /**
     * Set a text color for this item.
     *
     * @param color Text color
     */
    void setColor(QColor color) { m_textColor = color; }
protected:
    /**
     * Read data from KFileItem into DirectoryEntry.
     *
     * @param item A KFileItem representing the local file
     */
    void readFromFileItem(KFileItem *item);
    
    /**
     * Populate columns from the directory entry.
     */
    void createColumns();

    /**
     * @overload
     * Reimplemented from KListViewItem to implement proper comparation.
     */
    int compare(QListViewItem *i, int col, bool asc) const;
    
    /**
     * @overload
     * Reimplemented from KListView to implement nicer items.
     */
    void paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int alignment);
private:
    QColor m_textColor;
    KFTPEngine::DirectoryEntry m_dirEntry;
    KURL m_url;

    KFileItem *m_fileItem;
};

}

}

#endif
