/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#ifndef KFTPQUEUESCHEDULER_H
#define KFTPQUEUESCHEDULER_H

#include "kftpqueue.h"

#include <qobject.h>

namespace KFTPQueue {

/**
 * This class provides a bandwidth allocation scheduler. It is used whenever a
 * speed limit (download or upload) is in place. It will try to divide the
 * bandwidth among transfers the best as possible.
 *
 * @author Jernej Kos
 */
class Scheduler : public QObject
{
Q_OBJECT
public:
    static Scheduler *self();
    ~Scheduler();
    
    /**
     * Register the transfer with the scheduler. Every download or upload transfer
     * should register itself before the actual transfer starts.
     *
     * @param transfer The transfer object
     */
    void registerTransfer(TransferFile *transfer);
protected:
    Scheduler();
    static Scheduler *m_self;
private:
    QPtrList<TransferFile> m_transfers;
    QMap<long, filesize_t> m_limits;
    
    filesize_t m_unallocatedUp;
    filesize_t m_unallocatedDown;
    
    void deallocateUpload(filesize_t bandwidth) { m_unallocatedUp += bandwidth; }
    filesize_t allocateUpload();
    
    void deallocateDownload(filesize_t bandwidth) { m_unallocatedDown += bandwidth; }
    filesize_t allocateDownload();
    
    int countTransfers(TransferType type);
private slots:
    void slotConfigUpdate();
    
    void slotTransferUpdated();
    void slotTransferDone(long id);
    void slotTransferDestroyed(QObject *object);
};

}

#endif
