/* Copyright (C) 2003 Mark Kretschmann <markey@web.de>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public
* License as published by the Free Software Foundation version 2.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*
*/

#include "kfile_modplug.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/mman.h>

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <math.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string>
#include <map>

#include <kdebug.h>
#include <kfilemetainfo.h>
#include <kurl.h>
#include <kprocess.h>
#include <klocale.h>
#include <kgenericfactory.h>
#include <kstringvalidator.h>

#include <qcstring.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qdatetime.h>
#include <qdict.h>
#include <qmessagebox.h>
#include <qstring.h>
#include <qvalidator.h>
#include <qvariant.h>

#include "../libmodplug/stdafx.h"
#include "../libmodplug/sndfile.h"


typedef KGenericFactory<KModplugPlugin> ModplugFactory;
K_EXPORT_COMPONENT_FACTORY( kfile_modplug, ModplugFactory( "kfile_modplug" ) );

KModplugPlugin::KModplugPlugin( QObject *parent, const char *name,
                                const QStringList &preferredItems ) : KFilePlugin( parent, name, preferredItems )
{
    KFileMimeTypeInfo * info = addMimeTypeInfo( "audio/x-mod" );
    KFileMimeTypeInfo::GroupInfo* group;
    KFileMimeTypeInfo::ItemInfo* item;

    // GROUP: General
    group = addGroupInfo( info, "General", i18n( "General" ) );
    setAttributes( group, KFileMimeTypeInfo::Addable );

    item = addItemInfo( group, "Title", i18n( "Title" ), QVariant::String );
    setHint( item, KFileMimeTypeInfo::Name );

    item = addItemInfo( group, "Type", i18n( "Type" ), QVariant::String );

    item = addItemInfo( group, "Length", i18n( "Length" ), QVariant::Int );
    setAttributes( item, KFileMimeTypeInfo::Cummulative );
    setUnit( item, KFileMimeTypeInfo::Seconds );

    item = addItemInfo( group, "Channels", i18n( "Channels" ), QVariant::UInt );

    item = addItemInfo( group, "Patterns", i18n( "Patterns" ), QVariant::UInt );

    item = addItemInfo( group, "Instruments", i18n( "Instruments" ), QVariant::UInt );

    item = addItemInfo( group, "Speed", i18n( "Speed" ), QVariant::UInt );

    item = addItemInfo( group, "Tempo", i18n( "Tempo" ), QVariant::UInt );

    // GROUP: Message
    group = addGroupInfo( info, "Message", i18n( "Message" ) );

    item = addItemInfo( group, "Message", i18n( "Message" ), QVariant::String );
    setAttributes( item, KFileMimeTypeInfo::SqueezeText );
    setHint( item, KFileMimeTypeInfo::Description );
}


bool KModplugPlugin::readInfo( KFileMetaInfo& info, uint )
{
    int fd = ::open( info.path().latin1(), O_RDONLY );
    struct stat lStat;

    // open and mmap the file
    if ( fd == -1 )
    {
        QMessageBox::information( 0, "kfile_modplug", "ERROR: fd == -1", QMessageBox::Ok );
        return false;
    }

    ::fstat( fd, &lStat );
    int size = lStat.st_size;

    unsigned char *map = ( unsigned char* ) ::mmap( 0, size, PROT_READ, MAP_PRIVATE, fd, 0 );

    if ( map == NULL )
    {
        ::close( fd );
        QMessageBox::information( 0, "kfile_modplug", "ERROR: mmap error", QMessageBox::Ok );
        return false;
    }

    CSoundFile mod;
    mod.Create( map, size );

    KFileMetaInfoGroup group;

    group = appendGroup( info, "General" );
    appendItem( group, "Speed", static_cast<uint>( mod.GetMusicSpeed() ) );
    appendItem( group, "Tempo", static_cast<uint>( mod.GetMusicTempo() ) );
    appendItem( group, "Instruments", static_cast<uint>( mod.GetNumInstruments() ) );
    appendItem( group, "Patterns", static_cast<uint>( mod.GetNumPatterns() ) );
    appendItem( group, "Channels", static_cast<uint>( mod.GetNumChannels() ) );
    appendItem( group, "Length", static_cast<int>( mod.GetSongTime() ) );
    appendItem( group, "Type", typeToString( mod.GetType() ) );
    appendItem( group, "Title", QString( mod.GetTitle() ) );

    group = appendGroup( info, "Message" );
    char message[ 4000 ];
    mod.GetSongComments( message, 4000, 4000 );
    appendItem( group, "Message", QString( message ) );

    ::munmap( map, size );
    ::close( fd );
    return true;
}


QString KModplugPlugin::typeToString( unsigned int type )
{
    QString str;

    switch ( type )
    {
        case MOD_TYPE_MOD:
            str = "ProTracker";
            break;
        case MOD_TYPE_S3M:
            str = "Scream Tracker 3";
            break;
        case MOD_TYPE_XM:
            str = "Fast Tracker 2";
            break;
        case MOD_TYPE_IT:
            str = "Impulse Tracker";
            break;
        case MOD_TYPE_MED:
            str = "OctaMed";
            break;
        case MOD_TYPE_MTM:
            str = "MTM";
            break;
        case MOD_TYPE_669:
            str = "669 Composer / UNIS 669";
            break;
        case MOD_TYPE_ULT:
            str = "ULT";
            break;
        case MOD_TYPE_STM:
            str = "Scream Tracker";
            break;
        case MOD_TYPE_FAR:
            str = "Farandole";
            break;
        case MOD_TYPE_AMF:
            str = "ASYLUM Music Format";
            break;
        case MOD_TYPE_AMS:
            str = "AMS module";
            break;
        case MOD_TYPE_DSM:
            str = "DSIK Internal Format";
            break;
        case MOD_TYPE_MDL:
            str = "DigiTracker";
            break;
        case MOD_TYPE_OKT:
            str = "Oktalyzer";
            break;
        case MOD_TYPE_DMF:
            str = "Delusion Digital Music Fileformat (X-Tracker)";
            break;
        case MOD_TYPE_PTM:
            str = "PolyTracker";
            break;
        case MOD_TYPE_DBM:
            str = "DigiBooster Pro";
            break;
        case MOD_TYPE_MT2:
            str = "MT2";
            break;
        case MOD_TYPE_AMF0:
            str = "AMF0";
            break;
        case MOD_TYPE_PSM:
            str = "PSM";
            break;
        default:
            str = "Unknown";
            break;
    }
    return str;
}


#include "kfile_modplug.moc"
