/*
 * debug.c - NILFS debug code and Proc-fs handling code.
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * debug.c,v 1.40 2006/07/14 07:04:56 ryusuke Exp
 *
 * Written by Amagai Yoshiji <amagai@osrg.net>,
 *            Ryusuke Konishi <ryusuke@osrg.net>
 */

#include <linux/buffer_head.h>
#include <linux/mpage.h>
#include <linux/writeback.h>
#include <linux/blkdev.h>
#include <linux/proc_fs.h>
#include <linux/string.h>
#include <linux/parser.h>
#include "nilfs.h"

#if defined(NILFS_SEMAPHORE_DEBUG) || defined(NILFS_SPINLOCK_DEBUG) || defined(NILFS_LOCK_BUFFER_DEBUG)
spinlock_t nilfs_sem_h_lock = SPIN_LOCK_UNLOCKED;
struct nilfs_sem_holders holders_a[HOLDERS_N];
#endif

static int proc_calc_metrics(char *page, char **start, off_t off,
			     int count, int *eof, int len);

/*
 * Counters
 */
atomic_t nilfs_getblkbh_n;
atomic_t nilfs_putblkbh_n;
#ifdef NILFS_BH_COUNT_DETAIL
atomic_t nilfs_putblkbh_nilfs_allocated_n;
atomic_t nilfs_putblkbh_bbt_node_n;
atomic_t nilfs_putblkbh_ibt_node_n;
atomic_t nilfs_putblkbh_new_node_n;
atomic_t nilfs_putblkbh_inactive_node_n;
atomic_t nilfs_putblkbh_partial_node_n;
#endif

#ifdef NILFS_PAGE_COUNT
atomic_t nilfs_allocated_inode_page_n;
atomic_t nilfs_released_inode_page_n;
atomic_t nilfs_released_file_page_n;
#endif
atomic_t nilfs_released_inode_n;
atomic_t nilfs_allocated_inode_n;

#ifdef NILFS_KMALLOC_COUNT
atomic_t nilfs_kmalloc_n;
atomic_t nilfs_kfree_n;
#endif
#ifdef NILFS_BIO_COUNT
atomic_t nilfs_get_bio_n;
atomic_t nilfs_put_bio_n;
#endif
#ifdef NILFS_SLAB_COUNT
atomic_t nilfs_cache_alloc_n;
atomic_t nilfs_cache_free_n;
#endif
#ifdef NILFS_KMAP_COUNT
atomic_t nilfs_kmap_n;
atomic_t nilfs_kunmap_n;
#endif
#ifdef NILFS_ALLOCPAGE_COUNT
atomic_t nilfs_alloc_page_n;
atomic_t nilfs_free_page_n;
#endif

void nilfs_init_counters(void)
{
	/* nputblk */
	atomic_set(&nilfs_getblkbh_n, 0);
	atomic_set(&nilfs_putblkbh_n, 0);
#ifdef NILFS_BH_COUNT_DETAIL
	atomic_set(&nilfs_putblkbh_nilfs_allocated_n, 0);
	atomic_set(&nilfs_putblkbh_bbt_node_n, 0);
	atomic_set(&nilfs_putblkbh_ibt_node_n, 0);
	atomic_set(&nilfs_putblkbh_new_node_n, 0);
	atomic_set(&nilfs_putblkbh_inactive_node_n, 0);
	atomic_set(&nilfs_putblkbh_partial_node_n, 0);
#endif
	/* memstat */
#ifdef NILFS_PAGE_COUNT
	atomic_set(&nilfs_allocated_inode_page_n, 0);
	atomic_set(&nilfs_released_inode_page_n, 0);
	atomic_set(&nilfs_released_file_page_n, 0);
#endif
	atomic_set(&nilfs_released_inode_n, 0);
	atomic_set(&nilfs_allocated_inode_n, 0);
#ifdef NILFS_KMALLOC_COUNT
	atomic_set(&nilfs_kmalloc_n, 0);
	atomic_set(&nilfs_kfree_n, 0);
#endif
#ifdef NILFS_BIO_COUNT
	atomic_set(&nilfs_get_bio_n, 0);
	atomic_set(&nilfs_put_bio_n, 0);
#endif
#ifdef NILFS_SLAB_COUNT
	atomic_set(&nilfs_cache_alloc_n, 0);
	atomic_set(&nilfs_cache_free_n, 0);
#endif
#ifdef NILFS_KMAP_COUNT
	atomic_set(&nilfs_kmap_n, 0);
	atomic_set(&nilfs_kunmap_n, 0);
#endif
}

/*
 * debug info
 */
struct nilfs_debug_info nilfs_debug_info;

DEFINE_SPINLOCK(debug_print_lock);
DEFINE_SPINLOCK(debug_info_lock);

enum {
	Opt_quiet, Opt_verbose, Opt_verbose2, Opt_verbose3,
        /* Opt_quiet ~ Opt_verbose3 must be successive. */
	Opt_err
};

#define MAX_VLEVEL  3   /* Maximum level of Opt_verbose */

static match_table_t opt_tokens = {
	{Opt_verbose, "v"},
	{Opt_verbose2, "vv"},
	{Opt_verbose3, "vvv"},
	{Opt_quiet, "n"},
	{Opt_err, NULL}
};


static match_table_t class_tokens = {
	{NILFS_VERBOSE_FS, "fs"},
	{NILFS_VERBOSE_PAGE, "page"},
	{NILFS_VERBOSE_SHRINKER, "shrinker"},
	{NILFS_VERBOSE_SEGMENT, "segment"},
	{NILFS_VERBOSE_SEGINFO, "seginfo"},
	{NILFS_VERBOSE_INODE, "inode"},
	{NILFS_VERBOSE_RECOVERY, "recovery"},
	{NILFS_VERBOSE_GC, "gc"},
	{-1, NULL},
};

static inline const char *
find_token(int token, match_table_t tokens)
{
	struct match_token *pt;

	for (pt = tokens; pt->pattern != NULL; pt++)
		if (pt->token == token)
			return pt->pattern;
	return NULL;
}

void nilfs_fill_debug_info(int level)
{
	int i;

	for (i = 0; i < NILFS_VERBOSE_LIMIT; i++)
		nilfs_debug_info.verbose[i] = level;
}

static int nilfs_read_debug_option(char *page, char **start, off_t off,
				   int count, int *eof, void *data)
{
	int len = 0;
	int flag;

	spin_lock(&debug_info_lock);
	
	for (flag = 0; flag < NILFS_VERBOSE_LIMIT; flag ++) {
		const char *vopt, *p;
		int level = min(MAX_VLEVEL,
				(int)nilfs_debug_info.verbose[flag]);

		if (level >= 0) {
			vopt = find_token(Opt_quiet + level, opt_tokens);
			BUG_ON(vopt == NULL);

			p = find_token(flag, class_tokens);
			if (!p) break;
				
			if (len > 0)
				page[len++] = ' ';
			len += sprintf(page + len, "-%s %s", vopt, p);
		}
	}

	spin_unlock(&debug_info_lock);
	page[len++] = '\n';

	return proc_calc_metrics(page, start, off, count, eof, len);
}

static int
nilfs_parse_verbose_option(char **dp, char *vopt, substring_t args[],
			   int level)
{
	char *p = "";
	int flag;

	while ((p = strsep(dp, " \t\n")) != NULL) {
		if (!*p) continue;

		if (strcmp(p, "all") == 0) {
			nilfs_fill_debug_info(level);
			return 0;
		}
		flag = match_token(p, class_tokens, args);
		if (flag < 0) break;

		nilfs_debug_info.verbose[flag] = (char)level;
		return 0;
	}
	printk(KERN_ERR
	       "NILFS: Unrecognized verbose option \"-%s %s\"\n",
	       vopt, p);
	return -EINVAL;
}

static int nilfs_parse_debug_option(char *data)
{
	char *p;
	substring_t args[MAX_OPT_ARGS];
	int err;

	while ((p = strsep(&data, " \t\n")) != NULL) {
		int token, level = -1;

		if (!*p) continue;
		else if (*p != '-') goto bad_option;

		token = match_token(++p, opt_tokens, args);
		switch(token) {
		case Opt_verbose:  level = 1; break;
		case Opt_verbose2: level = 2; break;
		case Opt_verbose3: level = 3; break;
		case Opt_quiet: level = 0; break;
		default:
			goto bad_option;
		}
		if (level >= 0) {
			err = nilfs_parse_verbose_option(&data, p, args, level);
			if (err < 0)
				return err;
		}
	}
	return 0;

 bad_option:
	printk(KERN_ERR "NILFS: Unrecognized debug option \"%s\"\n", p);
	return -EINVAL;
}

static int
nilfs_write_debug_option(struct file *file, const char __user *buffer,
			 unsigned long count, void *data)
{
	char *tmp;
	int ret = -EFAULT;

	tmp = kmalloc(count + 1, GFP_KERNEL);
	if (unlikely(!tmp))
		return -ENOMEM;

	if (copy_from_user(tmp, buffer, count))
		goto out;

	tmp[count] = '\0';

	spin_lock(&debug_info_lock);

	ret = nilfs_parse_debug_option(tmp);
	if (!ret)
		ret = count;

	spin_unlock(&debug_info_lock);
 out:
	kfree(tmp);
	return ret;
}

/*
 * VINODE
 */
#define nbar(n) ((n)++?"|":"")
#define MSIZ 256
void nilfs_vinode_debug(const char *fname, int line, char *m, struct inode *inode)
{
	int n = 0, len = 0;
	char b[MSIZ];
	struct nilfs_inode_info *ii;

	if (inode == NULL) {
		printk(KERN_DEBUG "VINODE %s: inode=NULL %s at %d\n", m, fname, line);
		return;
	}
	ii = container_of(inode, struct nilfs_inode_info, vfs_inode);

	len += snprintf(b+len, MSIZ-len, "VINODE %p %s: current %p ino=%lu nlink=%u count=%u mode=0%o mapping=%p i_bh=%p",
	       inode, m, current,
	       inode->i_ino, inode->i_nlink, atomic_read(&inode->i_count), inode->i_mode,
	       inode->i_mapping, ii->i_bh);

	len += snprintf(b+len, MSIZ-len, " %s(%d) i_state=", fname, line);
	if (inode->i_state&I_DIRTY_SYNC) {len += snprintf(b+len, MSIZ-len, "DIRTY_SYNC"); n++;}
	if (inode->i_state&I_DIRTY_DATASYNC) {len += snprintf(b+len, MSIZ-len, "%sDIRTY_DATASYNC",nbar(n));}
	if (inode->i_state&I_DIRTY_PAGES) {len += snprintf(b+len, MSIZ-len, "%sDIRTY_PAGES",nbar(n));}
	if (inode->i_state&I_LOCK) {len += snprintf(b+len, MSIZ-len, "%sLOCK",nbar(n));}
	if (inode->i_state&I_FREEING) {len += snprintf(b+len, MSIZ-len, "%sFREEING",nbar(n));}
	if (inode->i_state&I_CLEAR) {len += snprintf(b+len, MSIZ-len, "%sCLEAR",nbar(n));}
	if (inode->i_state&I_NEW) {len += snprintf(b+len, MSIZ-len, "%sNEW",nbar(n));}
#ifdef I_WILL_FREE
	if (inode->i_state&I_WILL_FREE) {len += snprintf(b+len, MSIZ-len, "%sWILL_FREE",nbar(n));}
#endif

	n=0;
	if (ii->i_state)
		len += snprintf(b+len, MSIZ-len, " vi_state=");
	if (ii->i_state&NILFS_STATE_NEW) {len += snprintf(b+len, MSIZ-len, "NEW"); n++;}
	if (ii->i_state&NILFS_STATE_DIRTY) {len += snprintf(b+len, MSIZ-len, "%sDIRTY",nbar(n));}
	if (ii->i_state&NILFS_STATE_COLLECTED) {len += snprintf(b+len, MSIZ-len, "%sCOLLECTED",nbar(n));}
	if (ii->i_state&NILFS_STATE_UPDATED) {len += snprintf(b+len, MSIZ-len, "%sSTATE_UPDATED",nbar(n));}
	if (ii->i_state&NILFS_STATE_INODE_DIRTY) {len += snprintf(b+len, MSIZ-len, "%sINODE_DIRTY",nbar(n));}

	printk(KERN_DEBUG "%s\n", b);
	BH_DEBUG(ii->i_bh, "");
}

/*
 * BH_DEBUG
 */
void
nilfs_bh_debug(const char *fname, int line, char *m, struct buffer_head *bh)
{
	int n = 0, len = 0;
	char b[MSIZ];
	if (bh == NULL) {
		printk(KERN_DEBUG "BH %s: bh=NULL %s at %d\n", m, fname, line);
		return;
	}
	len += snprintf(b+len, MSIZ-len, "BH %p %s: page=%p cnt=%d blk#=%llu",
	       bh, m, bh->b_page, atomic_read(&bh->b_count), bh->b_blocknr);
	if (bh->b_page)
		len += snprintf(b+len, MSIZ-len, " pagecnt=%d", page_count(bh->b_page));
	len += snprintf(b+len, MSIZ-len, " %s(%d) state=", fname, line);
	if (buffer_uptodate(bh)) {len += snprintf(b+len, MSIZ-len, "Uptodate"); n++;}
	if (buffer_dirty(bh)) {len += snprintf(b+len, MSIZ-len, "%sDirty",nbar(n));}
	if (buffer_locked(bh)) {len += snprintf(b+len, MSIZ-len, "%sLocked",nbar(n));}
	if (buffer_req(bh)) {len += snprintf(b+len, MSIZ-len, "%sReq",nbar(n));}
	if (buffer_mapped(bh)) {len += snprintf(b+len, MSIZ-len, "%sMapped",nbar(n));}
	if (buffer_new(bh)) {len += snprintf(b+len, MSIZ-len, "%sNew",nbar(n));}
	if (buffer_async_read(bh)) {len += snprintf(b+len, MSIZ-len, "%sARead",nbar(n));}
	if (buffer_async_write(bh)) {len += snprintf(b+len, MSIZ-len, "%sAWrite",nbar(n));}
	if (buffer_delay(bh)) {len += snprintf(b+len, MSIZ-len, "%sDelay",nbar(n));}
	if (buffer_boundary(bh)) {len += snprintf(b+len, MSIZ-len, "%sBoundary",nbar(n));}
	if (buffer_write_io_error(bh)) {len += snprintf(b+len, MSIZ-len, "%sWriteIOErr",nbar(n));}
	if (buffer_ordered(bh)) {len += snprintf(b+len, MSIZ-len, "%sOrdered",nbar(n));}
	if (buffer_eopnotsupp(bh)) {len += snprintf(b+len, MSIZ-len, "%sENOTSUPP",nbar(n));}
	/* nilfs private */
	if (buffer_prepare_dirty(bh)) {len += snprintf(b+len, MSIZ-len, "%sPrepare_Dirty",nbar(n));}
	if (buffer_nilfs_allocated(bh)) {len += snprintf(b+len, MSIZ-len, "%sAllocated",nbar(n));}
	if (buffer_nilfs_node(bh)) {len += snprintf(b+len, MSIZ-len, "%sNode",nbar(n));}
	if (buffer_nilfs_bbt_node(bh)) {len += snprintf(b+len, MSIZ-len, "%sBBT_Node",nbar(n));}
	if (buffer_nilfs_ibt_node(bh)) {len += snprintf(b+len, MSIZ-len, "%sIBT_Node",nbar(n));}
	if (buffer_nilfs_new_node(bh)) {len += snprintf(b+len, MSIZ-len, "%sNewNode",nbar(n));}
	if (buffer_nilfs_inactive_node(bh)) {len += snprintf(b+len, MSIZ-len, "%sInactiveNode",nbar(n));}

	printk(KERN_DEBUG "%s\n", b);
}
#undef MSIZ
#undef nbar

void nilfs_count_freeing_bh(struct buffer_head *bufs)
{
	atomic_inc(&nilfs_putblkbh_n);
#ifdef NILFS_BH_COUNT_DETAIL
	if (buffer_nilfs_allocated(bufs))
		atomic_inc(&nilfs_putblkbh_nilfs_allocated_n);
	if (buffer_nilfs_bbt_node(bufs))
		atomic_inc(&nilfs_putblkbh_bbt_node_n);
	if (buffer_nilfs_ibt_node(bufs))
		atomic_inc(&nilfs_putblkbh_ibt_node_n);
	if (buffer_nilfs_new_node(bufs))
		atomic_inc(&nilfs_putblkbh_new_node_n);
	if (buffer_nilfs_inactive_node(bufs))
		atomic_inc(&nilfs_putblkbh_inactive_node_n);
	if (buffer_nilfs_partial_node(bufs))
		atomic_inc(&nilfs_putblkbh_partial_node_n);
#endif
}

/*
 * Seginfo
 */
void
nilfs_print_seginfo(struct nilfs_sc_info *sci, dbn_t seg_start, dbn_t seg_end)
{
	struct the_nilfs *nilfs = NILFS_SB(sci->sc_sb)->s_nilfs;

	if (!nilfs_debug_info.verbose[NILFS_VERBOSE_SEGINFO])
		return;
	else if (nilfs_debug_info.verbose[NILFS_VERBOSE_SEGINFO] == 1) {
		printk(KERN_DEBUG "NILFS(seginfo) pseg(blk=%lu,len=%u)"
		       " on fullseg(ind=%lu,blk=%lu->%lu)\n",
		       sci->sc_pseg_start, sci->sc_sum.nblocks,
		       nilfs->ns_segnum, seg_start, seg_end);
		return;
	}
	printk(KERN_DEBUG "========= NILFS SEGMENT INFORMATION ========\n");
	printk(KERN_DEBUG "full segment: segnum=%lu, start=%lu, end=%lu\n",
	       nilfs->ns_segnum, seg_start, seg_end);
	printk(KERN_DEBUG "partial segment: start=%lu, rest=%lu\n",
	       sci->sc_pseg_start, sci->sc_residual_blocks);
	printk(KERN_DEBUG "------------------ SUMMARY -----------------\n");
	printk(KERN_DEBUG "nfinfo     = %u (number of file)\n",
	       sci->sc_sum.nfinfo);
	printk(KERN_DEBUG "nfbinfo    = %u (number of fbinfo)\n",
	       sci->sc_sum.nfbinfo);
	printk(KERN_DEBUG "nblocks    = %u (number of blocks)\n",
	       sci->sc_sum.nblocks);
	printk(KERN_DEBUG "nblk_sum   = %u (number of summary blocks)\n",
	       sci->sc_sum.nblk_sum);
	printk(KERN_DEBUG "sum_bytes  = %lu (byte counts of summary)\n",
	       sci->sc_sum_bytes);
	printk(KERN_DEBUG "nblk_file  = %u (number of file blocks)\n",
	       sci->sc_sum.nblk_file);
	printk(KERN_DEBUG "nblk_fbt   = %u (number of file B-tree blocks)\n",
	       sci->sc_sum.nblk_fbt);
	printk(KERN_DEBUG "nblk_inode = %u (number of inode blocks)\n",
	       sci->sc_sum.nblk_inode);
	printk(KERN_DEBUG "nblk_ibt   = %u (number of inode B-tree blocks)\n",
	       sci->sc_sum.nblk_ibt);
	printk(KERN_DEBUG "nblk_cp    = %u (number of checkpoint blocks)\n",
	       NILFS_SEG_HAS_CP(&sci->sc_sum) ? 1 : 0);
	printk(KERN_DEBUG "============================================\n");
}

/*
 * Proc-fs entries
 */
struct proc_dir_entry *nilfs_proc_root;

static int proc_calc_metrics(char *page, char **start, off_t off,
			     int count, int *eof, int len)
{
	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;
	return len;
}

static int
nilfs_read_debug_counters(char *page, char **start, off_t off,
			  int count, int *eof, void *data)
{
	int len;
	len = sprintf(page, "getblkbh %d putblkbh %d\n",
		      atomic_read(&nilfs_getblkbh_n), atomic_read(&nilfs_putblkbh_n));
#ifdef NILFS_BH_COUNT_DETAIL
	len += sprintf(page + len, "putblkbh_nilfs_allocated %d\n",
		       atomic_read(&nilfs_putblkbh_nilfs_allocated_n));
	len += sprintf(page + len, "putblkbh_bbt_node %d\n",
		       atomic_read(&nilfs_putblkbh_bbt_node_n));
	len += sprintf(page + len, "putblkbh_ibt_node %d\n",
		       atomic_read(&nilfs_putblkbh_ibt_node_n));
	len += sprintf(page + len, "putblkbh_new_node %d\n",
		       atomic_read(&nilfs_putblkbh_new_node_n));
	len += sprintf(page + len, "putblkbh_inactive_node %d\n",
		       atomic_read(&nilfs_putblkbh_inactive_node_n));
	len += sprintf(page + len, "putblkbh_partial_node %d\n",
		       atomic_read(&nilfs_putblkbh_partial_node_n));
#endif
	return proc_calc_metrics(page, start, off, count, eof, len);
}

static int
nilfs_print_mem_stat(char *page, char **start, off_t off,
		     int count, int *eof, void *data)
{
	int len;

	len = sprintf(page, "inode: alloc=%d, free=%d\n",
		      atomic_read(&nilfs_allocated_inode_n),
		      atomic_read(&nilfs_released_inode_n));
	len += sprintf(page + len,
		       "blkbh: get=%d, put=%d\n",
		       atomic_read(&nilfs_getblkbh_n),
		       atomic_read(&nilfs_putblkbh_n));
#ifdef NILFS_ALLOCPAGE_COUNT
	len += sprintf(page + len,
		       "page: alloc=%d, free=%d\n",
		       atomic_read(&nilfs_alloc_page_n),
		       atomic_read(&nilfs_free_page_n));
#endif
#ifdef NILFS_PAGE_COUNT
	len += sprintf(page + len,
		       "file page: alloc=?, free=%d\n",
		       atomic_read(&nilfs_released_file_page_n));
	len += sprintf(page + len,
		       "inode page: alloc=%d, free=%d\n",
		       atomic_read(&nilfs_allocated_inode_page_n),
		       atomic_read(&nilfs_released_inode_page_n));
#endif
#ifdef NILFS_KMALLOC_COUNT
	len += sprintf(page + len,
		       "kmalloc (nof times): alloc=%d, free=%d\n",
		       atomic_read(&nilfs_kmalloc_n),
		       atomic_read(&nilfs_kfree_n));
#endif
#ifdef NILFS_BIO_COUNT
	len += sprintf(page + len,
		       "bio: get=%d, put=%d\n",
		       atomic_read(&nilfs_get_bio_n),
		       atomic_read(&nilfs_put_bio_n));
#endif
#ifdef NILFS_SLAB_COUNT
	len += sprintf(page + len,
		       "slab: alloc=%d, free=%d\n",
		       atomic_read(&nilfs_cache_alloc_n),
		       atomic_read(&nilfs_cache_free_n));
#endif
#ifdef NILFS_KMAP_COUNT
	len += sprintf(page + len,
		       "kmap: map=%d, unmap=%d\n",
		       atomic_read(&nilfs_kmap_n),
		       atomic_read(&nilfs_kunmap_n));
#endif
	return proc_calc_metrics(page, start, off, count, eof, len);
}

/* for debug free_inode_list */
static int
nilfs_print_free_inode_list(char *page, char **start, off_t off,
			 int count, int *eof, void *data)
{
	int len;
	struct nilfs_sb_info *sbi = NULL;
	struct super_block *sb;
	struct list_head *p;

 rescan:
	spin_lock(&nilfs_sb_lock);
	list_for_each(p, &nilfs_super_blocks) {
	        sbi = nilfs_sb_entry(p);
		sb = sbi->s_super;
		if (sb->s_type == &nilfs_fs_type) {
			sb->s_count++;
			spin_unlock(&nilfs_sb_lock);
			down_read(&sb->s_umount);
			if (sb->s_root)
				goto found;
			drop_super(sb); /* executes up_read(&sb->s_umount) */
			goto rescan;
		}
	}
	sb = NULL;
	spin_unlock(&nilfs_sb_lock);
 found:
	if (sb) {
		struct nilfs_free_inode_list *fi;
		struct list_head *p;
		struct nilfs_inode_hdr *hdr;

		down(&sbi->s_free_inodes_sem);
		list_for_each(p, &sbi->s_free_inodes) {
			fi = list_entry(p, struct nilfs_free_inode_list, buffers);
			hdr = (struct nilfs_inode_hdr *)fi->free_bh->b_data;
			printk("inode_list %p ino %llu nfree %u free %u\n", 
			       p,
			       le64_to_cpu(hdr->ih_ino),
			       le32_to_cpu(hdr->ih_nfree),
			       le32_to_cpu(hdr->ih_free));
		}
		up(&sbi->s_free_inodes_sem);
		len = sprintf(page, "nseg %lu\n", sbi->s_nsegment);
		drop_super(sb);
	} else {
		len = sprintf(page, "null sb\n");
	}
	return proc_calc_metrics(page, start, off, count, eof, len);
}

int nilfs_init_proc_entries(void)
{
	struct proc_dir_entry *entry;

	nilfs_proc_root = proc_mkdir("nilfs", proc_root_fs);
	if (!nilfs_proc_root) {
		printk(KERN_WARNING "NILFS: cannot create proc root\n");
		return 0; /* We don't abort when failed to make 
			     proc entries */
	}
	nilfs_proc_root->owner = THIS_MODULE;

	/* /proc entries */
	entry = create_proc_entry("debug_option", 
				  S_IFREG | S_IRUGO | S_IWUSR, 
				  nilfs_proc_root);
	if (entry) {
		entry->read_proc = nilfs_read_debug_option;
		entry->write_proc = nilfs_write_debug_option;
	}

	entry = create_proc_entry("nputblk", 0, nilfs_proc_root);
	if (entry)
		entry->read_proc = nilfs_read_debug_counters;

	entry = create_proc_entry("memstat", 0, nilfs_proc_root);
	if (entry)
		entry->read_proc = nilfs_print_mem_stat;

	entry = create_proc_entry("ifree", 0, nilfs_proc_root);
	if (entry)
		entry->read_proc = nilfs_print_free_inode_list;

	return 0;
}

void nilfs_remove_proc_entries(void)
{
	remove_proc_entry("ifree", nilfs_proc_root);
	remove_proc_entry("memstat", nilfs_proc_root);
	remove_proc_entry("nputblk", nilfs_proc_root);
	remove_proc_entry("debug_option", nilfs_proc_root);
	remove_proc_entry("nilfs", proc_root_fs);
}

/*
 * inode page debug
 */
int nilfs_release_inode_page(struct page *page, gfp_t gfp_mask)
{
	int ret;
	int b_count = 0;
	unsigned long b_state = 0;
	int verbose = (nilfs_debug_info.verbose[NILFS_VERBOSE_PAGE] > 1);

	if (!verbose) {
		struct address_space *mapping = page->mapping;

		if (mapping &&
		    !(NILFS_AS_SB(mapping)->s_super->s_flags & MS_ACTIVE))
			verbose = 1;
	}
	
	if (PagePrivate(page)) {
		struct buffer_head *bh = page_buffers(page);

		if (buffer_nilfs_allocated(bh))
			return nilfs_release_allocated_page(page, gfp_mask);
		if (buffer_nilfs_node(bh))
			return nilfs_release_node_page(page, gfp_mask);

		if (verbose) {
			b_count = atomic_read(&bh->b_count);
			b_state = bh->b_state;
		}
	}

	ret = try_to_free_buffers(page);
	if (verbose && !ret) {
		page_warn("try_to_free_buffers() failed (page=%p, "
			  "count=%d, b_count=%d, b_state=0x%lx)\n",
			   page, page_count(page), b_count, b_state);
	}

	if (ret && page->mapping) {
		if (page_count(page) > 2 + !PageLRU(page))
			page_warn("too many page_count=%d (page=%p)\n",
				  page_count(page), page);
#ifdef NILFS_PAGE_COUNT
		atomic_inc(&nilfs_released_inode_page_n);
#endif
	}
	return ret;
}

/*
 * File page debug
 */
int nilfs_releasepage(struct page *page, gfp_t gfp_mask)
{
	int ret;
	int verbose = (nilfs_debug_info.verbose[NILFS_VERBOSE_PAGE] > 1);

	if (!verbose) {
		struct address_space *mapping = page->mapping;

		if (mapping && mapping->host &&
		    !(mapping->host->i_sb->s_flags & MS_ACTIVE))
			verbose = 1;
	}

	if (PagePrivate(page)) {
		struct buffer_head *bh = page_buffers(page);

		if (buffer_nilfs_allocated(bh))
			return nilfs_release_allocated_page(page, gfp_mask);
		if (buffer_nilfs_node(bh))
			return nilfs_release_node_page(page, gfp_mask);
	}
	ret = try_to_free_buffers(page);
	if (verbose && !ret) {
		page_warn("try_to_free_buffers() failed (page=%p, count=%d)\n",
			   page, page_count(page));
	}

	if (ret && page->mapping && page->mapping->host) {
		if (page_count(page) > 2 + !PageLRU(page))
			page_warn("too many page_count=%d (page=%p)\n",
				  page_count(page), page);
#ifdef NILFS_PAGE_COUNT
		atomic_inc(&nilfs_released_file_page_n);
#endif
	}
	return ret;
}

/*
 * Radix-tree checker
 */
#define S_N_PAGEVEC  16
void
nilfs_check_radix_tree(const char *fname, struct address_space *mapping,
		       int blocksize_bits)
{
	struct page *pages[S_N_PAGEVEC];
	unsigned int nr_page;
	pgoff_t index = 0;
	int nr_found = 0;
	int i;

 repeat:
#if NEED_RWLOCK_FOR_PAGECACHE_LOCK
	read_lock_irq(&mapping->tree_lock);
#else
	spin_lock_irq(&mapping->tree_lock);
#endif
	nr_page = radix_tree_gang_lookup(&mapping->page_tree,
					 (void **)pages,
					 index,
					 S_N_PAGEVEC);
	for (i = 0; i < nr_page; i++)
		page_cache_get(pages[i]);
#if NEED_RWLOCK_FOR_PAGECACHE_LOCK
	read_unlock_irq(&mapping->tree_lock);
#else
	spin_unlock_irq(&mapping->tree_lock);
#endif

	if (nr_page == 0) {
		if (nr_found)
			printk(KERN_WARNING "%s: found %d leaking pages\n",
			       fname, nr_found);
		return;
	}
	index = pages[nr_page - 1]->index + 1;

	for(i = 0; i < nr_page; i++) {
		struct buffer_head *bh, *head;
		fbn_t fbn;

		fbn = pages[i]->index >> (PAGE_SHIFT - blocksize_bits);
		printk(KERN_WARNING 
		       "found leaking page(addr=%p, index=%lu)\n",
		       pages[i], fbn);
		nr_found++;
		if (!page_has_buffers(pages[i]))
			goto skip_page;
		bh = head = page_buffers(pages[i]);
		do {
			BH_DEBUG(bh, "leaking block");
			bh = bh->b_this_page;
			fbn++;
		} while (bh != head);
	skip_page:
		page_cache_release(pages[i]);
	}
	goto repeat;
}

void
nilfs_check_radix_tree_64(const char *fname, 
		       struct radix_tree_64_root *tree,
		       spinlock_t *tree_lock,
		       int blocksize_bits)
{
	struct page *pages[S_N_PAGEVEC];
	unsigned int nr_page;
	pgoff_t index = 0;
	int nr_found = 0;
	int i;

 repeat:
	spin_lock_irq(tree_lock);
	nr_page = radix_tree_64_gang_lookup(tree,
					    (void **)pages,
					    index,
					    S_N_PAGEVEC);
	for (i = 0; i < nr_page; i++)
		page_cache_get(pages[i]);
	spin_unlock_irq(tree_lock);

	if (nr_page == 0) {
		if (nr_found)
			printk(KERN_WARNING "%s: found %d leaking pages\n",
			       fname, nr_found);
		return;
	}
	index = nilfs_node_page_index(pages[nr_page - 1]) + 1;

	for(i = 0; i < nr_page; i++) {
		struct buffer_head *bh, *head;
		dbn_t dbn;

		dbn = nilfs_node_page_index(pages[i]) >> (PAGE_SHIFT - blocksize_bits);
		printk(KERN_WARNING 
		       "found leaking node page(addr=%p, index=%lu)\n",
		       pages[i], dbn);
		nr_found++;
		if (!page_has_buffers(pages[i]))
			goto skip_page;
		bh = head = page_buffers(pages[i]);
		do {
			BH_DEBUG(bh, "leaking block");
			bh = bh->b_this_page;
			dbn++;
		} while (bh != head);
	skip_page:
		page_cache_release(pages[i]);
	}
	goto repeat;
}


/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
