/*
 * sb.h - NILFS on-memory super block structure.
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * sb.h,v 1.15 2006/06/16 07:01:51 ryusuke Exp
 */

#ifndef _NILFS_SB
#define _NILFS_SB

#include "btree.h"

/* the_nilfs struct */
enum {
	THE_NILFS_INIT = 0,     /* Information from super_block is set */
	THE_NILFS_LOADED,       /* Roll-back/roll-forward has done and
				   the latest checkpoint was loaded */
	THE_NILFS_KOBJECT,      /* nilfs kobject has been created */
};

struct the_nilfs {
	unsigned long ns_flags;
        atomic_t ns_count;

	struct block_device *ns_bdev;
	struct nilfs_sb_info *ns_writer;    /* back pointer to the writable nilfs_sb_info
					       (protected by ns_writer_sem) */
	struct kobject ns_kobj;             /* nilfs kobject */
	spinlock_t ns_lock;	            /* Spinlock for the fields
					       that may be called from
					       bottom-half routines. */
	struct rw_semaphore ns_sem;         /* Semaphore for shared states */
	struct rw_semaphore ns_writer_sem;  /* Semaphore for writer attach/detach */

        /* 
	 * used for
	 * - loading the latest checkpoint exclusively.
	 * - allocating a new full segment.
	 * - protecting s_dirt in the struct super_block (see nilfs_write_super)
	 *   and the following fields.
	 */
	struct buffer_head *ns_sbh;         /* Buffer containing the super block */
	struct nilfs_super_block *ns_sbp;   /* Pointer to the disk super block
					       in the buffer */
	unsigned ns_mount_state;

	/*
	 * Following fields are lock free because these fields are dedicated 
	 * to a writable FS-instance except for the period seeking checkpoint.
	 * The writable FS-instance is sole during a lifetime of the_nilfs.
	 */
	u64 ns_seg_seq;                     /* Segment sequence counter */
        segnum_t ns_segnum;                 /* Full segment index last used */
	unsigned long ns_pseg_offset;       /* Offset of next partial segment
 					       in the last full segment */
	unsigned ns_prev_pseg;              /* Span of previous partial segment */

	/*
	 * Following fields are protected by ns_lock.
	 */
	u64 ns_last_seq;                    /* Sequence number of a full segment
					       having the last checkpoint */
	dbn_t ns_last_pseg;                 /* last partial segment having a valid
					       checkpoint */
	dbn_t ns_last_cp;                   /* Last checkpoint block */
	dbn_t ns_inode_root;                /* Inode root of the latest checkpoint */
	time_t ns_last_ctime;               /* Construction time of the last segment
					       having a valid CP */
	unsigned long ns_inodes_count;      /* Inode count of the latest checkpoint */
	unsigned long ns_blocks_count;      /* Block count of the latest checkpoint */

	unsigned long ns_free_segments_count;
};

/* Fields for segment constructor */
struct nilfs_segctor {
        struct list_head dirty_files;        /* dirty files list */
        struct list_head garbage_list;       /* list of inodes to be put */
	struct rw_semaphore sem;             /* semaphore */

	spinlock_t dirty_files_lock;         /* lock for the dirty files list */

	struct nilfs_sc_info *info;          /* segment constructor info */
	unsigned long interval;              /* construction interval */
	unsigned long block_max;             /* threshold of data amount
						for the segment construction */
};

/* list mounted nilfs and it's lock */
extern struct list_head nilfs_super_blocks;
extern spinlock_t nilfs_sb_lock;
#define nilfs_sb_entry(list)  list_entry((list), struct nilfs_sb_info, s_list)

/*
 * Mount options
 */
struct nilfs_mount_options {
	unsigned long mount_opt;
	dbn_t snapshot_pseg;
};

/*
 * NILFS super-block data in memory
 */
enum {
	NILFS_SB_IBLK_DIRTY,      /* One or more dirty inode blocks exist */
};

struct nilfs_sb_info {
	/* Disk layout information */
	unsigned long s_inodes_per_block;	/* Number of inodes per block */
        unsigned long s_blocks_per_segment;	/* Number of blocks per segment */
	unsigned long s_nsegment;		/* Number of segment in filesystem */
	unsigned long s_first_data_block;
	u32 s_crc_seed;                         /* Seed value of CRC32 calculation */
	int s_inode_size;
	int s_first_ino;

	/* Snapshot status */
        dbn_t s_snapshot_pseg;
	dbn_t s_snapshot_cp;                    /* Checkpoint block number */
	atomic_t s_inodes_count;
	atomic_t s_blocks_count;

	/* Mount options */
	unsigned long s_mount_opt;
	uid_t s_resuid;
	gid_t s_resgid;

	/* Fundamental members */
        struct list_head s_list;		/* nilfs list */
        struct super_block *s_super;		/* reverse pointer to linux super block */
	struct the_nilfs *s_nilfs;
	struct nilfs_segctor s_segctor;         /* segment constructor */
        struct address_space s_data;		/* embedded s_mapping */
	unsigned long s_flags;                  /* NILFS superblock flags */

	/* Inode B-tree */
	struct nilfs_btree s_inode_root;	/* inode B-Tree root block */
	struct rw_semaphore s_inode_root_sem;	/* semaphore for B-Tree operation */
        struct address_space *s_mapping;	/* for inode B-Tree Data Block */
	struct radix_tree_64_root s_inode_ntree; /* inode B-Tree Node Block tree */
	spinlock_t s_inode_ntree_lock;

	/* Inode allocator */
	struct list_head s_free_inodes;		/* free Inodes buffer_head list */
#define	S_FREE_INODES_LEN_WMARK	32		/* max of len == S_FREE_INODES_LEN_MARK*2 */
	long s_free_inodes_len;			/* length(s_free_inodes) */
	long s_free_inodes_count;		/* sum of free inodes on s_free_inodes */
        struct semaphore s_free_inodes_sem;	/* s_free_inodes_sem provides exclusive access
						   control for the followings:
						    1) s_free_inodes list
						    2) s_free_inodes_len & s_free_inodes_count
						    3) fields in the inode block header except
						       for static ones (e.g. ih_ino). */
	spinlock_t s_next_gen_lock;
	u32 s_next_generation;
	
	/* B-tree node block management */
	struct semaphore s_node_pages_sem;	/* for s_partial_node_pages */
	struct list_head s_partial_node_pages;	/* partial B-Tree node pages */
	struct list_head s_inactive_node_pages;	/* inactive B-Tree node pages */
	int s_inactive_node_pages_cnt;		/* counter of s_inactive_... */
        spinlock_t s_inactive_node_pages_lock;	/* for s_inactive_node_pages */
};

static inline struct nilfs_sb_info *NILFS_SB(struct super_block *sb)
{
	return sb->s_fs_info;
}

static inline struct nilfs_sb_info *
NILFS_AS_SB(struct address_space *mapping)
{
	return container_of(mapping, struct nilfs_sb_info, s_data);
}

#define NILFS_BLOCK_SIZE_BITS(s)   ((s)->s_blocksize_bits)
#define NILFS_BLOCK_SIZE(s)        (NILFS_SB(s)->s_blocksize)
#define NILFS_INODE_SIZE(s)        (NILFS_SB(s)->s_inode_size)
#define NILFS_FIRST_INO(s)         (NILFS_SB(s)->s_first_ino)

#if 0
/* nilfs V1.01 */
static inline int nilfs_sb_in_seg(struct nilfs_sb_info *sbi)
{
	struct the_nilfs *nilfs = sbi->s_nilfs;
	int ret;
	
	/* This reference is heavy.
	   We can mitigate it by copying the flag, or enlarge
	   critical section when making segusage exclusive */
	down_read(&nilfs->ns_sem);
	ret = (nilfs->ns_mount_state & NILFS_SB_IN_SEG) ? 1 : 0;
	up_read(&nilfs->ns_sem);
	return ret;
}
#endif

/* 
 * Bit operations for the mount option
 */
#define nilfs_clear_opt(sbi, opt)        (sbi)->s_mount_opt &= ~NILFS_MOUNT_##opt
#define nilfs_set_opt(sbi, opt)          (sbi)->s_mount_opt |= NILFS_MOUNT_##opt
#define nilfs_test_opt(sbi, opt)         ((sbi)->s_mount_opt & NILFS_MOUNT_##opt)
#define nilfs_write_opt(sbi, mask, opt)  (sbi)->s_mount_opt = (((sbi)->s_mount_opt & ~NILFS_MOUNT_##mask) | NILFS_MOUNT_##opt)

/* 
 * array of segment usage structure. on memory
 */
struct segusage_info {
	struct rw_semaphore su_write_sem; /* partial segment writing */
        spinlock_t su_lock;
	struct buffer_head *head;	/* point to segusage info */
};

#endif /* _NILFS_SB */

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
