/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "contactmanager.h"
#include "contactmanageradaptor.h"

#include "accountmanager.h"
#include "contactconnector.h"
#include "simplisticcontactconnector.h"

#include <QtCore/QHash>
#include <QtCore/QCoreApplication>
#include <QtCore/QPointer>
#include <QtCore/QDebug>
#include <QtCore/QTimer>

#include <QtTapioca/Connection>
#include <QtTapioca/Contact>
#include <QtTapioca/ContactList>
#include <QtTapioca/Handle>

/// @cond INCLUDE_PRIVATE

/**
 * @brief Private data class for the ContactManager.
 * @author Tobias Hunger <info@basyskom.de>
 */
class ContactManagerPrivate : public QObject
{
public:
    /** @brief Constructor. */
    ContactManagerPrivate(AccountManager * account_mgr,
                          ContactManager * contact_mgr) :
        accountManager(account_mgr),
        contactManager(contact_mgr),
        m_connector(new SimplisticContactConnector())
    {
        Q_ASSERT(0 != accountManager);
        Q_ASSERT(0 != m_connector);
    }

    /** @brief Destructor. */
    ~ContactManagerPrivate()
    {
        const QtTapioca::Connection * connection;
        foreach (connection, m_connectionlist.keys())
        { deregisterContacts(connection); }
        delete m_connector;
    }

    /**
     * @brief Register a new contact found on a connection.
     * @param protocol The protocol of the connection.
     * @param contact A pointer to the contact.
     * @param connection A pointer to the connection.
     *
     * Register a contact found on one of the connections in the mappings
     * to/from the external ID.
     */
    void registerContact(const QString & protocol,
                         QtTapioca::Contact * contact,
                         const QtTapioca::Connection * connection)
    {
        Q_ASSERT(0 != connection);
        Q_ASSERT(0 != contact);
        Q_ASSERT(!protocol.isEmpty());

        QString uri(contact->uri());
        QString uri_protocol(getProtocol(uri));
        if (uri_protocol.isEmpty())
        { uri = protocol + QString("://") + uri; }
        else
        {
            if (protocol != uri_protocol)
            {
                qWarning() << "CtM: Fixing protocol in URI, was" << protocol
                           << "changed to" << uri_protocol;
                uri = uri_protocol + "://" + getCMInternalUri(uri);
            }
        }
        Q_ASSERT(uri.indexOf(QString("://") != 0));
        Q_ASSERT(getProtocol(uri) == protocol);
        Q_ASSERT(getCMInternalUri(uri) == getCMInternalUri(contact->uri()));

        // mark the handle as belonging to the given connection:
        m_connectionlist[connection].append(contact);

        // Update mapping between handles:
        uint external_id = m_connector->findURI(uri);
        if (external_id == 0)
        {
            // FIXME: Add data to PIM?
        }
        else
        {
            Q_ASSERT(0 != external_id);
            // FIXME: Do I need to update presence, etc. info here?
            m_telepathy2external.insert(contact, external_id);
            m_external2telepathy.insert(external_id, contact);
        }
        Q_ASSERT(m_telepathy2external.size() == m_external2telepathy.size());
    }

    /**
     * @brief Deregister a contact.
     * @param connection A pointer to a connection.
     *
     * Deregister all contacts found on the given connection from the
     * mappings between internal and external contact IDs.
     */
    void deregisterContacts(const QtTapioca::Connection * connection)
    {
        Q_ASSERT(0 != connection);
        if (!m_connectionlist.contains(connection)) { return; }

        QtTapioca::Contact * internal_contact = 0;
        foreach (internal_contact, m_connectionlist[connection])
        {
            if (!m_telepathy2external.contains(internal_contact)) { continue; }
            Q_ASSERT(m_telepathy2external.contains(internal_contact));
            uint external_id = m_telepathy2external[internal_contact];
            Q_ASSERT(m_external2telepathy.contains(external_id));
            Q_ASSERT(m_external2telepathy[external_id] == internal_contact);

            m_external2telepathy.remove(external_id);
            m_telepathy2external.remove(internal_contact);
            Q_ASSERT(m_telepathy2external.size() == m_external2telepathy.size());
        }
        m_connectionlist.remove(connection);
    }

    /**
     * @brief Update the presence of a contact.
     * @param internal_contact internal (telepathy) contact handle.
     * @param presence The presence state.
     * @param message The presence message.
     *
     * Update the presence state and message of a contact.
     */
    void setPresence(QtTapioca::Contact * internal_contact,
                     const QtTapioca::ContactBase::Presence presence,
                     const QString message)
    {
        Q_ASSERT(0 != internal_contact);
        Q_ASSERT(presence >= QtTapioca::ContactBase::Offline &&
                 presence <= QtTapioca::ContactBase::Busy);

        uint external_id = mapToExternal(internal_contact);
        if (0 == external_id) { return; }
        m_connector->setPresence(external_id, presence, message);
    }

    /**
     * @brief Check whether a contact ID is defined.
     * @param external_id contact ID from the external PIM system.
     * @return true if the contact is defined and false otherwise.
     *
     * Check whether a contact ID is defined.
     */
    bool gotContact(const uint external_id)
    { return m_connector->gotContact(external_id); }

    /**
     * @brief Map a Contact to its external ID.
     * @param contact A pointer to the Contact.
     * @return 0 if the contact is not known in the external PIM system and
     * the contact's ID otherwise.
     *
     * Map a Contact to its external ID.
     */
    uint mapToExternal(QtTapioca::Contact * contact)
    {
        if (!m_telepathy2external.contains(contact)) { return 0; }
        else
        {
            Q_ASSERT(m_telepathy2external[contact] != 0);
            return m_telepathy2external[contact];
        }
    }

    /**
     * @brief Map a external contact ID to the internal Contact data.
     * @param external_id The contact's ID in the external PIM system.
     * @return A pointer to the internal Contact structure. This pointer is 0
     * if the contact is not know to telepathy.
     *
     * Map a external contact ID to the internal Contact data.
     */
    QtTapioca::Contact * mapFromExternal(const uint & external_id)
    {
        if (!m_external2telepathy.contains(external_id)) { return 0; }
        else
        {
            Q_ASSERT(m_external2telepathy[external_id] != 0);
            return m_external2telepathy[external_id];
        }
    }

    /** @brief A pointer to the AccountManager. */
    QPointer<AccountManager> accountManager;
    /** @brief A pointer to the D-Bus Adaptor of the ContactManager. */
    QPointer<ContactManagerAdaptor> adaptor;

    QStringList getURIs(const uint contact_id,
                        const QString & protocol) const
    { return m_connector->getURIs(contact_id, protocol); }

    QtTapioca::Connection * connectAccount(const int account_handle)
    {
        // check Account:
        if (!accountManager->gotAccount(account_handle))
        {
            qWarning() << "CtM: Account" << account_handle << "is invalid.";
            return 0;
        }

        // Bring up account (if not already up):
        if (accountManager->presence(account_handle) ==
            QtTapioca::ContactBase::Offline)
        {
            Q_ASSERT(0 == accountManager->connectionOf(account_handle));
            accountManager->setPresence(account_handle,
                                        QtTapioca::ContactBase::Available);
        }

        // Wait for connection to get established...
        QtTapioca::Connection * connection(0);
        while (0 == connection)
        {
            QCoreApplication::instance()->processEvents();
            connection = accountManager->connectionOf(account_handle);
        }

        return connection;
    }

    /** @brief Do Tapioca-Magic to set up the actual channel. */
    Decibel::ChannelInfo
    contactContactViaConnection(QtTapioca::Contact * const contact,
                                QtTapioca::Connection * const connection,
                                const int type,
                                const bool suppress_handler)
    {
        // Make sure data is valid:
        // * contact:
        if (0 == contact)
        {
            qWarning() << "CtM: Contact not set up.";
            return Decibel::ChannelInfo();
        }

        // * type:
        if (type < int(QtTapioca::Channel::Text) ||
            type > int(QtTapioca::Channel::Stream))
        {
            qWarning() << "CtM: Invalid type of connection requested:" << type;
            return Decibel::ChannelInfo();
        }

        // * connection:
        if (0 == connection)
        {
            qWarning() << "CtM: No connection set up.";
            return Decibel::ChannelInfo();
        }

        // Do the work:
        QtTapioca::Channel * channel =
            connection->createChannel(QtTapioca::Channel::Type(type), contact,
                                      suppress_handler);
        if (0 == channel)
        {
            qWarning() << "CtM: Failed to create channel of type" << type
                    << "to contact" << contact << "(suppress=" << suppress_handler
                    << ").";
            return Decibel::ChannelInfo();
        }
        qDebug() << "CtM: Got my channel!";

        // decouple execution of channel handlers:
        newChannelList.append(qMakePair(connection, channel));
        QTimer::singleShot(10, contactManager, SLOT(onNewChannelCreated()));

        Decibel::ChannelInfo channel_info(connection, channel, false);

        Q_ASSERT(!channel_info.isNull());
        return channel_info;
    }

    QString getProtocol(const QString & uri)
    {
        int pos = uri.indexOf(QString("://"));
        if (pos < 0) { return QString(); }
        else { return uri.left(pos); }
    }

    QString getCMInternalUri(const QString & uri)
    {
        int pos = uri.indexOf(QString("://"));
        if (pos < 0) { return uri; }
        else { return uri.mid(pos + 3); }
    }

    bool testUri(const QString & uri)
    {
        // Make sure contact_uri is valid:
        QString remote_protocol = getProtocol(uri);
        if (uri.isEmpty())
        {
            qWarning() << "CtM: Contact URI is empty.";
            return false;
        }
        else if (remote_protocol.isEmpty())
        {
            qWarning() << "CtM: Contact URI" << uri
                    << "does not name a protocol.";
            return false;
        }
        return true;
    }

    /**
     * @brief Find all accounts supporting a given protocol.
     * @param protocol The protocol the account must support.
     * @return A list of accounts supporting the requested protocol.
     *
     * Find all the accounts that support a given protocol. Accounts in an
     * online state are returned first in the list.
     */
    QList<uint> findAccountsSupporting(const QString & protocol)
    {
        QList<uint> allAccounts(accountManager->listAccounts());
        QList<uint> possibleAccounts;
        QList<uint> possibleOnlineAccounts;
        uint currentAccount;
        foreach(currentAccount, allAccounts)
        {
            if (accountManager->protocol(currentAccount) == protocol)
            {
                if (0 != accountManager->connectionOf(currentAccount))
                { possibleOnlineAccounts.append(currentAccount); }
                else { possibleAccounts.append(currentAccount); }
            }
        }

        return possibleOnlineAccounts + possibleAccounts;
    }

    QList<QPair<QtTapioca::Connection *, QtTapioca::Channel *> > newChannelList;

private:
    ContactManager * const contactManager;

    /** @brief A pointer to the ContactConnector used. */
    SimplisticContactConnector * m_connector;

    /**
     * @brief A mapping from the Tapioca Contact structure to the external ID
     * of the contact.
    */
    QHash<QtTapioca::Contact *, uint> m_telepathy2external;
    /**
     * @brief A mapping from the external ID of a contact to its Tapioca contact
     * structure.
     */
    QHash<uint, QtTapioca::Contact *> m_external2telepathy;
    /**
     * @brief A mapping between Connections and their Contacts.
     */
    QHash<const QtTapioca::Connection *,
          QList<QtTapioca::Contact *> > m_connectionlist;

};

/// @endcond

// **************************************************************************

ContactManager::ContactManager(AccountManager * account_mgr,
                               QObject * parent) :
        QObject(parent),
        d(new ContactManagerPrivate(account_mgr, this))
{
    Q_ASSERT(0 != d);
    d->adaptor = new ContactManagerAdaptor(this);
    Q_ASSERT(0 != d->adaptor);
}

ContactManager::~ContactManager()
{ delete d; }

Decibel::ChannelInfo
ContactManager::contactUrlUsingAccount(const QString & contact_uri,
                                       const int account_handle,
                                       const int type,
                                       const bool suppress_handler)
{
    // Make sure contact_uri is valid:
    if (!d->testUri(contact_uri))
    { return Decibel::ChannelInfo(); }
    QString remote_protocol = d->getProtocol(contact_uri);

    QString local_protocol = d->accountManager->protocol(account_handle);
    if (local_protocol != remote_protocol)
    {
        qWarning() << "CtM: Local protocol" << local_protocol
                   << "and remote protocol" << remote_protocol
                   << "differ!";
    }

    // This tests the account_handle:
    QtTapioca::Connection * connection = d->connectAccount(account_handle);
    if (0 == connection)
    {
        qWarning() << "CtM: Failed to setup Connection.";
        return Decibel::ChannelInfo();
    }

    // We have a connection from this point on:
    Q_ASSERT(0 != connection);

    // Create contact:
    QString cm_internal_uri = d->getCMInternalUri(contact_uri);
    QtTapioca::Contact * contact(connection->contactList()->
                                 contact(cm_internal_uri));
    if (0 == contact)
    { contact = connection->contactList()->addContact(cm_internal_uri); }
    d->registerContact(remote_protocol, contact, connection);

    return d->contactContactViaConnection(contact, connection,
                                          type, suppress_handler);
}

Decibel::ChannelInfo
ContactManager::contactContactUsingAccount(const uint contact_id,
                                           const int account_handle,
                                           const int type,
                                           const bool suppress_handler)
{
    // Make sure contact handle is valid:
    if (!d->gotContact(contact_id))
    {
        qWarning() << "CtM: Contact" << contact_id << "is invalid.";
        return Decibel::ChannelInfo();
    }

    // This tests the account_handle:
    QtTapioca::Connection * connection = d->connectAccount(account_handle);
    if (0 == connection)
    {
        qWarning() << "CtM: Failed to setup Connection.";
        return Decibel::ChannelInfo();
    }

    // We have a connection from this point on:
    Q_ASSERT(0 != connection);

    // Check whether contact is online:
    QtTapioca::Contact * contact = d->mapFromExternal(contact_id);
    if (0 == contact)
    {
        QString protocol(d->accountManager->protocol(account_handle));
        QStringList uris = d->getURIs(contact_id, protocol);
        if (uris.isEmpty())
        {
            qWarning() << "CtM: Contact" << contact_id
                       << "has no support for protocol" << protocol;
            return Decibel::ChannelInfo();
        }
        // FIXME: Handle multiple uris...
        contact = connection->contactList()->addContact(d->getCMInternalUri(uris[0]));
        d->registerContact(protocol, contact, connection);
        Q_ASSERT(d->mapFromExternal(contact_id) == contact);
    }

    return d->contactContactViaConnection(contact, connection,
                                          type, suppress_handler);
}

Decibel::ChannelInfo
ContactManager::contactUrl(const QString & contact_uri, const int type,
                           const bool suppress_handler)
{
    // Make sure contact_uri is valid:
    if (!d->testUri(contact_uri))
    { return Decibel::ChannelInfo(); }
    QString remote_protocol = d->getProtocol(contact_uri);

    // Find accounts supporting protocol:
    QList<uint> possibleAccounts = d->findAccountsSupporting(remote_protocol);

    if (possibleAccounts.isEmpty())
    {
        qWarning() << "CtM: No account supporting" << remote_protocol
                   << "found.";
        return Decibel::ChannelInfo();
    }

    return contactUrlUsingAccount(contact_uri, possibleAccounts[0], type,
                                  suppress_handler);
}

void ContactManager::onConnectionOpened(QtTapioca::Connection * connection)
{
    Q_ASSERT(0 != connection);
    Q_ASSERT(connection->status() == QtTapioca::Connection::Connected);

    const QtTapioca::ContactList * contact_list = connection->contactList();
    Q_ASSERT(0 != contact_list);

    const QString protocol(connection->protocol());
    Q_ASSERT(!protocol.isEmpty());

    QList<QtTapioca::Contact *> known_contacts = contact_list->knownContacts();
    QtTapioca::Contact * internal_contact = 0;
    foreach (internal_contact, known_contacts)
    {
        Q_ASSERT(0 != internal_contact);

        QtTapioca::Handle * internal_handle = internal_contact->handle();
        Q_ASSERT(0 != internal_handle);
        // FIXME: Handle non-contact handles properly:
        switch (internal_handle->type())
        {
            case QtTapioca::Handle::Contact:
                d->registerContact(protocol, internal_contact, connection);
                connect(internal_contact,
                        SIGNAL(presenceUpdated(QtTapioca::ContactBase *,
                                               QtTapioca::ContactBase::Presence,
                                               const QString &)),
                        this,
                        SLOT(onPresenceUpdated(QtTapioca::ContactBase *,
                                               QtTapioca::ContactBase::Presence,
                                               const QString &)));
                break;
            case QtTapioca::Handle::Room:
            case QtTapioca::Handle::List:
            case QtTapioca::Handle::None:
                // FIXME: Handle other types of handles...
                break;
            default:
                qWarning() << "Unknown Handle type encountered.";
        }
    }
}

void ContactManager::onConnectionClosed(QtTapioca::Connection * connection)
{
    Q_ASSERT(0 != connection);
    // This can hit all connections due to a crash, etc. so it does not
    // make sense to assert the status of the connection here.

    d->deregisterContacts(connection);
}

void ContactManager::onPresenceUpdated(QtTapioca::ContactBase * internal_contactbase,
                                       QtTapioca::ContactBase::Presence presence,
                                       const QString & message)
{
    Q_ASSERT(0 != internal_contactbase);
    QtTapioca::Contact * internal_contact =
        dynamic_cast<QtTapioca::Contact *>(internal_contactbase);
    if (0 == internal_contact) { return; }

    const QtTapioca::Handle * handle = internal_contact->handle();
    Q_ASSERT(0 != handle);
    switch(handle->type())
    {
        case QtTapioca::Handle::Contact:
            d->setPresence(internal_contact, presence, message);
            break;
        case QtTapioca::Handle::Room:
        case QtTapioca::Handle::List:
        case QtTapioca::Handle::None:
            // FIXME: Handle other types of handles...
            break;
        default:
            qWarning() << "Unknown Handle type encountered.";
    }
}

void ContactManager::onNewChannelCreated()
{
    QPair<QtTapioca::Connection *, QtTapioca::Channel *> pair;
    foreach (pair, d->newChannelList)
    {
        emit channelOpened(pair.first, pair.second, false);
    }
    d->newChannelList.clear();
}
