/*
 * Copyright (c) 2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: enma_dkim.c 826 2009-03-26 12:55:05Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: enma_dkim.c 826 2009-03-26 12:55:05Z takahiko $");

#include <stdbool.h>
#include <sys/types.h>

#include "loghandler.h"
#include "authresult.h"
#include "inetmailbox.h"
#include "dkim.h"

#include "enma_dkim.h"

bool
EnmaDkim_evaluate(DkimVerifySession *dkimverifier, AuthResult *authresult)
{
    dkim_stat_t vstat = DkimVerifySession_finishBody(dkimverifier);
    if (DSTAT_ISCRITERR(vstat)) {
        LogError("DkimVerifySession_finishBody failed: err=%s", DKIM_strerror(vstat));
        return false;
    }

    size_t signum = DkimVerifySession_getVerificationNum(dkimverifier);
    if (0 < signum) {
        for (size_t sigidx = 0; sigidx < signum; ++sigidx) {
            const InetMailbox *identity;
            dkim_score_t score =
                DkimVerifySession_getVerifyFrameResult(dkimverifier, sigidx, &identity);
            const char *scoreexp = DkimEnum_lookupScoreByValue(score);
            (void) AuthResult_appendMethodSpec(authresult, AUTHRES_METHOD_DKIM, scoreexp);
            if (NULL != identity) {
                (void) AuthResult_appendPropSpecWithAddrSpec(authresult, AUTHRES_PTYPE_HEADER,
                                                             AUTHRES_PROPERTY_I, identity);
                LogEvent("DKIM-auth", "%s.%s=%s@%s, score=%s", AUTHRES_PTYPE_HEADER,
                         AUTHRES_PROPERTY_I, InetMailbox_getLocalPart(identity),
                         InetMailbox_getDomain(identity), scoreexp);
            } else {
                LogEvent("DKIM-auth", "score=%s", scoreexp);
            }
        }
    } else {
        // DKIM 署名がひとつもない場合のスコアを "none" にする
        const char *scoreexp = DkimEnum_lookupScoreByValue(DKIM_SCORE_NONE);
        (void) AuthResult_appendMethodSpec(authresult, AUTHRES_METHOD_DKIM, scoreexp);
        LogEvent("DKIM-auth", "score=%s", scoreexp);
    }
    return true;
}

bool
EnmaDkimAdsp_evaluate(DkimVerifySession *dkimverifier, AuthResult *authresult)
{
    const char *authorFieldName = DkimVerifySession_getAuthorHeaderName(dkimverifier);
    const InetMailbox *authorMailbox = DkimVerifySession_getAuthorMailbox(dkimverifier);
    if (NULL != authorFieldName && NULL != authorMailbox) {
        dkim_adsp_score_t adsp_score = DkimVerifySession_evalAdsp(dkimverifier);
        if (DKIM_ADSP_SCORE_NULL == adsp_score) {
            LogError("DkimVerifySession_evalAdsp failed: domain=%s",
                     InetMailbox_getDomain(authorMailbox));
            return false;
        }   // end if
        const char *scoreexp = DkimEnum_lookupAdspScoreByValue(adsp_score);
        (void) AuthResult_appendMethodSpec(authresult, AUTHRES_METHOD_DKIMADSP, scoreexp);
        (void) AuthResult_appendPropSpecWithAddrSpec(authresult, AUTHRES_PTYPE_HEADER,
                                                     authorFieldName, authorMailbox);
        LogEvent("DKIM-ADSP-auth", "%s.%s=%s@%s, score=%s", AUTHRES_PTYPE_HEADER,
                 authorFieldName, InetMailbox_getLocalPart(authorMailbox),
                 InetMailbox_getDomain(authorMailbox), scoreexp);
    } else {
        // Author header is missing
        // "permerror" as adsp score
        const char *scoreexp = DkimEnum_lookupAdspScoreByValue(DKIM_ADSP_SCORE_PERMERROR);
        (void) AuthResult_appendMethodSpec(authresult, AUTHRES_METHOD_DKIMADSP, scoreexp);
        LogEvent("DKIM-ADSP-auth", "score=%s", scoreexp);
    }
    return true;
}
