/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimcanon.c 766 2009-03-22 08:16:29Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: dkimcanon.c 766 2009-03-22 08:16:29Z takahiko $");

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <assert.h>
#include <stdbool.h>

#include "dkimlogger.h"
#include "xskip.h"
#include "strtokarray.h"
#include "ptrop.h"
#include "dkim.h"
#include "dkimenum.h"
#include "dkimpolicy.h"
#include "dkimcanon.h"

struct DkimCanon {
    const DkimPolicy *policy;
    unsigned char *buf;
    size_t canonlen;            // buf が保持する意味のある内容のサイズ
    size_t capacity;            // buf に割り当てられているメモリの容量

    dkim_canonalg_t headeralg;
    dkim_canonalg_t bodyalg;

    int body_crlfcnt;
    int body_wspcnt;
    unsigned char body_lastchr;
    unsigned long totalBodyInputLen;

    dkim_stat_t (*canonHeader) (DkimCanon *, const char *, const char *, bool);
    dkim_stat_t (*canonBody) (DkimCanon *, const unsigned char *, size_t);
};

/**
 * DkimCanon オブジェクトの状態をリセットする.
 * ただし, 確保したメモリと canonicalization アルゴリズムは維持される.
 */
void
DkimCanon_reset(DkimCanon *self)
{
    assert(self);
    self->canonlen = 0;
    self->body_crlfcnt = 0;
    self->body_wspcnt = 0;
    self->body_lastchr = '\0';
    self->totalBodyInputLen = 0;
}   // end function : DkimCanon_reset

/**
 * buf メンバに割り当てられているメモリを指定したサイズに拡張する.
 * 既に指定サイズ以上割り当てられている場合は何もしない.
 */
static dkim_stat_t
DkimCanon_assureBuffer(DkimCanon *self, size_t newsize)
{
    if (self->capacity < newsize) {
        unsigned char *newbuf = (unsigned char *) realloc(self->buf, newsize);
        if (NULL == newbuf) {
            self->capacity = 0;
            DkimLogNoResource(self->policy);
            return DSTAT_SYSERR_NORESOURCE;
        }   // end if
        self->buf = newbuf;
        self->capacity = newsize;
    }   // end if

    return DSTAT_OK;
}   // end function : DkimCanon_assureBuffer

/**
 * ヘッダに対する "simple" canonicalization をおこなう.
 * @param headerf ヘッダフィールド名
 * @param headerv ヘッダフィールド値
 * @param append_crlf 末尾に CRLF を付加する場合は true, 何も付加しない場合は false
 */
static dkim_stat_t
DkimCanon_headerWithSimple(DkimCanon *self, const char *headerf,
                           const char *headerv, bool append_crlf)
{
    /*
     * [RFC4871] 3.4.1.
     * The "simple" header canonicalization algorithm does not change header
     * fields in any way.  Header fields MUST be presented to the signing or
     * verification algorithm exactly as they are in the message being
     * signed or verified.  In particular, header field names MUST NOT be
     * case folded and whitespace MUST NOT be changed.
     */

    // バッファへのメモリの割り当て
    // ": ", 末尾の CRLF + NULL 用で計 5 byte
    size_t buflen = strlen(headerf) + strlen(headerv) + strccount(headerv, '\n') + 5;
    dkim_stat_t assure_stat = DkimCanon_assureBuffer(self, buflen);
    if (DSTAT_OK != assure_stat) {
        self->canonlen = 0;
        return assure_stat;
    }   // end if

    // ヘッダ名をそのまま書き出す.
    // ": " を付加するのは milter 仕様のフォロー
    size_t headerflen = 0;
    if (self->policy->sendmail813) {
        // sendmail 8.13 or lower
        headerflen = snprintf((char *) self->buf, self->capacity, "%s: ", headerf);
    } else {
        // sendmail 8.14 or higher
        headerflen = snprintf((char *) self->buf, self->capacity, "%s:", headerf);
    }   // end if
    if (self->capacity <= headerflen) {
        self->canonlen = 0;
        DkimLogImplError(self->policy, "temporary buffer too small");
        return DSTAT_SYSERR_IMPLERROR;
    }   // end if

    // ヘッダ値の書き出し
    // 改行が LF のみの場合に CRLF に変換する.
    const char *p;
    unsigned char *q;
    char last = '\0';
    for (p = headerv, q = self->buf + headerflen; *p != '\0'; ++p) {
        if (*p == '\n' && last != '\r') {
            // LF -> CRLF に変換
            *(q++) = '\r';
        }   // end if
        *(q++) = *p;
        last = *p;
    }   // end for

    if (append_crlf) {
        *(q++) = '\r';
        *(q++) = '\n';
    }   // end if
    *q = '\0';

    assert(q <= self->buf + buflen);
    self->canonlen = q - self->buf;

    return DSTAT_OK;
}   // end function : DkimCanon_headerWithSimple

/**
 * ヘッダに対する "relaxed" canonicalization をおこなう.
 * @param headerf ヘッダフィールド名
 * @param headerv ヘッダフィールド値
 * @param append_crlf 末尾に CRLF を付加する場合は true, 何も付加しない場合は false
 */
static dkim_stat_t
DkimCanon_headerWithRelaxed(DkimCanon *self, const char *headerf,
                            const char *headerv, bool append_crlf)
{
    /*
     * [RFC4871] 3.4.2.
     * The "relaxed" header canonicalization algorithm MUST apply the
     * following steps in order:
     *
     * o  Convert all header field names (not the header field values) to
     *    lowercase.  For example, convert "SUBJect: AbC" to "subject: AbC".
     *
     * o  Unfold all header field continuation lines as described in
     *    [RFC2822]; in particular, lines with terminators embedded in
     *    continued header field values (that is, CRLF sequences followed by
     *    WSP) MUST be interpreted without the CRLF.  Implementations MUST
     *    NOT remove the CRLF at the end of the header field value.
     *
     * o  Convert all sequences of one or more WSP characters to a single SP
     *    character.  WSP characters here include those before and after a
     *    line folding boundary.
     *
     * o  Delete all WSP characters at the end of each unfolded header field
     *    value.
     *
     * o  Delete any WSP characters remaining before and after the colon
     *    separating the header field name from the header field value.  The
     *    colon separator MUST be retained.
     */

    /*
     * - header field name を小文字に
     * - RFC2822 に従って unfold する
     *   + ただし末尾の CRLF は削らないように注意
     * - WSP が連続する箇所を1個の SP で置き換える -> 削除すると "nowsp" になる
     *   + これには folding の CRLF の前後の WSP も含まれる
     * - unfold した後で，行末の全ての WSP を削除する
     * - header field name と header field value を分けるコロンの前後の WSP を削除する
     */

    // バッファへのメモリの割り当て
    // ":", 末尾の CRLF, 末尾の NULL 用で計 4 byte
    size_t buflen = strlen(headerf) + strlen(headerv) + 4;
    dkim_stat_t assure_stat = DkimCanon_assureBuffer(self, buflen);
    if (DSTAT_OK != assure_stat) {
        self->canonlen = 0;
        return assure_stat;
    }   // end if

    // ヘッダ名の書き出し
    const char *p;
    unsigned char *q;
    for (p = headerf, q = self->buf; *p != '\0'; ++p) {
        // header field name には WSP は入らない (RFC2822:ftext) ので
        // WSP 関連の canonicalization はおこなわなくてよい
        *(q++) = tolower(*p);
    }   // end for
    *(q++) = ':';

    // コロン直後の WSP をスキップする.
    for (p = headerv; IS_WSP(*p); ++p);

    // ヘッダ値の書き出し
    bool store_wsp = false;
    for (; *p != '\0'; ++p) {
        if (*p == '\r' || *p == '\n') {
            // header の場合 folding 以外の CR/LF は存在しないはずなので読み飛ばす
        } else if (IS_WSP(*p)) {
            store_wsp = true;
        } else {
            if (store_wsp) {
                // 連続した1つ以上の WSP を 1つの SP で置き換える.
                *(q++) = 0x20;  // この処理を挟まないと "nowsp" になる
                store_wsp = false;
            }   // end if
            *(q++) = *p;
        }   // end if
    }   // end for

    if (append_crlf) {
        *(q++) = '\r';
        *(q++) = '\n';
    }   // end if
    *q = '\0';

    assert(q <= self->buf + buflen);
    self->canonlen = q - self->buf;

    return DSTAT_OK;
}   // end function : DkimCanon_headerWithRelaxed

/**
 * ヘッダに対する canonicalization をおこなう.
 * @param headerf ヘッダフィールド名
 * @param headerv ヘッダフィールド値
 * @param append_crlf 末尾に CRLF を付加する場合は true, 何も付加しない場合は false
 * @param canonbuf canonicalization 済みの文字列を受け取るポインタ.
 *                 バッファは DkimCanon オブジェクト内部に確保され,
 *                 次に DkimCanon オブジェクトに対して操作をおこなうまで有効.
 * @param canonsize canonbuf が保持するデータのサイズ
 * @note headerf, headerv は milter から渡されたものをそのまま渡すといいようになっている.
 */
dkim_stat_t
DkimCanon_header(DkimCanon *self, const char *headerf, const char *headerv,
                 bool append_crlf, const unsigned char **canonbuf, size_t *canonsize)
{
    dkim_stat_t canon_stat = self->canonHeader(self, headerf, headerv, append_crlf);

    if (DSTAT_OK == canon_stat) {
        SETDEREF(canonbuf, self->buf);
        SETDEREF(canonsize, self->canonlen);
    }   // end if

    return canon_stat;
}   // end function : DkimCanon_header

/**
 * DKIM-Signature ヘッダに対する canonicalization をおこなう.
 * @param headerf ヘッダフィールド名
 * @param headerv ヘッダフィールド値
 * @param canonbuf canonicalization 済みの文字列を受け取るポインタ.
 *                 バッファは DkimCanon オブジェクト内部に確保され,
 *                 次に DkimCanon オブジェクトに対して操作をおこなうまで有効.
 * @param canonsize canonbuf が保持するデータのサイズを受け取る変数へのポインタ
 * @attention DKIM-Signature ヘッダの正当性についての検証はおこなわず，
 *            正当な DKIM-Signature であるとして処理をおこなう
 * @note headerf, headerv は milter から渡されたものをそのまま渡すといいようになっている.
 */
dkim_stat_t
DkimCanon_signheader(DkimCanon *self, const char *headerf, const char *headerv,
                     const char *b_tag_value_head, const char *b_tag_value_tail,
                     const unsigned char **canonbuf, size_t *canonsize)
{
    assert(b_tag_value_head != NULL);
    assert(b_tag_value_tail != NULL);

    const char *sign_tail = headerv + strlen(headerv);
    char *buf = (char *) malloc(sign_tail - headerv + 1);
    if (NULL == buf) {
        DkimLogNoResource(self->policy);
        return DSTAT_SYSERR_NORESOURCE;
    }   // end if

    // sig-b-tag 値を抜いてコピーする.
    /*
     * [RFC4871] 3.5.
     * The DKIM-Signature header field being created or verified is always
     * included in the signature calculation, after the rest of the header
     * fields being signed; however, when calculating or verifying the
     * signature, the value of the "b=" tag (signature value) of that DKIM-
     * Signature header field MUST be treated as though it were an empty
     * string.
     *
     * [RFC4871] 3.7.
     * 2.  The DKIM-Signature header field that exists (verifying) or will
     *     be inserted (signing) in the message, with the value of the "b="
     *     tag deleted (i.e., treated as the empty string), canonicalized
     *     using the header canonicalization algorithm specified in the "c="
     *     tag, and without a trailing CRLF.
     */

    // sig-b-tag タグの直前までをコピー
    char *q = buf;
    size_t len = b_tag_value_head - headerv;
    memcpy(q, headerv, len);
    q += len;

    // 残りの部分をコピー (sig-b-tag は1個しか存在しないことを仮定している)
    len = sign_tail - b_tag_value_tail + 1; // 終端の NULL 文字もコピーする
    memcpy(q, b_tag_value_tail, len);

    // DKIM-Signature ヘッダの末尾には CRLF を付加しない.
    dkim_stat_t canon_stat = DkimCanon_header(self, headerf, buf, false, canonbuf, canonsize);

    free(buf);
    return canon_stat;
}   // end function : DkimCanon_signheader

/// 溜め込んでいる CRLF をはき出す
#define FLUSH_CRLF(writep, crlfnum) \
	do { \
		if (0 < crlfnum) { \
			for (int __i = 0; __i < (crlfnum); ++__i) { \
				*((writep)++) = '\r'; \
				*((writep)++) = '\n'; \
			} \
			crlfnum = 0; \
		} \
	} while (0)

/// CRLF を読み込んだことを覚えておく
#define CATCH_CRLF(prevchar, readp, writep, crlfnum) \
	do { \
		if ((prevchar) == '\n') { \
			++(crlfnum); \
			++(readp); \
		} else { \
			FLUSH_CRLF(writep, crlfnum); \
			*((writep)++) = '\r'; \
		} \
	} while (0)

/**
 * メッセージ本文に対する "simple" canonicalization をおこなう.
 * @param bodyp canonicalization 前のメッセージ本文
 * @param bodylen bodyp が保持するデータの長さ
 * @note bodyp, bodylen は milter から渡されたものをそのまま渡すといいようになっている.
 */
static dkim_stat_t
DkimCanon_bodyWithSimple(DkimCanon *self, const unsigned char *bodyp, size_t bodylen)
{
    /*
     * [RFC4871] 3.4.3.
     * The "simple" body canonicalization algorithm ignores all empty lines
     * at the end of the message body.  An empty line is a line of zero
     * length after removal of the line terminator.  If there is no body or
     * no trailing CRLF on the message body, a CRLF is added.  It makes no
     * other changes to the message body.  In more formal terms, the
     * "simple" body canonicalization algorithm converts "0*CRLF" at the end
     * of the body to a single "CRLF".
     *
     * Note that a completely empty or missing body is canonicalized as a
     * single "CRLF"; that is, the canonicalized length will be 2 octets.
     */

    /*
     * - メッセージ末尾の空行を全て削除する.
     *   -> 末尾の CRLF 0*CRLF を CRLF で置き換える.
     */

    // 溜め込んでいる CRLF, CR だけ溜め込んでいる場合に備えた 1, 末尾の NULL に 1
    size_t buflen = bodylen + self->body_crlfcnt * 2 + 2;
    dkim_stat_t assure_stat = DkimCanon_assureBuffer(self, buflen);
    if (DSTAT_OK != assure_stat) {
        self->canonlen = 0;
        return assure_stat;
    }   // end if

    // 各種ポインタの初期化
    const unsigned char *p = bodyp;
    unsigned char *q = self->buf;
    const unsigned char *tail = bodyp + bodylen;

    // 前回の body の最後の文字が CR だった場合
    if (self->body_lastchr == '\r') {
        CATCH_CRLF(*p, p, q, self->body_crlfcnt);
    }   // end if

    for (; p < tail; ++p) {
        if (*p == '\r') {
            if (tail <= p + 1) {
                break;
            }   // end if
            CATCH_CRLF(*(p + 1), p, q, self->body_crlfcnt);
        } else {
            FLUSH_CRLF(q, self->body_crlfcnt);
            *(q++) = *p;
        }   // end if
    }   // end for
    *q = '\0';

    assert(q <= self->buf + buflen);
    self->canonlen = q - self->buf;
    self->body_lastchr = *(tail - 1);   // bodylen が 1 以上であることを確認しているので問題ない
    self->totalBodyInputLen += bodylen;

    return DSTAT_OK;
}   // end function : DkimCanon_bodyWithSimple

///溜め込んでいる CRLF/WSP をはき出す
#define FLUSH_CRLFWSP(writep, crlfnum, wspnum) \
	do { \
		if (crlfnum) { \
			for (int __i = 0; __i < (crlfnum); ++__i) { \
				*((writep)++) = '\r'; \
				*((writep)++) = '\n'; \
			} \
			crlfnum = 0; \
		} \
		if (wspnum) { \
			*((writep)++) = ' '; /* この処理を挟まないと "nowsp" になる */ \
			wspnum = 0; \
		} \
	} while (0)

/// CRLF/WSP を読み込んだことを覚えておく
#define CATCH_CRLFWSP(prevchar, readp, writep, crlfnum, wspnum) \
	do { \
		if ((prevchar) == '\n') { \
			++(crlfnum); \
			(wspnum) = 0; \
			++(readp); \
		} else { \
			FLUSH_CRLFWSP(writep, crlfnum, wspnum); \
			*((writep)++) = '\r'; \
		} \
	} while (0)

/**
 * メッセージ本文に対する "relaxed" canonicalization をおこなう.
 * @param bodyp canonicalization 前のメッセージ本文
 * @param bodylen bodyp が保持するデータの長さ
 * @note bodyp, bodylen は milter から渡されたものをそのまま渡すといいようになっている.
 */
static dkim_stat_t
DkimCanon_bodyWithRelaxed(DkimCanon *self, const unsigned char *bodyp, size_t bodylen)
{
    /*
     * [RFC4871] 3.4.4.
     * The "relaxed" body canonicalization algorithm:
     *
     * o  Ignores all whitespace at the end of lines.  Implementations MUST
     *    NOT remove the CRLF at the end of the line.
     *
     * o  Reduces all sequences of WSP within a line to a single SP
     *    character.
     *
     * o  Ignores all empty lines at the end of the message body.  "Empty
     *    line" is defined in Section 3.4.3.
     *
     *    INFORMATIVE NOTE: It should be noted that the relaxed body
     *    canonicalization algorithm may enable certain types of extremely
     *    crude "ASCII Art" attacks where a message may be conveyed by
     *    adjusting the spacing between words.  If this is a concern, the
     *    "simple" body canonicalization algorithm should be used instead.
     */

    /*
     * - 行末の WSP を削除する
     * - WSP が連続する箇所を1個の SP で置き換える -> 削除すると "nowsp" になる
     * - メッセージ末尾の空行を全て削除する
     *   -> 末尾の CRLF 0*CRLF を CRLF で置き換える
     */

    // 溜め込んでいる CRLF, CR だけ溜め込んでいる場合に備えた 1,
    // WSP だけ溜め込んでいる場合に備えた 1, 末尾の NULL に 1
    size_t buflen = bodylen + self->body_crlfcnt * 2 + 3;
    dkim_stat_t assure_stat = DkimCanon_assureBuffer(self, buflen);
    if (DSTAT_OK != assure_stat) {
        self->canonlen = 0;
        return assure_stat;
    }   // end if

    // 各種ポインタの初期化
    const unsigned char *p = bodyp;
    unsigned char *q = self->buf;
    const unsigned char *tail = bodyp + bodylen;

    // 前回の body の最後の文字が CR だった場合
    if (self->body_lastchr == '\r') {
        CATCH_CRLFWSP(*p, p, q, self->body_crlfcnt, self->body_wspcnt);
    }   // end if

    for (; p < tail; ++p) {
        if (IS_WSP(*p)) {
            self->body_wspcnt = 1;
        } else if (*p == '\r') {
            if (tail <= p + 1) {
                break;
            }   // end if
            CATCH_CRLFWSP(*(p + 1), p, q, self->body_crlfcnt, self->body_wspcnt);
        } else {
            FLUSH_CRLFWSP(q, self->body_crlfcnt, self->body_wspcnt);
            *(q++) = *p;
        }   // end if
    }   // end for
    *q = '\0';

    assert(q <= self->buf + buflen);
    self->canonlen = q - self->buf;
    self->body_lastchr = *(tail - 1);   // bodylen が 1 以上であることを確認しているので問題ない
    self->totalBodyInputLen += bodylen;

    return DSTAT_OK;
}   // end function : DkimCanon_bodyWithRelaxed

/**
 * メッセージ本文に対する canonicalization をおこなう.
 * @param bodyp canonicalization 前のメッセージ本文
 * @param bodylen bodyp が保持するデータの長さ
 * @param canonbuf canonicalization 済みの文字列を受け取るポインタ.
 *                 バッファは DkimCanon オブジェクト内部に確保され,
 *                 次に DkimCanon オブジェクトに対して操作をおこなうまで有効.
 * @param canonsize canonbuf が保持するデータのサイズを受け取る変数へのポインタ
 * @note bodyp, bodylen は milter から渡されたものをそのまま渡すといいようになっている.
 */
dkim_stat_t
DkimCanon_body(DkimCanon *self, const unsigned char *bodyp, size_t bodylen,
               const unsigned char **canonbuf, size_t *canonsize)
{
    if (bodylen == 0) {
        dkim_stat_t assure_stat = DkimCanon_assureBuffer(self, 1);
        if (DSTAT_OK != assure_stat) {
            self->canonlen = 0;
            return assure_stat;
        }   // end if
        self->buf[0] = '\0';
        SETDEREF(canonbuf, self->buf);
        SETDEREF(canonsize, 0);
        return DSTAT_OK;
    }   // end if

    dkim_stat_t canon_stat = self->canonBody(self, bodyp, bodylen);

    if (canon_stat == DSTAT_OK) {
        SETDEREF(canonbuf, self->buf);
        SETDEREF(canonsize, self->canonlen);
    }   // end if

    return canon_stat;
}   // end function : DkimCanon_body

/**
 * メッセージ本文に対する canonicalization を終了する.
 * 溜め込んでいた空白や改行を必要に応じてはき出す.
 * @param canonbuf canonicalization 済みの文字列を受け取るポインタ.
 *                 バッファは DkimCanon オブジェクト内部に確保され,
 *                 次に DkimCanon オブジェクトに対して操作をおこなうまで有効.
 * @param canonsize canonbuf が保持するデータのサイズを受け取る変数へのポインタ
 */
dkim_stat_t
DkimCanon_bodyfin(DkimCanon *self, const unsigned char **canonbuf, size_t *canonsize)
{
    /*
     * [RFC4871] 3.4.3.
     * Note that a completely empty or missing body is canonicalized as a
     * single "CRLF"; that is, the canonicalized length will be 2 octets.
     * 本文が空の場合は CRLF を付加する.
     */
    if (0 == self->totalBodyInputLen) {
        ++(self->body_crlfcnt);
    }   // end if

    // 溜め込んでいる CRLF/WSP, CR だけ溜め込んでいる場合に備えた 1, 末尾の NULL に 1
    size_t buflen = self->body_crlfcnt * 2 + self->body_wspcnt + 2;
    dkim_stat_t assure_stat = DkimCanon_assureBuffer(self, buflen);
    if (DSTAT_OK != assure_stat) {
        self->canonlen = 0;
        return assure_stat;
    }   // end if

    unsigned char *q = self->buf;
    if (self->body_lastchr == '\r') {   // CR を溜め込んでいる場合
        switch (self->bodyalg) {
        case DKIM_CANONALG_SIMPLE:
            FLUSH_CRLF(q, self->body_crlfcnt);
            break;
        case DKIM_CANONALG_RELAXED:
            FLUSH_CRLFWSP(q, self->body_crlfcnt, self->body_wspcnt);
            break;
        default:
            abort();
        }   // end switch
        *(q++) = '\r';
    } else if (self->body_crlfcnt) {    // CRLF を溜め込んでいる場合
        // いくつ CRLF を溜め込んでいても1個の CRLF を出力する
        *(q++) = '\r';
        *(q++) = '\n';
    }   // end if
    // この時点で溜め込まれている WSP は必ず行末のものなので, 無視すればよい

    assert(q <= self->buf + buflen);
    self->canonlen = q - self->buf;
    *q = '\0';
    self->body_crlfcnt = 0;
    self->body_wspcnt = 0;
    self->body_lastchr = '\0';

    SETDEREF(canonbuf, self->buf);
    SETDEREF(canonsize, self->canonlen);

    return DSTAT_OK;
}   // end function : DkimCanon_bodyfin

/**
 * DKIM header/body canonicalization オブジェクトを構築する.
 * @param headeralg ヘッダに適用する canonicalization アルゴリズム
 * @param bodyalg 本文に適用する canonicalization アルゴリズム
 * @return DkimCanon オブジェクト
 */
DkimCanon *
DkimCanon_new(const DkimPolicy *policy, dkim_canonalg_t headeralg, dkim_canonalg_t bodyalg,
              dkim_stat_t *dstat)
{
    DkimCanon *self = (DkimCanon *) malloc(sizeof(DkimCanon));
    if (NULL == self) {
        DkimLogNoResource(policy);
        SETDEREF(dstat, DSTAT_SYSERR_NORESOURCE);
        return NULL;
    }   // end if
    memset(self, 0, sizeof(DkimCanon));

    switch (headeralg) {
    case DKIM_CANONALG_SIMPLE:
        self->canonHeader = DkimCanon_headerWithSimple;
        break;
    case DKIM_CANONALG_RELAXED:
        self->canonHeader = DkimCanon_headerWithRelaxed;
        break;
    default:
        DkimLogPermFail(policy,
                        "unsupported header canonicalization method specified: headercanon=0x%x",
                        headeralg);
        SETDEREF(dstat, DSTAT_PERMFAIL_UNSUPPORTED_CANONALG);
        goto cleanup;
    }   // end switch

    switch (bodyalg) {
    case DKIM_CANONALG_SIMPLE:
        self->canonBody = DkimCanon_bodyWithSimple;
        break;
    case DKIM_CANONALG_RELAXED:
        self->canonBody = DkimCanon_bodyWithRelaxed;
        break;
    default:
        DkimLogPermFail(policy,
                        "unsupported body canonicalization method specified: bodycanon=0x%x",
                        bodyalg);
        SETDEREF(dstat, DSTAT_PERMFAIL_UNSUPPORTED_CANONALG);
        goto cleanup;
    }   // end switch

    self->policy = policy;
    self->headeralg = headeralg;
    self->bodyalg = bodyalg;
    self->totalBodyInputLen = 0;

    SETDEREF(dstat, DSTAT_OK);
    return self;

  cleanup:
    DkimCanon_free(self);
    return NULL;
}   // end function : DkimCanon_new

/**
 * DkimCanon オブジェクトを解放する.
 */
void
DkimCanon_free(DkimCanon *self)
{
    assert(NULL != self);
    free(self->buf);
    free(self);
}   // end function : DkimCanon_free
