/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimtvobj.c 653 2009-02-28 20:35:27Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: dkimtvobj.c 653 2009-02-28 20:35:27Z takahiko $");

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include "ptrop.h"
#include "dkimlogger.h"
#include "xskip.h"

#include "dkim.h"
#include "dkimtvobj.h"

static bool
DkimTvobj_getTagParsedFlag(DkimTvobj *self, DkimTvobjFieldMap *fieldmap)
{
    return STRUCT_MEMBER(bool, self, fieldmap->offset_to_parsed_flag);
}   // end function : DkimTvobj_getTagParsedFlag

static void
DkimTvobj_setTagParsedFlag(DkimTvobj *self, DkimTvobjFieldMap *fieldmap, bool val)
{
    bool *ptag_parsed_flag = (bool *) STRUCT_MEMBER_P(self, fieldmap->offset_to_parsed_flag);
    *ptag_parsed_flag = val;
}   // end function : DkimTvobj_setTagParsedFlag

static void
DkimTvobj_init(DkimTvobj *self)
{
    DkimTvobjFieldMap *fieldmap;
    for (fieldmap = self->ftbl; fieldmap->tagname != NULL; ++fieldmap) {
        DkimTvobj_setTagParsedFlag(self, fieldmap, false);
    }   // end for
}   // end function : DkimTvobj_init

/**
 * @error DSTAT_OK 成功
 * @error DSTAT_PERMFAIL_TAG_DUPLICATED 既にパース済みのタグに遭遇した
 * @error その他コールバック関数中で返されるエラー
 */
static dkim_stat_t
DkimTvobj_dispatchParser(DkimTvobj *self, const DkimTagParseContext *context, const char **nextp)
{
    DkimTvobjFieldMap *fieldmap;
    for (fieldmap = self->ftbl; fieldmap->tagname != NULL; ++fieldmap) {
        // tag-name の比較は case-sensitive におこなう
        // [RFC4871] 3.2.
        // Tags MUST be interpreted in a case-sensitive manner.  Values MUST be
        // processed as case sensitive unless the specific tag description of
        // semantics specifies case insensitivity.
        const char *match_tail;
        if (0 >= XSkip_string(context->taghead, context->tagtail, fieldmap->tagname, &match_tail)
            || context->tagtail != match_tail) {
            continue;   // tag-name にマッチしなかった
        }   // end if

        // tag-name がマッチした場合
        // 既にセットされてないかチェック
        // [RFC4871] 3.2.
        // Tags with duplicate names MUST NOT occur within a single tag-list; if
        // a tag name does occur more than once, the entire tag-list is invalid.
        if (DkimTvobj_getTagParsedFlag(self, fieldmap)) {
            DkimLogPermFail(self->policy, "tag duplicated: %s", fieldmap->tagname);
            return DSTAT_PERMFAIL_TAG_DUPLICATED;
        }   // end if

        if (NULL != fieldmap->ptagparser) {
            // 各 tag に対応する parser の呼び出し
            dkim_stat_t ret = (fieldmap->ptagparser) (self, context, nextp);
            if (ret == DSTAT_OK) {
                // parse 済みフラグを立てる
                DkimTvobj_setTagParsedFlag(self, fieldmap, true);
            }   // end if
            return ret;
        } else {
            // プログラムで処理する必要がないタグ
            break;
        }   // end if
    }   // end for

    // 不明なタグは単に無視する.
    // [RFC4871] 3.2.
    // Unrecognized tags MUST be ignored.
    // [RFC4871] 6.1.1.
    // Note however that this does not include the existence of unknown tags
    // in a DKIM-Signature header field, which are explicitly permitted.

    *nextp = context->valuetail;
    return DSTAT_OK;
}   // end function : DkimTvobj_dispatchParser

/**
 * @error DSTAT_OK 成功
 * @error DSTAT_PERMFAIL_MISSING_REQUIRED_TAG 必須タグがレコード中で指定されていない
 * @error DSTAT_SYSERR_IMPLERROR デフォルト値として設定されている値がパースできない
 * @error その他コールバック関数中で返されるエラー
 */
static dkim_stat_t
DkimTvobj_setDefaultValue(DkimTvobj *self)
{
    DkimTvobjFieldMap *fieldmap;
    DkimTagParseContext context;
    for (fieldmap = self->ftbl; fieldmap->tagname != NULL; ++fieldmap) {
        if (DkimTvobj_getTagParsedFlag(self, fieldmap)) {
            continue;   // 既に値がセットされている場合はスキップ
        }   // end if

        // 値がセットされていない
        if (fieldmap->required) {
            // 必須タグが指定されていない場合はエラーを返す
            DkimLogPermFail(self->policy, "missing required tag: %s", fieldmap->tagname);
            return DSTAT_PERMFAIL_MISSING_REQUIRED_TAG;
        }   // end if

        // 仕様でデフォルト値が定義されているタグにはそれを適用する
        if (NULL != fieldmap->default_value && NULL != fieldmap->ptagparser) {
            dkim_stat_t ret;
            const char *retp;

            context.tagno = -1;
            context.taghead = fieldmap->tagname;
            context.tagtail = STRTAIL(context.taghead);
            context.valuehead = fieldmap->default_value;
            context.valuetail = STRTAIL(context.valuehead);
            ret = (fieldmap->ptagparser) (self, &context, &retp);
            if (DSTAT_OK != ret) {
                DkimLogImplError(self->policy, "default value is unable to parse: %s=%s",
                                 fieldmap->tagname, context.valuehead);
                return DSTAT_SYSERR_IMPLERROR;
            }   // end if
        }   // end if
    }   // end for
    return DSTAT_OK;
}   // end function : DkimTvobj_setDefaultValue

/*
 * [RFC4871]
 * tag-list  =  tag-spec 0*( ";" tag-spec ) [ ";" ]
 * tag-spec  =  [FWS] tag-name [FWS] "=" [FWS] tag-value [FWS]
 * tag-name  =  ALPHA 0*ALNUMPUNC
 * tag-value =  [ tval 0*( 1*(WSP / FWS) tval ) ]
 *                   ; WSP and FWS prohibited at beginning and end
 * tval      =  1*VALCHAR
 * VALCHAR   =  %x21-3A / %x3C-7E
 *                   ; EXCLAMATION to TILDE except SEMICOLON
 * ALNUMPUNC =  ALPHA / DIGIT / "_"
 *
 * @attention 仕様通りに解釈する場合, レコード末尾に余分な CRLF がついているだけでもエラーになるので,
 * tag-list を以下のように独自に拡張して解釈する:
 * tag-list  =  tag-spec 0*( ";" tag-spec ) [ ";" [FWS] ]
 * @error DSTAT_OK 成功
 * @error DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION tag=value ペアの構造に文法エラーがある
 * @error DSTAT_PERMFAIL_MISSING_REQUIRED_TAG 必須タグがレコード中で指定されていない
 * @error DSTAT_PERMFAIL_TAG_DUPLICATED 既にパース済みのタグに遭遇した
 * @error DSTAT_SYSERR_IMPLERROR デフォルト値として設定されている値がパースできない
 * @error その他コールバック関数中で返されるエラー
 */
dkim_stat_t
DkimTvobj_build(DkimTvobj *self, const char *record_head, const char *record_tail,
                bool wsp_restriction)
{
    dkim_stat_t ret;
    DkimTagParseContext context;
    context.tagno = 0;

    // DKIM-Signature, 公開鍵レコードの空白文字には FWS,
    // ADSP レコードの空白文字には WSP が定められているので
    // 切り替えて対応する.
    xskip_funcp sp_skip_func = wsp_restriction ? XSkip_wspBlock : XSkip_fws;

    DkimTvobj_init(self);
    const char *p = record_head;

    do {
        // p が tag-spec の先頭を指すと仮定して parse 開始
        sp_skip_func(p, record_tail, &(context.taghead));

        // tag-name
        if (0 >= XSkip_tagName(context.taghead, record_tail, &(context.tagtail))) {
            DkimLogPermFail(self->policy, "missing tag-name: near %.50s", context.taghead);
            return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
        }   // end if

        sp_skip_func(context.tagtail, record_tail, &p);
        if (0 >= XSkip_char(p, record_tail, '=', &p)) {
            DkimLogPermFail(self->policy, "signature parse error, \'=\' missing: near %.50s",
                            context.taghead);
            return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
        }   // end if
        sp_skip_func(p, record_tail, &(context.valuehead));

        // 0-length tag-value pair is permitted
        XSkip_tagValue(context.valuehead, record_tail, &(context.valuetail));

        // 各 tag の parser に dispatch
        ret = DkimTvobj_dispatchParser(self, &context, &p);
        if (DSTAT_OK != ret) {
            return ret;
        }   // end if
        if (p < context.valuetail) {
            // tag 値に余りがある場合
            DkimLogPermFail(self->policy, "tag-value has unused portion: %d bytes, near %.50s",
                            context.valuetail - p, context.taghead);
            return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
        }   // end if

        // FWS (or WSP) after tag-value
        sp_skip_func(context.valuetail, record_tail, &p);

        // tag-spec を区切るセミコロンがない場合、tag-list の終了と見なしてループを抜ける
        if (0 >= XSkip_char(p, record_tail, ';', &p)) {
            break;
        }   // end if
        // レコード末尾に ';' が許されているので、';' があるからといって次のタグが存在するとは限らない
        ++context.tagno;
    } while (p < record_tail);

    // [spec-modification] accept trailing FWS (or WSP)
    sp_skip_func(p, record_tail, &p);

    if (p < record_tail) {
        // record に余りがある場合
        DkimLogPermFail(self->policy, "record has unused portion: %d bytes, near %.50s",
                        record_tail - p, p);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if

    return DkimTvobj_setDefaultValue(self);
}   // end function : DkimTvobj_build
