/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is part of Skrooge and defines classes SKGUnitObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgunitobject.h"
#include "skgunitvalueobject.h"
#include "skgdocumentbank.h"
#include "skgtraces.h"
#include <math.h>

SKGUnitObject::SKGUnitObject(const SKGDocument* iDocument, int iID)
                : SKGNamedObject(iDocument, "v_unit", iID)
{
}

SKGUnitObject::~SKGUnitObject()
{
}

SKGUnitObject::SKGUnitObject(const SKGUnitObject& iObject)
                : SKGNamedObject(iObject)
{
}

SKGUnitObject::SKGUnitObject(const SKGNamedObject& iObject)
{
        if (iObject.getRealTable()=="unit") {
                copyFrom(iObject);
        } else {
                *this=SKGNamedObject::SKGNamedObject(iObject.getDocument(), "v_unit", iObject.getID());
        }
}

SKGUnitObject::SKGUnitObject(const SKGObjectBase& iObject)
{
        if (iObject.getRealTable()=="unit") {
                copyFrom(iObject);
        } else {
                *this=SKGNamedObject::SKGNamedObject(iObject.getDocument(), "v_unit", iObject.getID());
        }
}

const SKGUnitObject& SKGUnitObject::operator= (const SKGObjectBase& iObject)
{
        copyFrom(iObject);
        return *this;
}

QString SKGUnitObject::getWhereclauseId() const
{
        QString output = SKGObjectBase::getWhereclauseId();
        if (output.isEmpty()) {
                QString name=getName();
                if (!name.isEmpty()) output="t_name='"+SKGServices::stringToSqlString(name)+'\'';

                QString symbol=getSymbol();
                if (!symbol.isEmpty()) {
                        if (!output.isEmpty()) output+=" OR ";
                        output+="t_symbol='"+SKGServices::stringToSqlString(symbol)+'\'';
                }
                if (!output.isEmpty()) output='('+output+')';
        }

        return output;
}
QStringList SKGUnitObject::getListofKnownCurrencies()
{
        QStringList output;
        output
        << tr("Euro (EUR)")
        << tr("United States Dollar (USD)")
        << tr("French Franc (FRF)")
        << tr("Belgium Franc (BEF)")
        << tr("Deutsche mark (DEM)")
        << tr("Espana Peseta (ESP)")
        << tr("Italian Lira (ITL)")
        << tr("Pound sterling (UK)")
        << tr("CAC 40")
        << tr("Dow Jones (DJIA)")
        << tr("NASDAQ")
        << tr("SBF 120")
        ;
        return output;
}

SKGError SKGUnitObject::createCurrencyUnit(const SKGDocumentBank* iDocument, const QString& iUnitName, SKGUnitObject& oUnit)
{
        SKGError err;
        if (iDocument) {
                oUnit=SKGUnitObject(iDocument);

                QString name;
                QString symbol;
                QString country;
                QString internet;
                SKGUnitObject parentUnit;
                QDate date;
                double value=0;
                SKGUnitObject::UnitType type=SKGUnitObject::CURRENCY;
                QString prim=((SKGDocumentBank*) iDocument)->getPrimaryUnit();
                QString seco=((SKGDocumentBank*) iDocument)->getSecondaryUnit();

                if (iUnitName==tr("Euro (EUR)") || iUnitName=="EUR" || iUnitName==QString::fromUtf8("€")) {
                        //Euro
                        name=tr("Euro (EUR)");
                        symbol=QString::fromUtf8("€");
                        country=tr("Europe");
                        date=QDate(2002,1,1);
                        value=1;
                        type=(prim.isEmpty() || prim==symbol ? SKGUnitObject::PRIMARY : SKGUnitObject::CURRENCY);
                        if (prim==QString::fromUtf8("$")) {
                                internet="EURUSD=X";
                                value=-1;

                                parentUnit=SKGUnitObject(iDocument);
                                err=parentUnit.setSymbol(QString::fromUtf8("$"));
                                if (err.isSucceeded()) err=parentUnit.load();
                        }
                } else if (iUnitName==tr("French Franc (FRF)") || iUnitName=="FRF" || iUnitName=="F") {
                        //Franc
                        name=tr("French Franc (FRF)");
                        symbol="F";
                        country=tr("France");
                        date=QDate(1963,1,1);
                        value=1.0/6.55957;
                        type=(seco.isEmpty() || seco==symbol  ? SKGUnitObject::SECONDARY : SKGUnitObject::CURRENCY);

                        err=SKGUnitObject::createCurrencyUnit(iDocument, "EUR", parentUnit);
                } else if (iUnitName==tr("Deutsche mark (DEM)") || iUnitName=="DEM" || iUnitName=="DM") {
                        //Deutch mark
                        name=tr("Deutsche mark (DEM)");
                        symbol="DM";
                        country=tr("Germany");
                        date=QDate(1948,1,1);
                        value=1.0/1.955083;
                        type=(seco.isEmpty() || seco==symbol  ? SKGUnitObject::SECONDARY : SKGUnitObject::CURRENCY);

                        err=SKGUnitObject::createCurrencyUnit(iDocument, "EUR", parentUnit);
                } else if (iUnitName==tr("Espana Peseta (ESP)") || iUnitName=="ESP" || iUnitName=="P") {
                        //Pesetas
                        name=tr("Espana Peseta (ESP)");
                        symbol="P";
                        country=tr("Spain");
                        date=QDate(1868,1,1);
                        value=1.0/166.386;
                        type=(seco.isEmpty() || seco==symbol  ? SKGUnitObject::SECONDARY : SKGUnitObject::CURRENCY);

                        err=SKGUnitObject::createCurrencyUnit(iDocument, "EUR", parentUnit);
                } else if (iUnitName==tr("Italian Lira (ITL)") || iUnitName=="ITL" || iUnitName==QString::fromUtf8("£")) {
                        //Lyre
                        name=tr("Italian Lira (ITL)");
                        symbol=QString::fromUtf8("£");
                        country=tr("Italia");
                        date=QDate(1970,1,1);
                        value=1.0/1936.27;
                        type=(seco.isEmpty() || seco==symbol  ? SKGUnitObject::SECONDARY : SKGUnitObject::CURRENCY);

                        err=SKGUnitObject::createCurrencyUnit(iDocument, "EUR", parentUnit);
                } else if (iUnitName==tr("Belgium Franc (BEF)") || iUnitName=="BEF" || iUnitName=="FB") {
                        //Lyre
                        name=tr("Belgium Franc (BEF)");
                        symbol="FB";
                        country=tr("Belgium");
                        date=QDate(1970,1,1);
                        value=1.0/40.3399;
                        type=(seco.isEmpty() || seco==symbol  ? SKGUnitObject::SECONDARY : SKGUnitObject::CURRENCY);

                        err=SKGUnitObject::createCurrencyUnit(iDocument, "EUR", parentUnit);
                } else if (iUnitName==tr("Pound sterling (UK)") || iUnitName=="GBP" || iUnitName==QString::fromUtf8("£")) {
                        //Lyre
                        name=tr("Pound sterling (UK)");
                        symbol=QString::fromUtf8("£");
                        country=tr("United Kingdom");
                        date=QDate(1990,9,8);
                        value=1;
                        type=(seco.isEmpty() || seco==symbol  ? SKGUnitObject::SECONDARY : SKGUnitObject::CURRENCY);
                        if (prim==QString::fromUtf8("€")) {
                                internet="GBPUSD=X /";
                                value=-1;

                                parentUnit=SKGUnitObject(iDocument);
                                err=parentUnit.setSymbol(QString::fromUtf8("€"));
                                if (err.isSucceeded()) err=parentUnit.load();
                        }
                } else if (iUnitName==tr("United States Dollar (USD)") || iUnitName=="USD" || iUnitName==QString::fromUtf8("$")) {
                        //US Dollar
                        name=tr("United States dollar (USD)");
                        symbol=QString::fromUtf8("$");
                        country=tr("United States");
                        date=QDate(1959,1,1);
                        value=1;
                        type=(prim.isEmpty() || prim==symbol ? SKGUnitObject::PRIMARY : SKGUnitObject::CURRENCY);
                        if (prim==QString::fromUtf8("€")) {
                                internet="EURUSD=X /";
                                value=-1;

                                parentUnit=SKGUnitObject(iDocument);
                                err=parentUnit.setSymbol(QString::fromUtf8("€"));
                                if (err.isSucceeded()) err=parentUnit.load();
                        }
                } else if (iUnitName==tr("CAC 40")) {
                        //US Dollar
                        name=iUnitName;
                        symbol=iUnitName;
                        country=tr("France");
                        date=QDate(1987,1,1);
                        value=-1;
                        internet="^FCHI";
                        type=SKGUnitObject::INDEX;
                } else if (iUnitName==tr("NASDAQ")) {
                        //NASDAQ
                        name=iUnitName;
                        symbol=iUnitName;
                        country=tr("United States");
                        date=QDate(1971,2,5);
                        value=-1;
                        internet="^IXIC";
                        type=SKGUnitObject::INDEX;
                } else if (iUnitName==tr("Dow Jones (DJIA)") || iUnitName=="DJIA") {
                        //Dow Jones
                        name=iUnitName;
                        symbol="DJIA";
                        country=tr("United States");
                        date=QDate(1884,1,1);
                        value=-1;
                        internet="^DJI";
                        type=SKGUnitObject::INDEX;
                } else if (iUnitName==tr("SBF 120")) {
                        //SBF 120
                        name=iUnitName;
                        symbol=iUnitName;
                        country=tr("France");
                        date=QDate(1990,12,31);
                        value=-1;
                        internet="^SBF120";
                        type=SKGUnitObject::INDEX;
                }


                else err=SKGError(ERR_INVALIDARG, tr("Unknown unit [%1]").arg(iUnitName));
                //TODO: other european currencies. see http://fr.wikipedia.org/wiki/Euro#Conversion_de_l.E2.80.99euro_dans_les_anciennes_devises_des_pays_membres

                if (err.isSucceeded()) err = oUnit.setName(name);
                if (err.isSucceeded() && oUnit.exist()) err=oUnit.load();
                if (err.isSucceeded()) err = oUnit.setType(type);
                if (err.isSucceeded()) err = oUnit.setSymbol(symbol);
                if (err.isSucceeded()) err = oUnit.setInternetCode(internet);
                if (err.isSucceeded()) err = oUnit.setCountry(country);
                if (err.isSucceeded() && parentUnit.exist()) err = oUnit.setUnit(parentUnit);
                if (err.isSucceeded()) err = oUnit.save();

                //Creation of the value
                if (value>0) {
                        SKGUnitValueObject unitValue;
                        if (err.isSucceeded()) err = oUnit.addUnitValue(unitValue);
                        if (err.isSucceeded()) err = unitValue.setDate(date);
                        if (err.isSucceeded()) err = unitValue.setQuantity(value);
                        if (err.isSucceeded()) err = unitValue.save();
                }
        }
        return err;
}

double SKGUnitObject::convert(double iValue, const SKGUnitObject& iUnitFrom, const SKGUnitObject& iUnitTo)
{
        double output=iValue;
        if (iUnitFrom!=iUnitTo) {
                double valFrom=SKGServices::stringToDouble(iUnitFrom.getAttribute("f_CURRENTAMOUNT"));
                double valTo=SKGServices::stringToDouble(iUnitTo.getAttribute("f_CURRENTAMOUNT"));
                output=iValue*valFrom/valTo;
        }
        return output;
}

SKGError SKGUnitObject::setSymbol(const QString& iSymbol)
{
        return setAttribute("t_symbol", iSymbol);
}

QString SKGUnitObject::getSymbol() const
{
        return getAttribute("t_symbol");
}

SKGError SKGUnitObject::setCountry(const QString& iCountry)
{
        return setAttribute("t_country", iCountry);
}

QString SKGUnitObject::getCountry() const
{
        return getAttribute("t_country");
}

SKGError SKGUnitObject::setInternetCode(const QString& iCode)
{
        return setAttribute("t_internet_code", iCode);
}

QString SKGUnitObject::getInternetCode() const
{
        return getAttribute("t_internet_code");
}

SKGError SKGUnitObject::setType(SKGUnitObject::UnitType iType)
{
        SKGError err;
        if (getAttribute("t_type").length()==0 || this->getType()!=iType) {
                //Guaranty that PRIMARY and SECONDARY is unique
                if (iType==PRIMARY || iType==SECONDARY) {
                        //Set old SECONDARY as CURRENCY
                        err=SKGServices::executeSqliteOrder(getDocument(), "UPDATE unit SET t_type='C' WHERE t_type='2'");

                        //Set old PRIMARY as SECONDARY
                        if (err.isSucceeded() && iType==PRIMARY) err=SKGServices::executeSqliteOrder(getDocument(), "UPDATE unit SET t_type='2' WHERE t_type='1'");
                }
        }
        if (err.isSucceeded()) err=setAttribute("t_type", (iType==CURRENCY ? "C" : (iType==PRIMARY ? "1" : (iType==SECONDARY ? "2" : (iType==SHARE ? "S" : (iType==INDEX ? "I" : "O"))))));
        return err;
}

SKGUnitObject::UnitType SKGUnitObject::getType() const
{
        QString typeString=getAttribute("t_type");
        return (typeString=="C" ? CURRENCY : (typeString=="S" ? SHARE : (typeString=="1" ? PRIMARY : (typeString=="2" ? SECONDARY : (typeString=="I" ? INDEX : OBJECT)))));
}

SKGError SKGUnitObject::getUnit(SKGUnitObject& oUnit) const
{
        SKGError err = getObject(getDocument(), "v_unit", "id=" + getAttribute("rd_unit_id"), oUnit);
        return err;
}

SKGError SKGUnitObject::setUnit(const SKGUnitObject& iUnit)
{
        return setAttribute("rd_unit_id", SKGServices::intToString(iUnit.getID()));
}

SKGError SKGUnitObject::removeUnit()
{
        return setAttribute("rd_unit_id", "0");
}

SKGError SKGUnitObject::addUnitValue(SKGUnitValueObject& oUnitValue)
{
        SKGError err;
        if (getID() == 0) err=SKGError(ERR_FAIL, tr("%1 failed because linked object is not yet saved in the database.").arg("SKGUnitObject::addUnitValue"));
        else {
                oUnitValue = SKGUnitValueObject((SKGDocumentBank*) getDocument());
                err = oUnitValue.setAttribute("rd_unit_id", SKGServices::intToString(getID()));
        }
        return err;
}

SKGError SKGUnitObject::getLastUnitValue(SKGUnitValueObject& oUnitValue) const
{
        return SKGObjectBase::getObject(getDocument(), "v_unitvalue",
                                        "rd_unit_id=" + SKGServices::intToString(getID()) +" AND d_date=(select MAX(u2.d_date) from unitvalue u2 where u2.rd_unit_id=" + SKGServices::intToString(getID()) + ')',
                                        oUnitValue);
}

SKGError SKGUnitObject::getUnitValue(const QDate& iDate, SKGUnitValueObject& oUnitValue) const
{
        QString ids=SKGServices::intToString(getID());
        QString dates=SKGServices::dateToSqlString(QDateTime(iDate));
        SKGError err=SKGObjectBase::getObject(getDocument(), "v_unitvalue",
                                              "rd_unit_id=" + ids +" AND d_date<='"+dates+
                                              "' AND  ABS(strftime('%s','"+dates+
                                              "')-strftime('%s',d_date))=(select MIN(ABS(strftime('%s','"+dates+
                                              "')-strftime('%s',u2.d_date))) from unitvalue u2 where u2.rd_unit_id=" + ids +
                                              " AND u2.d_date<='"+dates+"')",
                                              oUnitValue);

        //If not found then get first
        if (err.isFailed()) err=SKGObjectBase::getObject(getDocument(), "v_unitvalue",
                                        "rd_unit_id=" + SKGServices::intToString(getID()) +" AND d_date=(select MIN(d_date) from unitvalue where rd_unit_id=" +
                                        SKGServices::intToString(getID()) + ')',
                                        oUnitValue);
        return err;
}

double SKGUnitObject::getAmount(const QDate& iDate) const
{
        double output=0;

        //Search result in cache
        QString ids=SKGServices::intToString(getID());
        QString dates=SKGServices::dateToSqlString(QDateTime(iDate));
        QString key="unitvalue-"+ids+'-'+dates;
        QString val=getDocument()->getCachedValue(key);
        if (val.isEmpty()) {
                //Get quantity
                double quantity=1;
                SKGUnitValueObject uv;
                if (getUnitValue(iDate, uv).isSucceeded()) quantity=uv.getQuantity();

                SKGUnitObject unit;
                double coef=1;
                if (getUnit(unit).isSucceeded()) coef=unit.getAmount(iDate);

                output=coef*quantity;
                ((SKGDocument*) getDocument())->addValueInCache(key, SKGServices::doubleToString(output));

                if (getAttribute("i_NBVALUES")=="1") {
                        //Store value for this symbol for all date
                        ((SKGDocument*) getDocument())->addValueInCache("unitvalue-"+ids, SKGServices::doubleToString(output));
                }
        } else {
                output=SKGServices::stringToDouble(val);
        }
        return output;

}

double SKGUnitObject::getDailyChange(const QDate& iDate) const
{
        double output=0;
        SKGStringListList result;
        SKGError err=SKGServices::executeSelectSqliteOrder(getDocument(),
                        "SELECT d_date, f_quantity from unitvalue where rd_unit_id="+
                        SKGServices::intToString(getID())+
                        " AND d_date<='"+SKGServices::dateToSqlString(QDateTime(iDate))+
                        "' ORDER BY d_date DESC LIMIT 2",
                        result);
        if (err.isSucceeded() && result.count()==3) {
                double v2=SKGServices::stringToDouble(result.at(1).at(1));
                double v1=SKGServices::stringToDouble(result.at(2).at(1));

                QDate d2=SKGServices::stringToTime(result.at(1).at(0)).date();
                QDate d1=SKGServices::stringToTime(result.at(2).at(0)).date();

                output=100*(exp(log(v2/v1)/(d1.daysTo(d2)))-1);
        }
        return output;
}

SKGError SKGUnitObject::split(double iRatio) const
{
        SKGError err;
        if (iRatio>0) {

                err=SKGServices::executeSqliteOrder(getDocument(), "UPDATE unitvalue SET f_quantity=f_quantity/"+SKGServices::doubleToString(iRatio)+
                                                    " WHERE rd_unit_id="+SKGServices::intToString(getID()));
                if (err.isSucceeded()) err=SKGServices::executeSqliteOrder(getDocument(), "UPDATE suboperation SET f_value=f_value*"+SKGServices::doubleToString(iRatio)+
                                                   " WHERE rd_operation_id IN (SELECT id FROM operation WHERE rc_unit_id="+SKGServices::intToString(getID())+')');
        } else err=SKGError(ERR_INVALIDARG, tr("Invalid ratio. Ratio must be greater than 0."));
        return err;
}

#include "skgunitobject.moc"
