/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGDOCUMENT_H
#define SKGDOCUMENT_H
/** @file
 * This file is part of Skrooge and defines classes SKGDocument.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgservices.h"

#include <QStringList>
#include <QDateTime>
#include <QHash>

class QSqlDatabase;
class SKGObjectBase;
class SKGError;

/**
 * This structure defines a type of modification
 */
enum KDE_EXPORT ModificationType {
        U, /**< Update */
        I, /**< Insert */
        D  /**< Delete */
};
/**
* Describe a modification of an object
*/
struct KDE_EXPORT SKGObjectModification {
        QString uuid; 				/**< The uuid of the object */
        int id; 				/**< The id of the object */
        QString table; 				/**< The table of the object */
        ModificationType type; 			/**< The type of modification */
} ;

/**
 * A list of SKGObjectModification ==> SKGObjectModificationList
 */
typedef QList<SKGObjectModification> SKGObjectModificationList;

/**
* This class manages skg documents
*/
class KDE_EXPORT SKGDocument : public QObject
{
        Q_OBJECT;
        Q_ENUMS(UndoRedoMode);

public:
        /**
         * This structure defines the direction of the UNDO / REDO mechanism
         */
        enum UndoRedoMode {
                UNDOLASTSAVE, 	/**< To do/get an undo=cancel of the last successfully extecuted transactions until last save */
                UNDO, 		/**< To do/get an undo=cancel of the last successfully extecuted transaction */
                REDO  		/**< To do/get a redo=replay the last cancelled transaction */
        };

        /**
         * This structure defines the direction of the UNDO / REDO mechanism
         */
        enum DatabaseMode {
                CopiedInMemory, 	/**< The sqlite database is copied in memory (Very good for select, Not good for load ans Save) */
                CopiedInFile, 		/**< The sqlite database is copied into a temporary file (Good for select, Good for load ans Save) */
                DirectAccess, 		/**< The sqlite database is directly opened (Good for select, Good for load ans Save, save is automatically done) */
        };

        /**
         * Constructor
         * @param iMode the mode of the database
         */
        SKGDocument(DatabaseMode iMode=SKGDocument::CopiedInMemory);

        /**
         * Destructor
         */
        virtual ~SKGDocument();

        /**
         * Get the database mode
         * @return the database mode
         */
        virtual SKGDocument::DatabaseMode getDatabaseMode();

        /**
         * Set the callback function to follow the progress of the transaction.
         * the first parameter is the progress between 0% and 100%.
         * the callback must return 0 to continue and !=0 to cancel the transaction.
         * @param iProgressFunction the pointer of the function
         * @param iData the data for the progress call back
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError setProgressCallback(int (*iProgressFunction)(int, void*), void* iData=NULL);

        /**
         * Call the progress callstack.
         * @param iPosition the position in the current transaction.
         * The value must be between 0 and the value passed to beginTransaction.
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError stepForward(int iPosition);

        /**
         * Start a transaction.
         * A transaction is needed to modify the SKGDocument.
         * This transaction is also used to manage the undo/redo.
         * @see endTransaction
         * @param iName the name of the transaction
         * @param iNbStep the number of step in this transaction.
         * It's used to call the progress callback.
         * @param iDate date of the transaction.
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError beginTransaction(const QString & iName, int iNbStep=0, const QDateTime& iDate=QDateTime::currentDateTime());

        /**
         * close the current transaction.
         * A transaction is needed to modify the SKGDocument.
         * This transaction is also used to manage the undo/redo.
         * @see beginTransaction
         * @param succeedded : true to indicate that current transaction has been successfully executed
         *                   : false to indicate that current transaction has failed
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError endTransaction(bool succeedded);

        /**
         * Send a message attached to the current transaction.
         * @param iMessage the message
         * @param iPopup to create a popup message
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError sendMessage(const QString& iMessage, bool iPopup=true);

        /**
         * Get message attached to a current transaction.
         * @param iIdTransaction the identifier of a transaction
         * @param oMessages the messages
         * @param iAll to get all message (true) or only popup message (false)
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError getMessages(int iIdTransaction, QStringList& oMessages, bool iAll=true);

        /**
         * Get list of direct modifications done in a transaction
         * @param iIdTransaction the identifier of a transaction
         * @param oModifications list of modifications
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError getModifications(int iIdTransaction, SKGObjectModificationList& oModifications) const;

        /**
         * Undo or redo the last transaction.
         * @param iMode the mode
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError undoRedoTransaction(const UndoRedoMode& iMode = SKGDocument::UNDO);

        /**
         * Group transactions
         * @param iFrom the first id transaction of the group, it will be the master of the group.
         * @param iTo the last id transaction of the group.
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError groupTransactions(int iFrom, int iTo);

        /**
         * Return the number of transaction stored including the current one
        * @param iMode the mode
        * @return the number of transaction
         */
        virtual int getNbTransaction(const UndoRedoMode& iMode = SKGDocument::UNDO) const;

        /**
         * Return the internal identifier of the transaction
         * which must be treated for an undo or a redo
         * @param iMode the mode
         * @param oName if you want also the name of the transaction
         * @param oSaveStep if you want also to know if it's a save step
         * @param oDate if you want also the date of the transaction
         * @return the internal identifier of the last transaction
         * 0 if no transaction found
         */
        virtual int getTransactionToTreat(const UndoRedoMode& iMode = SKGDocument::UNDO, QString* oName = NULL,
                                          bool* oSaveStep=NULL, QDateTime* oDate=NULL) const;

        /**
         * Return the identifier of the current transaction
         * @return the internal identifier of the current transaction
         * 0 if no transaction found
         */
        virtual int getCurrentTransaction() const;

        /**
         * Return the depth of the current transaction
         * @return the depth
         */
        virtual int getDepthTransaction() const;

        /**
         * To know if a transaction is opened or not
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError checkExistingTransaction() const;

        /**
         * Change the passord of the document.
         * WARNING: This method must be used in a transaction.
         * @see beginTransaction
         * @param inewPassword the new password
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError changePassword(const QString & inewPassword);

        /**
         * Initialize a new document.
         * WARNING: This method must NOT be used in a transaction.
         * @see endTransaction
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError initialize();

        /**
         * load an existing document.
         * WARNING: This method must NOT be used in a transaction.
         * @see endTransaction
         * @param iName the file name to load.
         * @param iPassword the password of the SKGDocument.
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError load(const QString & iName, const QString & iPassword="");

        /**
         * To know if the current document has been modified or not.
         * @return true: the document has been modified, save is possible/needed.
         *         false: the document hasn't been modified, save is not needed.
         */
        virtual bool isFileModified() const;

        /**
         * Return the file name of the current document.
         * To set it, you must use saveAs.
         * @return the file name of the current document.
         */
        virtual QString getCurrentFileName() const;

        /**
         * save the current SKGDocument.
         * WARNING: This method must NOT be used in a transaction.
         * @see endTransaction.
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError save();

        /**
         * save the current SKGDocument.
         * WARNING: This method must NOT be used in a transaction.
         * @see endTransaction
         * @param iName the file name to save.
         * @param iOverwrite to authorize the overwrite or not.
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError saveAs(const QString & iName, bool iOverwrite=false);

        /**
         * close the current SKGDocument.
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError close();

        /**
         * Set the message associated to this SKGError.
         * @param iName the parameter unique identifier.
         */
        virtual QString getParameter(const QString& iName) const;

        /**
         * Set a parameter.
         * WARNING: This method must be used in a transaction.
         * @see beginTransaction
         * @param iName the parameter unique identifier.
         * @param iValue the parameter value.
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError setParameter(const QString& iName, const QString& iValue) const;

        /**
         * Get the database pointer.
         * @return the database pointer.
         *   MUST NOT BE REMOVED
         */
        virtual QSqlDatabase* getDatabase() const;

        /**
         * dump the document in the std output.
         * It's useful for debug.
         * @param iMode the select what you want to dump.
         * @code
         * document->dump (DUMPPARAMETERS|DUMPTRANSACTIONS);
         * @endcode
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError dump(int iMode = DUMPPARAMETERS | DUMPTRANSACTIONS) const;


        /**
         * Create a consolidated view
         * @param iTable Table name
         * @param iAsColumn Attribute used as column names
         * @param iAsRow Attribute used as lines names
         * @param iAttribute Attribute
         * @param iOpAtt Operation to apply on @p iAttribute
         * @param iWhereClause Where clause
         * @param oTable the consolidated view
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError getConsolidatedView(const QString& iTable,
                                             const QString& iAsColumn,
                                             const QString& iAsRow,
                                             const QString& iAttribute,
                                             const QString& iOpAtt,
                                             const QString& iWhereClause,
                                             SKGStringListList& oTable) const;

        /**
         * Refresh all views and indexes in the database
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError refreshViewsIndexesAndTriggers() const;

        /**
         * Get the display string for any modeler object (table, attribute)
         * @param iString the name of the object (example: v_operation, v_unit.t_name)
         * @return the display string
         */
        virtual QString getDisplay(const QString& iString) const;

        /**
         * Add a value in the cache. Cache is cleaned after each transaction
         * @param iKey the key
         * @param iValue the value
         */
        virtual void addValueInCache(const QString& iKey, const QString& iValue);

        /**
         * Get the cached value
         * @param iKey key
         * @return the value
         */
        virtual QString getCachedValue(const QString& iKey) const;

signals:
        /**
         * This signal is launched by endTransaction on all tables modified when a huge modification occures on the model
         * @param iTableName the name of the modified table. iTableName="" if all tables must be refreshed
         * @param iIdTransaction the id of the transaction for direct modifications of the table (update of modify objects is enough)
         *or 0 in case of modifications by impact (full table must be refreshed)
         */
        void tableModified(const QString& iTableName, int iIdTransaction);

        /**
         * This signal is launched by endTransaction when a transaction is successfully ended
         * @param iIdTransaction the id of the transaction for direct modifications of the table (update of modify objects is enough)
         */
        void transactionSuccessfullyEnded(int iIdTransaction);

        /**
         * This signal is launched by endTransaction on all objects added in the model
         * @param iObject the object added.
         */
        void objectAdded(const SKGObjectBase& iObject);

        /**
         * This signal is launched by endTransaction on all objects updated in the model
         * @param iObject the object updated.
         */
        void objectUpdated(const SKGObjectBase& iObject);

        /**
         * This signal is launched by endTransaction on all objects removed in the model
         * @param iObject the object removed.
         */
        void objectRemoved(const SKGObjectBase& iObject);

protected:
        /**
         * Migrate the current SKGDocument to the latest version of the data model.
         * WARNING: This method must be used in a transaction.
         * @see beginTransaction
         * @param oMigrationDone to know if a migration has been done or not.
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError migrate(bool& oMigrationDone);

        /**
         * Create dynamic triggers for "undoable" tables.
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError createUndoRedoTemporaryTriggers() const;

        /**
         * This list must contain a list of object
         * where the undo / redo is NOT applicable.
         * For a full table, the syntax is: T.nameofthetable
         * For an attribute, the syntax is: A.nameofthetable.nameoftheattribute
         */
        QStringList SKGListNotUndoable;

        /**
         * Get impacted tables if one object of @p iTable is modifier.
         * @param iTable name of a table
         * @return impacted tables
         */
        virtual QStringList getImpactedTable(const QString& iTable) const;

private:
        Q_DISABLE_COPY(SKGDocument);


        static SKGError lastCallbackError;

        static int databaseUniqueIdentifier;

        int lastSavedTransaction;
        void* progressFunction;
        void* progressData;
        QString currentFileName;
        QString databaseIdentifier;
        QSqlDatabase* currentDatabase;
        SKGIntList nbStepForTransaction;
        SKGIntList posStepForTransaction;
        int inundoRedoTransaction;
        int currentTransaction;
        DatabaseMode mode;
        QString temporaryFile;
        QStringList unTransactionnalMessages;

        QHash<QString, QString> cache;
        bool inProgress;
};

#endif
