/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGOBJECTBASE_H
#define SKGOBJECTBASE_H
/** @file
 * This file is part of Skrooge and defines classes SKGObjectBase.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgservices.h"
#include "skgerror.h"
#include "skgdefine.h"

#include <QList>
#include <QString>

class SKGDocument;
/**
 * This class is the base class for all objects.
 * This class is a generic way to manipulate objects.
 * Some important rules:
 * R1: Each table must have a view named v_[tablename] where [tablename] is the name of the table.
 * R2: Each computed attribute of the view must have a name starting with v_
 */
class KDE_EXPORT SKGObjectBase : public QObject
{
        Q_OBJECT;
        Q_PROPERTY( QString uniqueID READ getUniqueID);
        Q_PROPERTY( int ID READ getID);
        Q_PROPERTY( QString table READ getTable);

public:
        /**
         * A list of SKGObjectBase ==> SKGListSKGObjectBase
         */
        typedef QList<SKGObjectBase> SKGListSKGObjectBase;

        /**
         * An iterator for SKGListSKGObjectBase
         */
        typedef QList<SKGObjectBase>::Iterator SKGListSKGObjectBaseIterator;

        /**
         * Return a number of objects (@p oNbObjects) corresponding to a where clause (@p iWhereClause )
         * @param iDocument the document where to search
         * @param iTable the table where to search
         * @param iWhereClause the where clause
         * @param oNbObjects the result
         * @return an object managing the error
         *   @see SKGError
         */
        static SKGError getNbObjects(const SKGDocument* iDocument, const QString& iTable,
                                     const QString& iWhereClause, int& oNbObjects);

        /**
         * Return a list of objects (@p oListObject) corresponding to a where clause (@p iWhereClause )
         * @param iDocument the document where to search
         * @param iTable the table where to search
         * @param iWhereClause the where clause
         * @param oListObject the result
         * @return an object managing the error
         *   @see SKGError
         */
        static SKGError getObjects(const SKGDocument* iDocument, const QString& iTable,
                                   const QString& iWhereClause, SKGListSKGObjectBase& oListObject);

        /**
         * Return the object (@p oObject) corresponding to a where clause (@p iWhereClause ).
         * If more than one objects are returned by the query, then an error is generated
         * If 0 object is returned by the query, then an error is generated
         * @param iDocument the document where to search
         * @param iTable the table where to search
         * @param iWhereClause the where clause
         * @param oObject the result
         * @return an object managing the error
         *   @see SKGError
         */
        static SKGError getObject(const SKGDocument* iDocument, const QString& iTable,
                                  const QString& iWhereClause, SKGObjectBase& oObject);

        /**
         * Return the object (@p oObject) corresponding to an id (@p iId ).
         * If more than one objects are returned by the query, then an error is generated
         * If 0 object is returned by the query, then an error is generated
         * @param iDocument the document where to search
         * @param iTable the table where to search
         * @param iId id of the object in @p iTable
         * @param oObject the result
         * @return an object managing the error
         *   @see SKGError
         */
        static SKGError getObject(const SKGDocument* iDocument, const QString& iTable,
                                  int iId, SKGObjectBase& oObject);

        /**
         * Constructor
         * @param iDocument the document containing the object
         * @param iTable the table of the object
         * @param iID the identifier in @p iTable of the object
         */
        explicit SKGObjectBase(const SKGDocument* iDocument=NULL, const QString& iTable="", int iID=0);

        /**
         * Copy constructor
         * @param iObject the object to copy
         */
        SKGObjectBase(const SKGObjectBase& iObject);

        /**
         * Operator comparison
         * @param iObject The object to compare
         */
        virtual bool operator==(const SKGObjectBase& iObject) const;

        /**
         * Operator comparison
         * @param iObject The object to compare
         */
        virtual bool operator!=(const SKGObjectBase& iObject) const;

        /**
         * Operator affectation
         * @param iObject the object to copy
         */
        virtual const SKGObjectBase& operator= (const SKGObjectBase &iObject);

        /**
         * Destructor
         */
        virtual ~SKGObjectBase();

        /**
         * Copy from an existing object
         * @param iObject the object to copy
         */
        void copyFrom(const SKGObjectBase& iObject);

        /**
         * Return the unique id of this object
         * @return unique id of this object
         */
        virtual QString getUniqueID() const;

        /**
         * Return the id of this object
         * @return id of this object
         */
        virtual int getID() const;

        /**
         * Return the name of the object for the display
         * @return name of the object
         */
        virtual QString getDisplayName() const;

        /**
         * Reset the ID of this object.
         * It's used to create a new object based on an existing one.
         * By reseting the ID, the save will try an INSERT instead of an UPDATE.
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError resetID();

        /**
         * Return the table name of this object
         * @return table name of this object
         */
        virtual QString getTable() const;

        /**
         * Return the real table name of this object.
         * This is useful for modification (UPDATE, DELETE)
         * @return real table name of this object
         */
        virtual QString getRealTable() const;

        /**
         * Return the document of this object
         * @return document of this object
         */
        virtual const SKGDocument* getDocument() const;

        /**
         * Return the attributes of this object
         * @return attributes of this object
         */
        virtual SKGQStringQStringMap getAttributes() const;

        /**
         * Return the number of attributes
         * @return number of attributes
         */
        virtual int getNbAttributes() const;

        /**
         * Get the value for an attribute
         * @param iName the name of the attribute
         * @return the value of this attribute
         */
        virtual QString getAttribute(const QString& iName) const;

        /**
         * Set the value for an attribute
         * @param iName the name of the attribute
         * @param iValue the value of the attribute
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError setAttribute(const QString& iName, const QString& iValue);

        /**
         * Set the values for attributes
         * @param iNames list of attributes
         * @param iValues list of values
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError setAttributes(const QStringList& iNames, const QStringList& iValues);

        /**
         * Get the list of properties
         * @return the list of properties
         */
        virtual QStringList getProperties() const;

        /**
         * Get the value for a property
         * @param iName the name of the property
         * @return the value of this property
         */
        virtual QString getProperty(const QString& iName) const;

        /**
         * Get the property object
         * @param iName the name of the property
         * @return the property object
         */
        virtual SKGObjectBase getPropertyObject(const QString& iName) const;

        /**
         * Set the value for a property
         * @param iName the name of the property
         * @param iValue the value of the property
         * @return an object managing the property
         *   @see SKGError
         */
        virtual SKGError setProperty(const QString& iName, const QString& iValue);

        /**
         * To know if an object exists or not
         * @return "true" if the object exists else "false".
         */
        virtual bool exist() const;

        /**
         * load or reload the object from the database
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError load();

        /**
         * save the object into the database
         * @param iInsertOrUpdate the save mode.
         *    true: try an insert, if the insert failed then try an update.
         *    false: try an insert, if the insert failed then return an error.
         * @param iReloadAfterSave to reload the object after save. Set false if you are sure that you will not use this object after save
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError save(bool iInsertOrUpdate=true, bool iReloadAfterSave=true);

        /**
         * delete the object into the database
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError remove() const;

        /**
         * dump the object
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError dump();

protected:
        /**
         * Get where clause needed to identify objects.
         * For this class, the whereclause is based on id
         * @return the where clause
         */
        virtual QString getWhereclauseId() const;

private:
        /**
         * internal id
         */
        int id;

        /**
         * internal table
         */
        QString table;

        /**
         * internal document
         */
        const SKGDocument* document;

        /**
         * internal attributes
         */
        SKGQStringQStringMap attributes;
};

#endif
