/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for bank management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgbankpluginwidget.h"
#include "skgsortfilterproxymodel.h"
#include "skgmainpanel.h"
#include "skgobjectmodel.h"
#include "skgtransactionmng.h"
#include "skgbankobject.h"
#include "skgaccountobject.h"
#include "skgdocumentbank.h"
#include "skgtraces.h"

#include <kstandarddirs.h>

#include <QDomDocument>
#include <QDir>

SKGBankPluginWidget::SKGBankPluginWidget(SKGMainPanel* iParent, SKGDocumentBank* iDocument)
                : SKGTabWidget(iParent, iDocument)
{
        SKGTRACEIN(10, "SKGBankPluginWidget::SKGBankPluginWidget");

        ui.setupUi(this);
        //Add Standard KDE Icons to buttons to Accounts
        ui.kAccountCreatorUpdate->setIcon(KIcon("dialog-ok-apply"));
        ui.kAccountCreatorAdd->setIcon(KIcon("list-add"));

        //Add NLS values for type of account
        //C=current D=credit card P=passif (for objects) I=Investment O=other
        ui.kAccountCreatorType->addItem(i18n("Current"));
        ui.kAccountCreatorType->addItem(i18n("Credit card"));
        ui.kAccountCreatorType->addItem(i18n("Investment"));
        ui.kAccountCreatorType->addItem(i18n("Actif"));
        ui.kAccountCreatorType->addItem(i18n("Other"));

        //Bind account creation view
        objectModel = new SKGObjectModel((SKGDocumentBank*) getDocument(), "v_account_display", "1=0", this);
        SKGSortFilterProxyModel* modelProxy = new SKGSortFilterProxyModel(this);
        modelProxy->setSourceModel(objectModel);
        modelProxy->setSortRole(Qt::UserRole);
        modelProxy->setDynamicSortFilter(true);

        ui.kAccountTableViewEdition->setModel(modelProxy);
        ui.kAccountTableViewEdition->setWindowTitle(i18n("Accounts"));
        ui.kAccountTableViewEdition->setDefaultSaveParameters(getDocument(), "SKG_DEFAULT_ACCOUNT");

        //Add registered global action in contextual menu
        ui.kAccountTableViewEdition->insertAction(0, iParent->getGlobalAction("edit_delete"));
        QAction* sep=new QAction(this);
        sep->setSeparator(true);
        ui.kAccountTableViewEdition->insertAction(0, sep);
        ui.kAccountTableViewEdition->insertAction(0, iParent->getGlobalAction("open_report"));

        connect(ui.kAccountTableViewEdition->selectionModel(), SIGNAL(selectionChanged(QItemSelection,QItemSelection)),
                this, SLOT(onSelectionChanged()));

        connect(objectModel, SIGNAL(modelAboutToBeReset()), ui.kAccountTableViewEdition, SLOT(saveSelection()));
        connect(objectModel, SIGNAL(modelReset()), ui.kAccountTableViewEdition, SLOT(resetSelection()));

        ui.kAccountTableViewEdition->sortByColumn(0, Qt::AscendingOrder);

        //Logo for banks
        ui.kAccountCreatorIcon->addItem("");
        QDir dirLogo(KStandardDirs::locate("data", QString::fromLatin1("skrooge/images/logo/")+KGlobal::locale()->language()+'/'));
        dirLogo.setFilter(QDir::Files);
        QStringList filters;
        filters << "*.png";
        dirLogo.setNameFilters(filters);
        QStringList listLogo = dirLogo.entryList();
        int nb=listLogo.size();
        for (int i=0; i<nb ; ++i) {
                QString bankName=listLogo.at(i);
                bankName.remove(".png");
                bankName.replace('_', ' ');

                QRegExp rx("(.+) {2,}(.+)");
                if (rx.indexIn(bankName)!=-1) {
                        //Icon is something like <bank name>__<banknumber>.png
                        listBankNumbers.append(rx.cap(2));
                        bankName=rx.cap(1);
                } else {
                        listBankNumbers.append("");
                }

                ui.kAccountCreatorIcon->addItem (QIcon(dirLogo.absoluteFilePath(listLogo.at(i))), bankName);
        }

        //Refresh
        connect((const QObject*) getDocument(), SIGNAL(transactionSuccessfullyEnded(int)), this, SLOT(refresh()), Qt::QueuedConnection);

        refresh();
}

SKGBankPluginWidget::~SKGBankPluginWidget()
{
        SKGTRACEIN(10, "SKGBankPluginWidget::~SKGBankPluginWidget");
}

void SKGBankPluginWidget::onSelectionChanged()
{
        SKGTRACEIN(10, "SKGBankPluginWidget::onSelectionChanged");
        //Mapping
        QItemSelectionModel *selModel=ui.kAccountTableViewEdition->selectionModel();
        QModelIndexList indexes=selModel->selectedRows();
        if (indexes.count()) {
                QModelIndex idx=indexes[indexes.count()-1];

                QSortFilterProxyModel* proxyModel=(QSortFilterProxyModel*) ui.kAccountTableViewEdition->model();
                QModelIndex idxs=proxyModel->mapToSource(idx);

                //set the icon
                SKGAccountObject account(objectModel->getObject(idxs));
                SKGBankObject bank;
                account.getBank(bank);
                QString iconName=bank.getIcon();
                if (iconName.length()) {
                        iconName.remove(".png");
                        iconName.replace('_', ' ');

                        QRegExp rx("(.+) {2,}(.+)");
                        if (rx.indexIn(iconName)!=-1) {
                                iconName=rx.cap(1);
                        }
                        ui.kAccountCreatorIcon->setText(iconName);
                } else ui.kAccountCreatorIcon->setText("");

                ui.kAccountCreatorBank->setText(account.getAttribute("t_BANK"));
                ui.kAccountCreatorAccount->setText(account.getAttribute("t_name"));
                ui.kAccountCreatorBankNumber->setText(account.getAttribute("t_BANK_NUMBER"));
                ui.kAccountCreatorAgencyNumber->setText(account.getAttribute("t_agency_number"));
                ui.kAccountCreatorNumber->setText(account.getAttribute("t_number"));
                ui.kAccountCreatorType->setText(account.getAttribute("t_TYPENLS"));
                ui.kAccountCreatorAddress->setText(account.getAttribute("t_agency_address"));
                ui.kAccountCreatorComment->setText(account.getAttribute("t_comment"));
        } else {
                ui.kAccountCreatorIcon->setText("");
                ui.kAccountCreatorBank->setText("");
                ui.kAccountCreatorAccount->setText("");
                ui.kAccountCreatorBankNumber->setText("");
                ui.kAccountCreatorAgencyNumber->setText("");
                ui.kAccountCreatorNumber->setText("");
                //Not done because of it's a combo ui.kAccountCreatorType->setText("");
                ui.kAccountCreatorAddress->setText("");
                ui.kAccountCreatorComment->setText("");
        }

        onAccountCreatorModified();
        emit selectionChanged();
}

void SKGBankPluginWidget::onAccountCreatorModified()
{
        SKGTRACEIN(10, "SKGBankPluginWidget::onAccountCreatorModified");

        bool activated=ui.kAccountCreatorBank->text().length()>0 &&
                       ui.kAccountCreatorAccount->text().length()>0;

        int nbSelect=getNbSelectedObjects();
        ui.kAccountCreatorAdd->setEnabled(activated);
        ui.kAccountCreatorUpdate->setEnabled(activated && nbSelect>0);
        ui.kAccountCreatorAccount->setEnabled(nbSelect<=1);
        ui.kAccountCreatorNumber->setEnabled(nbSelect<=1);

        //Facilitate bank name
        if (ui.kAccountCreatorBank->text().length()==0) {
                ui.kAccountCreatorBank->setText(ui.kAccountCreatorIcon->currentIndex()!=0 ? ui.kAccountCreatorIcon->currentText() : "");
        }

        //Facilitate bank number
        if (ui.kAccountCreatorBankNumber->text().length()==0) {
                int pos=ui.kAccountCreatorIcon->currentIndex();
                ui.kAccountCreatorBankNumber->setText(pos>=1 && pos-1<listBankNumbers.count() ? listBankNumbers[pos-1] : "");
        }
}

void SKGBankPluginWidget::onAddAccountClicked()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGBankPluginWidget::onAddAccountClicked",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                SKGAccountObject accountObj;

                QString bankname=ui.kAccountCreatorBank->text();
                QString accountname=ui.kAccountCreatorAccount->text();
                QString name=bankname+'-'+accountname;
                {
                        SKGBEGINTRANSACTION(*getDocument(), i18n("Account creation [%1]", name), err);

                        //Create bank object in case of missing
                        SKGBankObject bankObj(getDocument());
                        if (err.isSucceeded()) err=bankObj.setName(ui.kAccountCreatorBank->text());
                        if (err.isSucceeded()) {
                                //Build icon name
                                QString icon;
                                int pos=ui.kAccountCreatorIcon->currentIndex();
                                if (pos!=0) {
                                        icon=ui.kAccountCreatorIcon->currentText();
                                        if (listBankNumbers[pos-1].length()) icon+="  "+listBankNumbers[pos-1];

                                        icon.replace(' ', '_');
                                        icon+=".png";

                                }

                                err=bankObj.setIcon(icon);
                        }
                        if (err.isSucceeded()) err=bankObj.setNumber(ui.kAccountCreatorBankNumber->text());

                        if (err.isSucceeded()) err=bankObj.save();
                        if (err.isSucceeded()) err=bankObj.load();

                        //Create account object in case of missing
                        if (err.isSucceeded()) err=bankObj.addAccount(accountObj);
                        if (err.isSucceeded()) err=accountObj.setName(accountname);
                        if (err.isSucceeded()) err=accountObj.setAgencyNumber(ui.kAccountCreatorAgencyNumber->text());
                        if (err.isSucceeded()) err=accountObj.setAgencyAddress(ui.kAccountCreatorAddress->text());
                        if (err.isSucceeded()) err=accountObj.setComment(ui.kAccountCreatorComment->text());
                        if (err.isSucceeded()) err=accountObj.setNumber(ui.kAccountCreatorNumber->text());
                        if (err.isSucceeded()) err=accountObj.setType((SKGAccountObject::AccountType) (ui.kAccountCreatorType->currentIndex()));
                        if (err.isSucceeded()) err=accountObj.save(false);
                }

                //status bar
                if (err.isSucceeded()) {
                        err=SKGError(0, i18n("Account [%1] created", name));
                        ui.kAccountTableViewEdition->selectObject(accountObj.getUniqueID());
                } else err.addError(ERR_FAIL, i18n("Account creation failed"));
        }
        QApplication::restoreOverrideCursor();

        //Display error
        getMainPanel()->displayErrorMessage(err);
}

SKGObjectBase::SKGListSKGObjectBase SKGBankPluginWidget::getSelectedObjects()
{
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection;
        QItemSelectionModel *selModel=ui.kAccountTableViewEdition->selectionModel();
        QSortFilterProxyModel* proxyModel=(QSortFilterProxyModel*) ui.kAccountTableViewEdition->model();
        SKGObjectModel* model=(SKGObjectModel*) proxyModel->sourceModel();
        if (model) {
                QModelIndexList indexes=selModel->selectedRows();
                foreach(const QModelIndex& index, indexes) {
                        SKGObjectBase obj=model->getObject(proxyModel->mapToSource(index));
                        selection.push_back(obj);
                }
        }
        return selection;
}

int SKGBankPluginWidget::getNbSelectedObjects()
{
        QItemSelectionModel *selModel=ui.kAccountTableViewEdition->selectionModel();
        return (selModel ? selModel->selectedRows().count() : 0);
}

void SKGBankPluginWidget::onModifyAccountClicked()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGBankPluginWidget::onModifyAccountClicked",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                //Get Selection
                SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

                int nb=selection.count();
                SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18n("Account update"), err, nb);
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        SKGAccountObject accountObj=selection[i];

                        //Update account if single selection
                        if (nb==1) {
                                err=accountObj.setName(ui.kAccountCreatorAccount->text());
                                if (err.isSucceeded()) err=accountObj.setNumber(ui.kAccountCreatorNumber->text());
                        }
                        if (err.isSucceeded()) err=accountObj.setType((SKGAccountObject::AccountType) (ui.kAccountCreatorType->currentIndex()));
                        if (err.isSucceeded()) err=accountObj.setAgencyNumber(ui.kAccountCreatorAgencyNumber->text());
                        if (err.isSucceeded()) err=accountObj.setAgencyAddress(ui.kAccountCreatorAddress->text());
                        if (err.isSucceeded()) err=accountObj.setComment(ui.kAccountCreatorComment->text());
                        if (err.isSucceeded()) err=accountObj.save();

                        //Update bank
                        SKGBankObject bankObj;
                        if (SKGNamedObject::getObjectByName(getDocument(), "bank", ui.kAccountCreatorBank->text(), bankObj).isSucceeded()) {
                                //The created bank already exist ==> update parent bank
                                if (err.isSucceeded()) err=accountObj.setBank(bankObj);
                                if (err.isSucceeded()) err=accountObj.save();
                        } else {
                                //The bank does not exist ==> update this one
                                if (err.isSucceeded()) err=accountObj.getBank(bankObj);
                                if (err.isSucceeded()) err=bankObj.setName(ui.kAccountCreatorBank->text());
                        }
                        if (err.isSucceeded()) {
                                //Build icon name
                                QString icon;
                                int pos=ui.kAccountCreatorIcon->currentIndex();
                                if (pos!=0) {
                                        icon=ui.kAccountCreatorIcon->currentText();
                                        if (listBankNumbers[pos-1].length()) icon+="  "+listBankNumbers[pos-1];

                                        icon.replace(' ', '_');
                                        icon+=".png";

                                }

                                err=bankObj.setIcon(icon);
                        }
                        if (err.isSucceeded()) err=bankObj.setNumber(ui.kAccountCreatorBankNumber->text());
                        if (err.isSucceeded()) err=bankObj.save();

                        if (err.isSucceeded()) err=getDocument()->stepForward(i+1);
                }

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18n("Account updated."));
                else err.addError(ERR_FAIL, i18n("Update failed"));
        }
        QApplication::restoreOverrideCursor();

        //Display error
        getMainPanel()->displayErrorMessage(err);
}

void SKGBankPluginWidget::onDoubleClickedAccount()
{
        SKGTRACEIN(10, "SKGBankPluginWidget::onDoubleClickedAccount");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        //Open in operation plugin
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        int nb=selection.count();
        for (int i=0; i<nb; ++i) {
                SKGAccountObject accountObj=selection[i];

                QDomDocument doc("SKGML");
                QDomElement root = doc.createElement("parameters");
                doc.appendChild(root);
                root.setAttribute("account", accountObj.getName());
                getMainPanel()->setNewTabContent(getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, doc.toString());

        }
        QApplication::restoreOverrideCursor();
}

void SKGBankPluginWidget::onFilterRegExpChanged()
{
        SKGTRACEIN(10, "SKGBankPluginWidget::onFilterRegExpChanged");
        QRegExp regExp(ui.kFilterEdit->text(), Qt::CaseInsensitive);
        ((SKGSortFilterProxyModel*) ui.kAccountTableViewEdition->model())->setFilterRegExp(regExp);
}

void SKGBankPluginWidget::onFilterChanged()
{
        SKGTRACEIN(10, "SKGBankPluginWidget::onFilterChanged");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        //Compute where clause
        QString filter;

        if (ui.kHideUseless->checkState()==Qt::Checked) {
                filter=" t_close='N'";
        }

        //Update model
        QSortFilterProxyModel* modelproxy = (QSortFilterProxyModel*) ui.kAccountTableViewEdition->model();
        SKGObjectModel* modelview=(SKGObjectModel*) modelproxy->sourceModel ();
        modelview->setFilter(filter);
        modelview->refresh();

        //Correction bug 2299394 vvv
        if (ui.kAccountTableViewEdition->isAutoResized()) ui.kAccountTableViewEdition->resizeColumnsToContents();
        //Correction bug 2299394 ^^^

        QApplication::restoreOverrideCursor();
}

void SKGBankPluginWidget::onComputeRIB()
{
        QString sb=ui.kAccountCreatorBankNumber->text().rightJustified(5, '0', true);
        QString sg=ui.kAccountCreatorAgencyNumber->text().rightJustified(5, '0', true);
        QString sc=ui.kAccountCreatorNumber->text().rightJustified(11, '0', true);

        QString l1="ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        QString l2="12345678912345678923456789";

        for (int i=0; i<26; ++i) {
                sc=sc.replace(l1[i], l2[i]);
        }

        int b=SKGServices::stringToInt(sb);
        int g=SKGServices::stringToInt(sg);
        int d=SKGServices::stringToInt(sc.left(6));
        int c=SKGServices::stringToInt(sc.right(5));

        ui.kRib->setText(SKGServices::intToString(97 - ((89 * b + 15 * g + 76 * d + 3 * c) % 97)));
}

QString SKGBankPluginWidget::getState()
{
        SKGTRACEIN(10, "SKGBankPluginWidget::getState");
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);

        root.setAttribute("hideUseless", ui.kHideUseless->checkState()==Qt::Checked ? "Y" : "N");
        root.setAttribute("filter", ui.kFilterEdit->text());

        //Memorize table settings
        root.setAttribute("view", ui.kAccountTableViewEdition->getState());
        return doc.toString();
}

void SKGBankPluginWidget::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGBankPluginWidget::setState");
        QDomDocument doc("SKGML");
        doc.setContent(iState);
        QDomElement root = doc.documentElement();

        QString hideUseless=root.attribute ( "hideUseless");
        QString filter=root.attribute ( "filter");

        if (!hideUseless.isEmpty()) ui.kHideUseless->setCheckState (hideUseless=="Y" ? Qt::Checked : Qt::Unchecked);
        if (!filter.isEmpty()) ui.kFilterEdit->setText(filter);
        ui.kAccountTableViewEdition->setState(root.attribute ( "view"));

        onFilterChanged();
}

void SKGBankPluginWidget::refresh()

{
        SKGTRACEIN(10, "SKGBankPluginWidget::refresh");

        //Refresh widgets
        QSqlDatabase* db = getDocument()->getDatabase();
        setEnabled(db!=NULL);
        if (db!=NULL) {
                //Correction bug 2299394 vvv
                if (ui.kAccountTableViewEdition->isAutoResized()) ui.kAccountTableViewEdition->resizeColumnsToContents();
                //Correction bug 2299394 ^^^

                //Refresh info area
                SKGStringListList listTmp;
                SKGServices::executeSelectSqliteOrder(getDocument(),
                                                      "SELECT SUM(f_CURRENTAMOUNT), SUM(f_CHECKED), SUM(f_COMING_SOON) from v_account_display",
                                                      listTmp);
                if (listTmp.count()==2) {
                        KLocale* locale=KGlobal::locale();

                        QString pUnit=((SKGDocumentBank*) getDocument())->getPrimaryUnit();
                        double v1=SKGServices::stringToDouble(listTmp.at(1).at(0));
                        double v2=SKGServices::stringToDouble(listTmp.at(1).at(1));
                        double v3=SKGServices::stringToDouble(listTmp.at(1).at(2));
                        QString s1=QString("<font color=\"")+(v1<0 ? "red" : "black")+"\">"+locale->formatMoney (v1, pUnit,2)+"</font>";
                        QString s2=QString("<font color=\"")+(v2<0 ? "red" : "black")+"\">"+locale->formatMoney (v2, pUnit,2)+"</font>";
                        QString s3=QString("<font color=\"")+(v3<0 ? "red" : "black")+"\">"+locale->formatMoney (v3, pUnit,2)+"</font>";
                        ui.kInfo->setText(i18n("Balance: %1     Checked: %2     Foreseen: %3", s1, s2, s3));

                        QString secondaryUnit=((SKGDocumentBank*) getDocument())->getSecondaryUnit();
                        double secondaryUnitValue=((SKGDocumentBank*) getDocument())->getSecondaryUnitValue();
                        if (!secondaryUnit.isEmpty() && secondaryUnitValue) {
                                s1=QString("<font color=\"")+(v1<0 ? "red" : "black")+"\">"+locale->formatMoney (v1/secondaryUnitValue, secondaryUnit,2)+"</font>";
                                s2=QString("<font color=\"")+(v2<0 ? "red" : "black")+"\">"+locale->formatMoney (v2/secondaryUnitValue, secondaryUnit,2)+"</font>";
                                s3=QString("<font color=\"")+(v3<0 ? "red" : "black")+"\">"+locale->formatMoney (v3/secondaryUnitValue, secondaryUnit,2)+"</font>";
                        }
                        ui.kInfo->setToolTip(i18n("<p>Balance: %1</p><p>Checked: %2</p><p>Foreseen: %3</p>", s1, s2, s3));
                }
        }
}

QWidget* SKGBankPluginWidget::getWidgetForPrint()
{
        return ui.kAccountTableViewEdition;
}
#include "skgbankpluginwidget.moc"
