/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin to generate report.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgreportplugin.h"
#include "skgreportpluginwidget.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skgservices.h"
#include "skgaccountobject.h"
#include "skgcategoryobject.h"
#include "skgunitobject.h"

#include <kaction.h>
#include <kactioncollection.h>

#include <QDomDocument>

K_PLUGIN_FACTORY(SKGReportPluginFactory, registerPlugin<SKGReportPlugin>();)
K_EXPORT_PLUGIN(SKGReportPluginFactory("skrooge_report", "skrooge_report"))

SKGReportPlugin::SKGReportPlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent)
{
        SKGTRACEIN(10, "SKGReportPlugin::SKGReportPlugin");
}

SKGReportPlugin::~SKGReportPlugin()
{
        SKGTRACEIN(10, "SKGReportPlugin::~SKGReportPlugin");
        parent=NULL;
        currentBankDocument=NULL;
}

void SKGReportPlugin::setupActions(SKGMainPanel* iParent, SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGReportPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;
        parent=iParent;

        setComponentData( SKGReportPluginFactory::componentData() );
        setXMLFile("skrooge_report.rc");

        //Menu
        openReportAction = new KAction(KIcon("skrooge_open_report"), i18n("Open &report"), this);
        connect(openReportAction, SIGNAL(triggered(bool)), this, SLOT(actionOpenReport()));
        actionCollection()->addAction( QLatin1String("open_report"), openReportAction );
        openReportAction->setShortcut(Qt::CTRL+Qt::Key_G);

        iParent->registedGlobalAction("open_report", openReportAction);
}

void SKGReportPlugin::refresh()
{
        SKGTRACEIN(10, "SKGReportPlugin::refresh");
        SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();

        if (selection.count()>0) {
                QString table=selection.at(0).getRealTable();
                bool onOperation=(table=="operation" || table=="account" || table=="unit" || table=="category");
                openReportAction->setEnabled(onOperation);
        } else {
                openReportAction->setEnabled(false);
        }
}

void SKGReportPlugin::close()
{
        SKGTRACEIN(10, "SKGReportPlugin::close");
}

SKGTabWidget* SKGReportPlugin::getWidget()
{
        SKGTRACEIN(10, "SKGReportPlugin::getWidget");
        return new SKGReportPluginWidget(parent, (SKGDocumentBank*) currentBankDocument);
}

QString SKGReportPlugin::title() const
{
        return i18n("Report");
}

QString SKGReportPlugin::icon() const
{
        return "view-statistics";
}

QString SKGReportPlugin::statusTip () const
{
        return i18n("Generate report");
}

QString SKGReportPlugin::toolTip () const
{
        return i18n("Generate report");
}

int SKGReportPlugin::getOrder() const
{
        return 40;
}

QStringList SKGReportPlugin::tips() const
{
        QStringList output;
        output.push_back(i18n("<p>... you can double click on a value in \"%1\" to show corresponding operations.</p>", title()));
        output.push_back(i18n("<p>... you can open a report from selection done in other pages.</p>"));
        output.push_back(i18n("<p>... you can export reports in many formats.</p>"));
        return output;
}

bool SKGReportPlugin::isInContext() const
{
        return true;
}

void SKGReportPlugin::actionOpenReport()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGReportPlugin::actionOpenReport",err);
        SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();

        int nb=selection.count();
        if (nb>0) {
                QDomDocument doc("SKGML");
                QString wc;
                QString title;
                QString table=selection.at(0).getRealTable();
                if (table=="account") {
                        wc="rd_account_id in (";
                        title=i18n("Operations of account ");

                        for (int i=0; i<nb; ++i) {
                                SKGAccountObject tmp=selection.at(i);
                                if (i) {
                                        wc+=',';
                                        title+=',';
                                }
                                wc+=SKGServices::intToString(tmp.getID());
                                title+='\''+tmp.getName()+'\'';
                        }
                        wc+=')';
                } else if (table=="unit") {
                        wc="rc_unit_id in (";
                        title=i18n("Operations with Unit equal to ");

                        for (int i=0; i<nb; ++i) {
                                SKGUnitObject tmp=selection.at(i);
                                if (i) {
                                        wc+=',';
                                        title+=',';
                                }
                                wc+=SKGServices::intToString(tmp.getID());
                                title+='\''+tmp.getName()+'\'';
                        }
                        wc+=')';
                } else if (table=="category") {
                        wc="t_REALCATEGORY in (";
                        QString wc2;
                        title=i18n("Operations with Category equal to ");

                        for (int i=0; i<nb; ++i) {
                                SKGCategoryObject tmp=selection.at(i);
                                if (i) {
                                        wc+=',';
                                        wc2+=" OR ";
                                        title+=',';
                                }
                                wc+='\''+SKGServices::stringToSqlString(tmp.getFullName())+'\'';
                                wc2+="t_REALCATEGORY like '"+SKGServices::stringToSqlString(tmp.getFullName())+"%'";
                                title+='\''+tmp.getFullName()+'\'';
                        }
                        wc+=") OR "+wc2;
                } else if (table=="operation") {
                        wc="id in (";
                        title=i18n("Selected operations");

                        for (int i=0; i<nb; ++i) {
                                if (i) {
                                        wc+=',';
                                        title+=',';
                                }
                                wc+=SKGServices::intToString(selection.at(i).getID());
                        }
                        wc+=')';
                }

                //Call report plugin
                QDomElement root = doc.createElement("parameters");
                doc.appendChild(root);
                root.setAttribute("operationWhereClause", wc);
                root.setAttribute("title", title);
                root.setAttribute("title_icon", "view-statistics");
                parent->setNewTabContent(parent->getPluginByName("Skrooge report plugin"), -1, doc.toString());
        }
}
#include "skgreportplugin.moc"
